package com.biz.crm.dms.business.order.local.notifier;

import com.alibaba.fastjson.JSONArray;
import com.biz.crm.dms.business.costpool.capital.sdk.dto.CostPoolCapitalDto;
import com.biz.crm.dms.business.costpool.capital.sdk.enums.CapitalAdjustTypeEnum;
import com.biz.crm.dms.business.costpool.credit.sdk.dto.CreditPayDto;
import com.biz.crm.dms.business.costpool.credit.sdk.enums.CashAdjustOperateEnum;
import com.biz.crm.dms.business.costpool.credit.sdk.enums.CashAdjustTypeEnum;
import com.biz.crm.dms.business.costpool.discount.sdk.dto.CostPoolDiscountDto;
import com.biz.crm.dms.business.costpool.discount.sdk.enums.PoolOperationTypeEnum;
import com.biz.crm.dms.business.costpool.replenishment.sdk.dto.CostPoolReplenishmentDto;
import com.biz.crm.dms.business.costpool.sdk.service.CostPoolVoService;
import com.biz.crm.dms.business.order.common.sdk.enums.ItemTypeEnum;
import com.biz.crm.dms.business.order.common.sdk.enums.OrderCategoryEnum;
import com.biz.crm.dms.business.order.common.sdk.enums.OrderStatusEnum;
import com.biz.crm.dms.business.order.common.sdk.enums.OrderTypeEnum;
import com.biz.crm.dms.business.order.common.sdk.register.TallyItemRegister;
import com.biz.crm.dms.business.order.local.entity.Order;
import com.biz.crm.dms.business.order.local.entity.OrderDetail;
import com.biz.crm.dms.business.order.local.repository.OrderRepository;
import com.biz.crm.dms.business.order.local.service.assist.OrderAssist;
import com.biz.crm.dms.business.policy.sdk.service.SalePolicyRecordVoService;
import com.biz.crm.dms.business.psi.product.sdk.dto.productstock.ProductStockOperationDto;
import com.biz.crm.dms.business.psi.product.sdk.service.productstock.ProductStockVoService;
import com.biz.crm.mdm.business.warehouse.sdk.vo.WarehouseVo;
import com.biz.crm.workflow.sdk.constant.enums.ActApproveStatusEnum;
import com.biz.crm.workflow.sdk.dto.CallBackDto;
import com.biz.crm.workflow.sdk.listener.CallBackListener;
import com.google.common.collect.Lists;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;

import javax.transaction.Transactional;
import java.math.BigDecimal;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;

/**
 * 销售订单的工作流的回调处理
 *
 * @author hefan
 */
@Component
public class SalesOrderFlowCallBackImpl implements CallBackListener {

  @Autowired(required = false)
  private OrderRepository orderRepository;

  @Autowired(required = false)
  private CostPoolVoService costPoolVoService;

  @Transactional
  @Override
  public void onCallBack(CallBackDto dto) {
    String formType = dto.getFormType();
    if (!OrderCategoryEnum.SALES_ORDER.getDictCode().equals(formType)) {
      return;
    }
    // 查询订单是否存在
    String formNo = dto.getFormNo();
    Order order = this.findByOrderCode(formNo);
    if (Objects.isNull(order)) {
      return;
    }
    // 原订单数据的状态是否可以被回调修改
    String orderStatus = order.getOrderStatus();
    String awaitApprove = OrderStatusEnum.AWAIT_APPROVE.getDictCode();
    boolean equals = awaitApprove.equals(orderStatus);
    Validate.isTrue(equals, "此订单非审批中状态，无法接受工作流的操作！");
    // 修改订单的状态
    Integer processState = dto.getProcessState();
    // - 把工作流状态转换为业务状态
    orderStatus = this.getOrderStatus(processState);
    this.orderRepository.updateOrderStatusById(orderStatus, order.getId());
    // 工作流操作延申的业务操作
    boolean isWaitShipped = OrderStatusEnum.WAIT_SHIPPED.getDictCode().equals(orderStatus);
    if (isWaitShipped) {
      // 审批通过，把资金占用变成扣减
      this.payOrder(order);
      return;
    }
    // - 驳回：回退扣减的费用；释放占用的库存
    // - 流程撤回：回退扣减的费用；释放占用的库存
    /**
     * 释放费用
     * 归还促销政策额度
     * 归还库存
     */
    this.releaseOrder(order);
    this.releaseSalePolicy(order);
    this.returnStock(order);

  }

  @Autowired(required = false)
  private OrderAssist orderAssist;

  @Autowired(required = false)
  private ProductStockVoService productStockVoService;

  /**
   * 归还库存
   *
   * @param order 订单
   */
  private void returnStock(Order order) {
    WarehouseVo warehouse = orderAssist.findWarehouseVo(order);
    Validate.notNull(warehouse, "订单占用库存的时候，没有找到仓库信息");
    List<ProductStockOperationDto> list = orderAssist.getProductStockOperationDtos(order, warehouse);
    this.productStockVoService.thawBatch(list);
  }


  /**
   * 根据订单编码查询订单
   *
   * @param orderCode 指令码
   * @return {@link Order}
   */
  private Order findByOrderCode(String orderCode) {
    List<Order> orderList = orderRepository.findDetailByOrderCodes(Lists.newArrayList(orderCode));
    Validate.isTrue(!CollectionUtils.isEmpty(orderList), "没有查询到订单信息");
    Validate.isTrue(orderList.size() == 1, "查询出了多条数据，这应该是一个数据错误。");
    return orderList.get(0);
  }

  @Autowired(required = false)
  private SalePolicyRecordVoService salePolicyRecordVoService;

  /**
   * 归还 促销政策的额度
   *
   * @param order 订单
   */
  private void releaseSalePolicy(Order order) {
    boolean isMaterial = order.getOrderCategory().equals(OrderCategoryEnum.MATERIAL_ORDER.getDictCode());
    if (isMaterial) {
      return;
    }
    this.salePolicyRecordVoService.deleteByBillCode(order.getOrderCode(), order.getTenantCode() , order.getRelateCode());
  }

  /**
   * 支付订单
   *
   * @param order 订单
   */
  private void payOrder(Order order) {
    /**
     * 先释放
     * 后扣减
     */
    this.releaseOrder(order);
    this.useAmount(order);
  }

  /**
   * 扣减资金
   *
   * @param order 订单
   */
  private void useAmount(Order order) {
    boolean isFree = OrderTypeEnum.FREE.getDictCode().equals(order.getOrderType());
    boolean isMaterial = order.getOrderCategory().equals(OrderCategoryEnum.MATERIAL_ORDER.getDictCode());
    if (isFree || isMaterial) {
      return;
    }
    JSONArray array = new JSONArray();
    // 货补、折扣、资金、授信都要去费用池扣减。
    CostPoolCapitalDto costPoolCapitalDto = this.getCapitalDto(order,CapitalAdjustTypeEnum.ORDER_USE);
    if (Objects.nonNull(costPoolCapitalDto)) {
      array.add(costPoolCapitalDto);
    }
    CreditPayDto creditPayDto = this.getCreditPayDto(order, CashAdjustOperateEnum.ORDER, CashAdjustTypeEnum.CREDIT_REDUCE);
    if (Objects.nonNull(creditPayDto)) {
      array.add(creditPayDto);
    }
    CostPoolDiscountDto discountDto = this.getDiscountDto(order, PoolOperationTypeEnum.ORDER_USE);
    if (Objects.nonNull(discountDto)) {
      array.add(discountDto);
    }
    List<CostPoolReplenishmentDto> replenishmentDto = this.getReplenishmentDto(order, com.biz.crm.dms.business.costpool.replenishment.sdk.enums.PoolOperationTypeEnum.ORDER_USE);
    if (!CollectionUtils.isEmpty(replenishmentDto)) {
      array.addAll(replenishmentDto);
    }
    this.costPoolVoService.handleAdjust(array);
  }

  /**
   * 释放订单占用的资金
   *
   * @param order 订单
   */
  private void releaseOrder(Order order) {
    boolean isFree = OrderTypeEnum.FREE.getDictCode().equals(order.getOrderType());
    boolean isMaterial = order.getOrderCategory().equals(OrderCategoryEnum.MATERIAL_ORDER.getDictCode());
    if (isFree || isMaterial) {
      return;
    }
    JSONArray array = new JSONArray();
    // 货补、折扣、资金、授信都要去费用池扣减。
    CostPoolCapitalDto costPoolCapitalDto = this.getCapitalDto(order, CapitalAdjustTypeEnum.OCCUPY_RELEASE);
    if (Objects.nonNull(costPoolCapitalDto)) {
      array.add(costPoolCapitalDto);
    }
    CreditPayDto creditPayDto = this.getCreditPayDto(order, CashAdjustOperateEnum.OCCUPY_RELEASE, CashAdjustTypeEnum.OCCUPY_RELEASE);
    if (Objects.nonNull(creditPayDto)) {
      array.add(creditPayDto);
    }
    CostPoolDiscountDto discountDto = this.getDiscountDto(order, PoolOperationTypeEnum.OCCUPY_RELEASE);
    if (Objects.nonNull(discountDto)) {
      array.add(discountDto);
    }
    List<CostPoolReplenishmentDto> replenishmentDto = this.getReplenishmentDto(order, com.biz.crm.dms.business.costpool.replenishment.sdk.enums.PoolOperationTypeEnum.OCCUPY_RELEASE);
    if (!CollectionUtils.isEmpty(replenishmentDto)) {
      array.addAll(replenishmentDto);
    }
    this.costPoolVoService.handleAdjust(array);
  }

  /**
   * 得到货补池dto
   *
   * @param order             订单
   * @param operationTypeEnum 操作类型枚举
   * @return {@link List}<{@link CostPoolReplenishmentDto}>
   */
  private List<CostPoolReplenishmentDto> getReplenishmentDto(Order order, com.biz.crm.dms.business.costpool.replenishment.sdk.enums.PoolOperationTypeEnum operationTypeEnum) {
    List<OrderDetail> orderDetails = order.getOrderDetails();
    if (CollectionUtils.isEmpty(orderDetails)) {
      return null;
    }
    List<CostPoolReplenishmentDto> list = Lists.newArrayListWithCapacity(orderDetails.size());
    // 只考虑货补商品
    for (OrderDetail orderDetail : orderDetails) {
      // --行上的销售总金额
      BigDecimal salesAmount = orderDetail.getSalesAmount();
      String itemType = orderDetail.getItemType();
      if (!itemType.equals(ItemTypeEnum.COMPENSATED_GOODS.getDictCode())) {
        continue;
      }
      CostPoolReplenishmentDto dto = OrderAssist.getCostPoolReplenishmentDto(order, orderDetail, salesAmount, operationTypeEnum);
      list.add(dto);
    }
    return list;
  }


  @Qualifier("DiscountApportionTallyItemRegister")
  @Autowired(required = false)
  private TallyItemRegister discountApportionTallyItemRegister;

  /**
   * 得到折扣池dto
   *
   * @param order             订单
   * @param operationTypeEnum 操作类型枚举
   * @return {@link CostPoolDiscountDto}
   */
  private CostPoolDiscountDto getDiscountDto(Order order, PoolOperationTypeEnum operationTypeEnum) {
    BigDecimal itemAmount = OrderAssist.getTallyItemAmount(order, discountApportionTallyItemRegister);
    if (itemAmount.compareTo(BigDecimal.ZERO) == 0) {
      return null;
    }
    return OrderAssist.getCostPoolDiscountDto(order, itemAmount, operationTypeEnum);
  }

  @Qualifier("CreditTallyItemRegister")
  @Autowired(required = false)
  private TallyItemRegister creditTallyItemRegister;

  /**
   * 得到授信池dto
   *
   * @param order          订单
   * @param operateEnum    枚举操作
   * @param adjustTypeEnum 调整类型枚举
   * @return {@link CreditPayDto}
   */
  private CreditPayDto getCreditPayDto(Order order,CashAdjustOperateEnum operateEnum, CashAdjustTypeEnum adjustTypeEnum) {
    BigDecimal itemAmount = OrderAssist.getTallyItemAmount(order, creditTallyItemRegister);
    if (itemAmount.compareTo(BigDecimal.ZERO) == 0) {
      return null;
    }
    return OrderAssist.getCreditPayDto(order, itemAmount, operateEnum, adjustTypeEnum);
  }

  @Qualifier("CapitalTallyItemRegister")
  @Autowired(required = false)
  private TallyItemRegister capitalTallyItemRegister;

  /**
   * 获得资金池dto
   *
   * @param order                 订单
   * @param capitalAdjustTypeEnum 资本调整枚举类型
   * @return {@link CostPoolCapitalDto}
   */
  private CostPoolCapitalDto getCapitalDto(Order order, CapitalAdjustTypeEnum capitalAdjustTypeEnum) {
    BigDecimal itemAmount = OrderAssist.getTallyItemAmount(order, capitalTallyItemRegister);
    if (itemAmount.compareTo(BigDecimal.ZERO) == 0) {
      return null;
    }
    /**
     * 资金
     * CapitalAdjustTypeEnum.getKey()
     */
    return OrderAssist.getCostPoolCapitalDto(order, itemAmount, capitalAdjustTypeEnum);
  }

  /**
   * 把工作流状态转换为业务状态
   *  fixme：状态对应不一定正确，因为工作的流使用还得讨论。
   * @param processState
   * @return
   */
  private String getOrderStatus(Integer processState) {
    Map<String, String> map = new HashMap<>(ActApproveStatusEnum.values().length);
    map.put(ActApproveStatusEnum.APPROVED.getCode(), OrderStatusEnum.WAIT_SHIPPED.getDictCode());
    map.put(ActApproveStatusEnum.REJECTED.getCode(), OrderStatusEnum.REJECTED.getDictCode());
    map.put(ActApproveStatusEnum.INTERRUPT.getCode(), OrderStatusEnum.CANCELED.getDictCode());
    map.put(ActApproveStatusEnum.CLOSED.getCode(), OrderStatusEnum.CLOSE.getDictCode());
    return map.get(processState.toString());
  }
}
