package com.biz.crm.dms.business.order.local.service.internal;

import com.alibaba.fastjson.JSONArray;
import com.biz.crm.business.common.sdk.service.LoginUserService;
import com.biz.crm.dms.business.costpool.capital.sdk.dto.CostPoolCapitalDto;
import com.biz.crm.dms.business.costpool.capital.sdk.enums.CapitalAdjustTypeEnum;
import com.biz.crm.dms.business.costpool.credit.sdk.dto.CreditPayDto;
import com.biz.crm.dms.business.costpool.credit.sdk.enums.CashAdjustOperateEnum;
import com.biz.crm.dms.business.costpool.credit.sdk.enums.CashAdjustTypeEnum;
import com.biz.crm.dms.business.costpool.discount.sdk.dto.CostPoolDiscountDto;
import com.biz.crm.dms.business.costpool.discount.sdk.enums.PoolOperationTypeEnum;
import com.biz.crm.dms.business.costpool.replenishment.sdk.dto.CostPoolReplenishmentDto;
import com.biz.crm.dms.business.costpool.sdk.service.CostPoolVoService;
import com.biz.crm.dms.business.costpool.sdk.vo.CostPoolVo;
import com.biz.crm.dms.business.order.common.sdk.enums.ItemTypeEnum;
import com.biz.crm.dms.business.order.common.sdk.enums.OrderCategoryEnum;
import com.biz.crm.dms.business.order.common.sdk.enums.OrderStatusEnum;
import com.biz.crm.dms.business.order.common.sdk.register.TallyItemRegister;
import com.biz.crm.dms.business.order.local.entity.Order;
import com.biz.crm.dms.business.order.local.entity.OrderDetail;
import com.biz.crm.dms.business.order.local.repository.OrderRepository;
import com.biz.crm.dms.business.order.local.service.assist.OrderAssist;
import com.biz.crm.dms.business.order.sdk.dto.OrderEventDto;
import com.biz.crm.dms.business.order.sdk.event.OrderCloseListener;
import com.biz.crm.dms.business.order.sdk.event.OrderLogEventListener;
import com.biz.crm.dms.business.order.sdk.service.OrderCloseService;
import com.biz.crm.dms.business.order.sdk.vo.OrderVo;
import com.biz.crm.dms.business.policy.sdk.service.SalePolicyRecordVoService;
import com.biz.crm.dms.business.psi.product.sdk.dto.productstock.ProductStockOperationDto;
import com.biz.crm.dms.business.psi.product.sdk.service.productstock.ProductStockVoService;
import com.biz.crm.mdm.business.warehouse.sdk.vo.WarehouseVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.event.sdk.function.SerializableBiConsumer;
import com.bizunited.nebula.event.sdk.service.NebulaNetEventClient;
import com.bizunited.nebula.security.sdk.vo.LoginDetails;
import com.google.common.collect.Lists;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.math.BigDecimal;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Objects;

/**
 * 订单关闭服务impl
 *
 * @author hefan
 * @date 2022/05/12
 */
@Service
public class OrderCloseServiceImpl implements OrderCloseService {

  @Autowired(required = false)
  private OrderRepository orderRepository;

  @Autowired(required = false)
  private NebulaToolkitService nebulaToolkitService;

  @Autowired(required = false)
  private List<OrderCloseListener> orderCloseListeners;

  @Autowired(required = false)
  private NebulaNetEventClient nebulaNetEventClient;

  @Autowired(required = false)
  private LoginUserService loginUserService;

  @SuppressWarnings("AlibabaTransactionMustHaveRollback")
  @Transactional
  @Override
  public void handleClose(String orderCode) {
    /**
     * - 查询订单
     * - 校验订单能否关闭
     * - 关闭【待发货】订单
     * - 关闭【部分发货】订单
     * - 修改订单状态
     */
    Validate.isTrue(StringUtils.isNotBlank(orderCode), "订单编码必须传入！");
    // - 根据订单编码查询订单
    Order order = this.findByOrderCode(orderCode);
    // - 校验订单状态
    this.isCloseable(order);
    // - 关闭【待发货】订单
    this.closeWaitShipped(order);
    // - 关闭【部分发货】订单
    this.closePartShipped(order);
    // 修改订单状态
    String dictCode = OrderStatusEnum.CLOSE.getDictCode();
    orderRepository.updateOrderStatusById(dictCode, order.getId());
    // 记录操作日志
    this.recordOperationLog(order, dictCode);
  }

  /**
   * 记录操作日志
   *
   * @param order    订单
   * @param dictCode dict类型代码
   */
  private void recordOperationLog(Order order, String dictCode) {
    LoginDetails loginUser = this.loginUserService.getLoginUser();
    OrderEventDto orderEventDto = new OrderEventDto();
    OrderVo original = new OrderVo();
    original.setId(order.getId());
    original.setOrderStatus(order.getOrderStatus());
    original.setModifyAccount(order.getModifyAccount());
    original.setOrderCode(order.getOrderCode());
    original.setRelateCode(order.getRelateCode());
    original.setCreateAccount(order.getCreateAccount());
    orderEventDto.setOriginal(original);
    OrderVo newest = new OrderVo();
    newest.setId(order.getId());
    newest.setOrderStatus(dictCode);
    newest.setModifyAccount(loginUser.getAccount());
    newest.setOrderCode(order.getOrderCode());
    newest.setRelateCode(order.getRelateCode());
    newest.setCreateAccount(order.getCreateAccount());
    orderEventDto.setNewest(newest);
    SerializableBiConsumer<OrderLogEventListener, OrderEventDto> onClose = OrderLogEventListener::onClose;
    this.nebulaNetEventClient.publish(orderEventDto, OrderLogEventListener.class, onClose);
  }

  @Override
  public void handleFlowClose(String orderCode) {
    Validate.isTrue(StringUtils.isNotBlank(orderCode), "订单编码必须传入！");
    // - 根据订单编码查询订单
    Order order = this.findByOrderCode(orderCode);
    boolean isMaterial = order.getOrderCategory().equals(OrderCategoryEnum.MATERIAL_ORDER.getDictCode());
    if (!isMaterial) {
      // - 归还费用
      this.returnFee(order);
      // - 归还促销政策额度
      this.salePolicyRecordVoService.deleteByBillCode(order.getOrderCode(), order.getTenantCode() , order.getRelateCode());
    }
    // - 归还库存
    this.returnStock(order);
  }

  /**
   * 判断该订单能否关闭
   *
   * @param order 订单
   */
  private void isCloseable(Order order) {
    String orderStatus = order.getOrderStatus();
    OrderStatusEnum orderStatusEnum = OrderStatusEnum.findByCode(orderStatus);
    Validate.notNull(orderStatusEnum, "非法状态的订单");
    boolean isWaitShipped = OrderStatusEnum.WAIT_SHIPPED.getDictCode().equals(orderStatus);
    boolean isPartShipped = OrderStatusEnum.PART_SHIPPED.getDictCode().equals(orderStatus);
    Validate.isTrue(isWaitShipped || isPartShipped, "状态：%s 的订单无法关闭", orderStatusEnum.getValue());
  }

  @Autowired(required = false)
  private SalePolicyRecordVoService salePolicyRecordVoService;

  /**
   * 关闭待发货
   *
   * @param order 订单
   */
  private void closeWaitShipped(Order order) {
    String orderStatus = order.getOrderStatus();
    boolean isWaitShipped = OrderStatusEnum.WAIT_SHIPPED.getDictCode().equals(orderStatus);
    if (!isWaitShipped) {
      return;
    }
    /**
     * 1. 待发货状态的订单可以关闭，
     * - 已扣减的资金、费用、授信需全额归还，
     * - 占用的促销政策额度也需归还。
     */
    // - 归还费用
    this.returnFee(order);
    // - 归还库存
    this.returnStock(order);
    // - 归还促销政策额度
    salePolicyRecordVoService.deleteByBillCode(order.getOrderCode(), order.getTenantCode() , order.getRelateCode());
  }

  /**
   * 关闭部分发货
   *
   * @param order 订单
   */
  private void closePartShipped(Order order) {
    String orderStatus = order.getOrderStatus();
    boolean isPartShipped = OrderStatusEnum.PART_SHIPPED.getDictCode().equals(orderStatus);
    if (!isPartShipped) {
      return;
    }
    /**
     * 1. 部分发货状态的订单可以关闭，
     * - 未发货部分的商品所扣减的资金、授信、费用需归还，
     * 按照揉价或者均摊后的信息予以归还，若有账目不平的问题，客户和经销商可协商后在账款中心进行手动调整
     */
    // - 上层模块消耗了该订单多少价值（或者，上层模块允许我归还多少价值）
    Validate.isTrue(!CollectionUtils.isEmpty(orderCloseListeners), "部分关闭订单时，没有订单关闭监听者是不可能的");
    // - 产品说只有一种
    OrderCloseListener orderCloseListener = orderCloseListeners.get(0);
    OrderVo orderVoCopy =  this.nebulaToolkitService.copyObjectByBlankList(order, OrderVo.class, HashSet.class, LinkedList.class);
    OrderVo orderVo = orderCloseListener.onReturn(orderVoCopy);
    Order returnOrder = nebulaToolkitService.copyObjectByBlankList(orderVo, Order.class, HashSet.class, LinkedList.class);
    // - 归还费用
    this.returnFee(returnOrder);
    // - 归还库存
    this.returnStock(returnOrder);
  }

  @Autowired(required = false)
  private OrderAssist orderAssist;

  @Autowired(required = false)
  private ProductStockVoService productStockVoService;

  /**
   * 归还库存
   *
   * @param order 订单
   */
  private void returnStock(Order order) {
    WarehouseVo warehouse = orderAssist.findWarehouseVo(order);
    Validate.notNull(warehouse, "订单占用库存的时候，没有找到仓库信息");
    List<ProductStockOperationDto> list = orderAssist.getProductStockOperationDtos(order, warehouse);
    this.productStockVoService.thawBatch(list);
  }

  @Autowired(required = false)
  private CostPoolVoService costPoolVoService;

  /**
   * 返回费用
   *
   * @param order 订单
   */
  private void returnFee(Order order) {
    JSONArray array = new JSONArray();
    // 货补、折扣、资金、授信都要去费用池归还。
    CostPoolCapitalDto costPoolCapitalDto = this.getCapitalDto(order);
    if (Objects.nonNull(costPoolCapitalDto)) {
      array.add(costPoolCapitalDto);
    }
    CreditPayDto creditPayDto = this.getCreditPayDto(order);
    if (Objects.nonNull(creditPayDto)) {
      // - 查询授信已使用金额
      List<CostPoolVo> costPoolVos = this.costPoolVoService.handleRequestCostPoolVos(creditPayDto);
      CostPoolVo costPoolVo = costPoolVos.get(0);
      BigDecimal hasUseAmount = costPoolVo.getHasUseAmount();
      int compare = hasUseAmount.compareTo(creditPayDto.getAmount());
      // - 待还授信 大于 等于 归还授信，则给归还
      if (compare >= 0) {
        array.add(creditPayDto);
      } else {
        // - 将这笔授信，归还给资金
        CostPoolCapitalDto creditToCapitalDto = OrderAssist.getCostPoolCapitalDto(order, creditPayDto.getAmount(), CapitalAdjustTypeEnum.ORDER_CLOSE);
        array.add(creditToCapitalDto);
      }
    }
    CostPoolDiscountDto discountDto = this.getDiscountDto(order);
    if (Objects.nonNull(discountDto)) {
      array.add(discountDto);
    }
    List<CostPoolReplenishmentDto> replenishmentDto = this.getReplenishmentDto(order);
    if (!CollectionUtils.isEmpty(replenishmentDto)) {
      array.addAll(replenishmentDto);
    }
    this.costPoolVoService.handleAdjust(array);
  }

  /**
   * 根据订单编码查询订单
   *
   * @param orderCode 指令码
   * @return {@link Order}
   */
  private Order findByOrderCode(String orderCode) {
    List<Order> orderList = orderRepository.findDetailByOrderCodes(Lists.newArrayList(orderCode));
    Validate.isTrue(!CollectionUtils.isEmpty(orderList), "没有查询到订单信息");
    Validate.isTrue(orderList.size() == 1, "查询出了多条数据，这应该是一个数据错误。");
    return orderList.get(0);
  }

  @Qualifier("CapitalTallyItemRegister")
  @Autowired(required = false)
  private TallyItemRegister capitalTallyItemRegister;

  /**
   * 创建 资金归还需要的对象
   *
   * @return
   */
  private CostPoolCapitalDto getCapitalDto(Order order) {
    BigDecimal itemAmount = OrderAssist.getTallyItemAmount(order, capitalTallyItemRegister);
    if (itemAmount.compareTo(BigDecimal.ZERO) == 0) {
      return null;
    }
    /**
     * 资金
     * CapitalAdjustTypeEnum.getKey()
     */
    return OrderAssist.getCostPoolCapitalDto(order, itemAmount, CapitalAdjustTypeEnum.ORDER_CLOSE);
  }


  @Qualifier("CreditTallyItemRegister")
  @Autowired(required = false)
  private TallyItemRegister creditTallyItemRegister;

  /**
   * 创建 信用归还需要的对象
   * CreditRefundDto
   *
   * @param order
   * @return
   */
  private CreditPayDto getCreditPayDto(Order order) {
    BigDecimal itemAmount = OrderAssist.getTallyItemAmount(order, creditTallyItemRegister);
    if (itemAmount.compareTo(BigDecimal.ZERO) == 0) {
      return null;
    }
    return OrderAssist.getCreditPayDto(order, itemAmount, CashAdjustOperateEnum.CLOSE_ORDER, CashAdjustTypeEnum.CREDIT_RELEASE);
  }


  @Qualifier("DiscountApportionTallyItemRegister")
  @Autowired(required = false)
  private TallyItemRegister discountApportionTallyItemRegister;

  /**
   * 创建 折扣归还需要的对象
   *
   * @param order
   * @return
   */
  private CostPoolDiscountDto getDiscountDto(Order order) {
    BigDecimal itemAmount = OrderAssist.getTallyItemAmount(order, discountApportionTallyItemRegister);
    if (itemAmount.compareTo(BigDecimal.ZERO) == 0) {
      return null;
    }
    return OrderAssist.getCostPoolDiscountDto(order, itemAmount, PoolOperationTypeEnum.ORDER_CLOSE);
  }


  /**
   * 创建 货补归还需要的对象
   *
   * @param order
   * @return
   */
  private List<CostPoolReplenishmentDto> getReplenishmentDto(Order order) {
    List<OrderDetail> orderDetails = order.getOrderDetails();
    if (CollectionUtils.isEmpty(orderDetails)) {
      return null;
    }
    List<CostPoolReplenishmentDto> list = Lists.newArrayListWithCapacity(orderDetails.size());
    // 只考虑货补商品
    for (OrderDetail orderDetail : orderDetails) {
      // --行上的销售总金额
      BigDecimal salesAmount = orderDetail.getSalesAmount();
      String itemType = orderDetail.getItemType();
      if (!itemType.equals(ItemTypeEnum.COMPENSATED_GOODS.getDictCode())) {
        continue;
      }
      CostPoolReplenishmentDto dto = OrderAssist.getCostPoolReplenishmentDto(order, orderDetail, salesAmount, com.biz.crm.dms.business.costpool.replenishment.sdk.enums.PoolOperationTypeEnum.ORDER_CLOSE);
      list.add(dto);
    }
    return list;
  }


}
