package com.biz.crm.dms.business.order.local.service.internal;

import com.biz.crm.business.common.sdk.model.LoginUserDetailsForEMS;
import com.biz.crm.business.common.sdk.service.GenerateCodeService;
import com.biz.crm.business.common.sdk.service.LoginUserService;
import com.biz.crm.business.common.sdk.vo.FileVo;
import com.biz.crm.dms.business.costpool.sdk.dto.CostPoolDto;
import com.biz.crm.dms.business.costpool.sdk.strategy.CostPoolRegister;
import com.biz.crm.dms.business.costpool.sdk.strategy.CostPoolStrategy;
import com.biz.crm.dms.business.costpool.sdk.vo.CostPoolVo;
import com.biz.crm.dms.business.order.apportion.sdk.model.OrderApportionContext;
import com.biz.crm.dms.business.order.apportion.sdk.service.OrderApportionVoService;
import com.biz.crm.dms.business.order.apportion.sdk.service.OrderApportionVoServiceChain;
import com.biz.crm.dms.business.order.apportion.sdk.service.OrderApportionVoServiceChainInstance;
import com.biz.crm.dms.business.order.apportion.sdk.vo.OrderApportionVo;
import com.biz.crm.dms.business.order.common.sdk.dto.OrderDto;
import com.biz.crm.dms.business.order.common.sdk.dto.OrderTabulateDto;
import com.biz.crm.dms.business.order.common.sdk.enums.ItemTypeEnum;
import com.biz.crm.dms.business.order.common.sdk.enums.OrderCategoryEnum;
import com.biz.crm.dms.business.order.common.sdk.enums.OrderCommitPhaseEnum;
import com.biz.crm.dms.business.order.common.sdk.enums.OrderPaymentMethodEnum;
import com.biz.crm.dms.business.order.common.sdk.enums.OrderStatusEnum;
import com.biz.crm.dms.business.order.common.sdk.enums.OrderTypeEnum;
import com.biz.crm.dms.business.order.common.sdk.enums.OrderValidationProcessNodeEnum;
import com.biz.crm.dms.business.order.common.sdk.model.TallyItemRegisterModel;
import com.biz.crm.dms.business.order.common.sdk.register.TallyItemRegister;
import com.biz.crm.dms.business.order.common.sdk.vo.OrderConfigVo;
import com.biz.crm.dms.business.order.config.sdk.service.OrderValidateVoService;
import com.biz.crm.dms.business.order.local.entity.Order;
import com.biz.crm.dms.business.order.local.entity.OrderDetail;
import com.biz.crm.dms.business.order.local.entity.PurchaseHistory;
import com.biz.crm.dms.business.order.local.service.OrderFlowService;
import com.biz.crm.dms.business.order.local.service.OrderService;
import com.biz.crm.dms.business.order.local.service.PurchaseHistoryService;
import com.biz.crm.dms.business.order.local.service.assist.OrderAssist;
import com.biz.crm.dms.business.order.sdk.constant.OrderConstant;
import com.biz.crm.dms.business.order.sdk.dto.OrderConfirmDto;
import com.biz.crm.dms.business.order.sdk.dto.OrderPayConfirmDto;
import com.biz.crm.dms.business.order.sdk.service.OrderConfirmService;
import com.biz.crm.dms.business.order.sdk.vo.OrderConfirmVo;
import com.biz.crm.dms.business.order.sdk.vo.OrderDetailPayPreviewVo;
import com.biz.crm.dms.business.order.sdk.vo.OrderDetailPreviewVo;
import com.biz.crm.dms.business.order.sdk.vo.OrderPayPreviewVo;
import com.biz.crm.dms.business.order.sdk.vo.OrderPreviewVo;
import com.biz.crm.dms.business.order.verification.sdk.model.OrderVerificationContext;
import com.biz.crm.dms.business.order.verification.sdk.service.OrderVerificationService;
import com.biz.crm.mdm.business.dictionary.sdk.service.DictDataVoService;
import com.biz.crm.mdm.business.dictionary.sdk.vo.DictDataVo;
import com.biz.crm.mdm.business.product.sdk.service.ProductVoService;
import com.biz.crm.mdm.business.product.sdk.vo.ProductVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.google.common.collect.Lists;
import org.apache.commons.lang3.ArrayUtils;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.apache.commons.lang3.time.DateFormatUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.util.Collection;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

/**
 * @author he fan
 */
@Service
public class OrderConfirmServiceImpl implements OrderConfirmService {

  @Autowired(required = false)
  private OrderValidateVoService orderValidateVoService;

  @Autowired(required = false)
  private NebulaToolkitService nebulaToolkitService;

  @Autowired(required = false)
  private OrderVerificationService orderVerificationService;

  @Autowired(required = false)
  private List<OrderApportionVoService> orderApportionVoServiceList;

  @Autowired(required = false)
  private OrderApportionVoServiceChainInstance orderApportionVoServiceChainInstance;

  @Autowired(required = false)
  private OrderService orderService;

  @Autowired(required = false)
  private OrderFlowService orderFlowService;

  @Autowired(required = false)
  private PurchaseHistoryService purchaseHistoryService;

  @Autowired(required = false)
  private OrderAssist orderAssist;

  /**
   * 订单确认 (且有支付方式判断）
   *
   * @param orderConfirmDto 订单确认dto
   * @return {@link OrderConfirmVo}
   */
  @Transactional
  @Override
  public OrderConfirmVo handleConfirmWithPaymentMethod(OrderConfirmDto orderConfirmDto) {
    String orderPaymentMethod = orderConfirmDto.getOrderPaymentMethod();
    OrderConfirmVo orderConfirmVo = null;
    if (OrderPaymentMethodEnum.COST_POOL.getDictCode().equals(orderPaymentMethod)) {
      // 资金 + 授信
      orderConfirmVo = this.handleConfirm(orderConfirmDto);
    }
    if (OrderPaymentMethodEnum.ONLINE_PAYMENT.getDictCode().equals(orderPaymentMethod)) {
      // 中金 条码 聚合码 正扫
      orderConfirmVo = this.handleConfirmWithoutCostPool(orderConfirmDto);
    }
    return orderConfirmVo;
  }

  /**
   * 中金 条码 聚合码 正扫
   * - 不用费用池的订单
   * @param order
   * @return {@link OrderConfirmVo}
   */
  private OrderConfirmVo handleConfirmWithoutCostPool(OrderConfirmDto order) {
    //确保会有订单编号
    Order orderDb = this.setOrderCode(order);
    // 过滤掉赠品信息
    this.orderAssist.filterAndInitializeData(order);
    // 查询订单配置
    OrderConfigVo orderConfigVo = this.findOrderConfig(order);
    OrderDto orderContextDto = this.nebulaToolkitService.copyObjectByBlankList(order, OrderDto.class, HashSet.class, LinkedList.class);
    // 校验商品关系
    OrderVerificationContext orderVerificationContext = this.validateOrder(orderContextDto, orderConfigVo, OrderValidationProcessNodeEnum.ONE);
    // 分摊计算 -- 分摊流水数据持久化
    OrderApportionVo orderApportionVo = this.apportionPromotion(orderConfigVo, orderContextDto, OrderCommitPhaseEnum.CONFIRM);
    // 组装返回vo
    // - 将货补，与本品，赠品 收集在一起
    OrderConfirmVo orderConfirmVo = this.getOrderConfirmVo(order, orderApportionVo);
    // - 特殊订单的数据构造
    this.buildTheSpecialOrderData(orderConfirmVo);
    // - 返回结算信息的部分展示数据
    this.tabulateData(orderConfirmVo);
    // - 特殊订单的数据清洗
    this.cleanTheSpecialOrderData(orderConfirmVo);
    // 后续校验
    List<OrderValidationProcessNodeEnum> list = Lists.newArrayList(
        OrderValidationProcessNodeEnum.THREE,
        OrderValidationProcessNodeEnum.TEN
    );
    this.rearValidate(order, list , orderVerificationContext, orderConfirmVo);
    // 保存订单数据
    this.saveOrder(order, orderDb, orderConfigVo, orderConfirmVo);
    // 不扣减费用，并发起审批流程
    this.orderFlowService.occupyResource(order, true);
    return orderConfirmVo;
  }


  /**
   * 订单提交--带结算信息的订单预览
   * --前端将所有信息传入【基本信息、商品信息、政策信息、结算信息】
   * --这里要做基本业务的校验
   * --不做任何数据记录处理
   * --留意，不在【配置管理】设置内的其他校验事项
   * --预计算、最终分摊结果展示
   *
   * @param order
   * @return
   */
  @Override
  public OrderConfirmVo handleSubmit(OrderConfirmDto order, List<OrderValidationProcessNodeEnum> exclude) {
    /**
     * 行为的操作步骤
     * -- 查询订单配置
     * -- 业务校验
     * --不做任何数据记录处理
     * --分摊service排序后，获取分摊chain
     * --组装返回vo
     * --返回结算信息的部分展示数据
     * --数据完整性校验
     * --结算信息里带过来的特殊校验
     */
    // 过滤掉赠品信息
    this.orderAssist.filterAndInitializeData(order);
    // 查询订单配置
    OrderConfigVo orderConfigVo = this.findOrderConfig(order);
    OrderDto orderContextDto = this.nebulaToolkitService.copyObjectByBlankList(order, OrderDto.class, HashSet.class, LinkedList.class);
    // 校验商品关系
    OrderVerificationContext orderVerificationContext = this.validateOrder(orderContextDto, orderConfigVo, OrderValidationProcessNodeEnum.ONE);
    // 分摊计算 -- 不做任何数据记录处理
    OrderApportionVo orderApportionVo = this.apportionPromotion(orderConfigVo, orderContextDto, OrderCommitPhaseEnum.SUBMIT);
    // 组装返回vo
    // - 将货补，与本品，赠品 收集在一起
    OrderConfirmVo orderConfirmVo = this.getOrderConfirmVo(order, orderApportionVo);
    // - 特殊订单的数据构造
    this.buildTheSpecialOrderData(orderConfirmVo);
    // - 返回结算信息的部分展示数据
    this.tabulateData(orderConfirmVo);
    // - 特殊订单的数据清洗
    this.cleanTheSpecialOrderData(orderConfirmVo);
    // - 按照实际支付设置支付金额
    this.setOrderPays(orderConfirmVo, order);
    // 后续校验
    this.rearValidate(order, exclude, orderVerificationContext, orderConfirmVo);
    // 补sku图片
    this.findSkuMedia(orderConfirmVo);
    return orderConfirmVo;
  }

  /**
   * 后续校验
   *
   * @param order                    订单
   * @param exclude                  排除
   * @param orderVerificationContext 为了验证上下文
   * @param orderConfirmVo           订单确认签证官
   */
  private void rearValidate(OrderConfirmDto order, List<OrderValidationProcessNodeEnum> exclude, OrderVerificationContext orderVerificationContext, OrderConfirmVo orderConfirmVo) {
    OrderTabulateDto orderTabulateDto = this.nebulaToolkitService.copyObjectByBlankList(orderConfirmVo, OrderTabulateDto.class, HashSet.class, LinkedList.class);
    orderVerificationContext.setOrderTabulateDto(orderTabulateDto);
    OrderDto orderDto = this.nebulaToolkitService.copyObjectByBlankList(order, OrderDto.class, HashSet.class, LinkedList.class);
    orderVerificationContext.setOrderDto(orderDto);
    List<OrderValidationProcessNodeEnum> list = Lists.newArrayList(
        OrderValidationProcessNodeEnum.TWO,
        OrderValidationProcessNodeEnum.THREE,
        OrderValidationProcessNodeEnum.FOUR,
        OrderValidationProcessNodeEnum.TEN
    );
    Collection subtract = org.apache.commons.collections.CollectionUtils.subtract(list, exclude);
    OrderValidationProcessNodeEnum[] orderValidationProcessNodeEnums = new OrderValidationProcessNodeEnum[subtract.size()];
    orderValidationProcessNodeEnums = (OrderValidationProcessNodeEnum[]) subtract.toArray(orderValidationProcessNodeEnums);
    this.validateOrderByProcessNodes(orderVerificationContext, orderValidationProcessNodeEnums);
  }

  @Autowired(required = false)
  private ProductVoService productVoService;

  /**
   * 找到sku媒体
   *
   * @param orderPreviewVo 订单预览签证官
   */
  private void findSkuMedia(OrderPreviewVo orderPreviewVo) {
    List<OrderDetailPreviewVo> orderDetails = orderPreviewVo.getOrderDetails();
    List<String> productCodeList = orderDetails.stream().map(OrderDetailPreviewVo::getGoodsCode).collect(Collectors.toList());
    // 获取sku的信息
    List<ProductVo> productList = this.productVoService.findDetailsByIdsOrProductCodes(null, productCodeList);
    Map<String, ProductVo> productCodeMap = new HashMap<>(productList.size());
    for (ProductVo productVo : productList) {
      productCodeMap.put(productVo.getProductCode(), productVo);
    }
    for (OrderDetailPreviewVo orderDetail : orderDetails) {
      String goodsCode = orderDetail.getGoodsCode();
      ProductVo productVo = productCodeMap.get(goodsCode);
      if (ObjectUtils.isEmpty(productVo)) {
        continue;
      }
      List<FileVo> pictureFileList = OrderAssist.getFileVos(productVo.getPictureMediaList());
      List<FileVo> videoFileList = OrderAssist.getFileVos(productVo.getVideoMediaList());
      orderDetail.setPictureMediaList(pictureFileList);
      orderDetail.setVideoMediaList(videoFileList);
    }
  }

  /**
   * 查询订单配置
   *
   * @param order
   * @return
   */
  private OrderConfigVo findOrderConfig(OrderConfirmDto order) {
    String orderType = order.getOrderType();
    String relateCode = order.getRelateCode();
    OrderConfigVo orderValidateDto = this.orderValidateVoService.findByOrderTypeAndCustomerCode(orderType, relateCode);
    return orderValidateDto;
  }


  /**
   * 把数据收集计算并制作成表格需要的数据
   *
   * @param orderConfirmVo
   */
  private void tabulateData(OrderConfirmVo orderConfirmVo) {
    //--商品总数
    BigDecimal totalCountGoods = BigDecimal.ZERO;
    // --订单总金额 = 所有商品行销售价*数量汇总
    BigDecimal totalOrderAmount = BigDecimal.ZERO;
    // --政策优惠金额 = 本单所有优惠金额汇总，商品优惠政策和整单优惠政策都包含
    BigDecimal policyDiscountAmount = BigDecimal.ZERO;
    // --使用货补金额 = 本单所有货补项金额汇总
    BigDecimal totalCompensatedAmount = BigDecimal.ZERO;
    // --使用折扣费用--总记账项目内该扣减的汇总
    BigDecimal totalDiscountAmount = BigDecimal.ZERO;
    List<OrderPayPreviewVo> orderPays = orderConfirmVo.getOrderPays();
    for (OrderPayPreviewVo orderPay : orderPays) {
      Boolean itemGroupType = orderPay.getItemGroupType();
      if (!itemGroupType) {
        BigDecimal itemAmount = orderPay.getItemAmount();
        itemAmount = Objects.nonNull(itemAmount) ? itemAmount : BigDecimal.ZERO;
        totalDiscountAmount = totalDiscountAmount.add(itemAmount);
      }
    }
    // --遍历订单明细
    List<OrderDetailPreviewVo> orderDetails = orderConfirmVo.getOrderDetails();
    if (!CollectionUtils.isEmpty(orderDetails)) {
      for (OrderDetailPreviewVo orderDetail : orderDetails) {
        BigDecimal quantity = orderDetail.getQuantity();
        totalCountGoods = totalCountGoods.add(quantity);
        // --行上的销售总金额
        BigDecimal salesAmount = orderDetail.getSalesAmount();
        // --订单明细的扣款项目
        // ----订单明细行上总的优惠(所有该扣 货补、折扣、政策)金额
        BigDecimal totalItemAmount = BigDecimal.ZERO;
        // ----订单明细行上总的政策优惠金额
        BigDecimal totalPolicyItemAmount = BigDecimal.ZERO;
        List<OrderDetailPayPreviewVo> orderDetailPays = orderDetail.getOrderDetailPays();
        if (!CollectionUtils.isEmpty(orderDetailPays)) {
          for (OrderDetailPayPreviewVo orderDetailPay : orderDetailPays) {
            Boolean available = orderDetailPay.getAvailable();
            if (available) {
              BigDecimal itemAmount = orderDetailPay.getItemAmount();
              totalItemAmount = totalItemAmount.add(itemAmount);
              // 优惠政策记账模块注册的项目组key
              String itemGroupKey = "salepolicyGroup";
              boolean equals = itemGroupKey.equals(orderDetailPay.getItemGroupKey());
              if (equals) {
                totalPolicyItemAmount = totalPolicyItemAmount.add(itemAmount);
              }
            }
          }
        }
        // --实际支付金额（组内分摊后）== 销售金额 - 优惠金额 （即，本应该支付金额）
        BigDecimal shouldPaymentAmount = salesAmount.subtract(totalItemAmount);
        orderDetail.setShouldPaymentAmount(shouldPaymentAmount);
        orderDetail.setAverageUnitPrice(shouldPaymentAmount.divide(quantity, 4, RoundingMode.HALF_UP));
        policyDiscountAmount = policyDiscountAmount.add(totalPolicyItemAmount);
        totalOrderAmount = totalOrderAmount.add(salesAmount);
        String itemType = orderDetail.getItemType();
        if (itemType.equals(ItemTypeEnum.COMPENSATED_GOODS.getDictCode())) {
          totalCompensatedAmount = totalCompensatedAmount.add(salesAmount);
        }
      }

    }
    orderConfirmVo.setTotalOrderAmount(totalOrderAmount);
    // --本单实际支付 ==  订单总金额（元）-政策优惠金额（元）-使用货补金额-使用折扣费用（元）
    BigDecimal actualAmountPaid = totalOrderAmount.subtract(policyDiscountAmount).subtract(totalCompensatedAmount).subtract(totalDiscountAmount);
    orderConfirmVo.setActualAmountPaid(actualAmountPaid);
    orderConfirmVo.setPolicyDiscountAmount(policyDiscountAmount);
    orderConfirmVo.setTotalCompensatedAmount(totalCompensatedAmount);
    orderConfirmVo.setTotalCountGoods(totalCountGoods);
    orderConfirmVo.setCreateTime(new Date());
    // --订单的总的该扣项目: 已经从dto拷入vo

    // -- 去负分摊
    this.pruneNegativeAmount(orderConfirmVo);

  }

  /**
   * 删除负数金额
   *
   * @param orderConfirmVo 订单确认签证官
   */
  @SuppressWarnings("unused")
  private void pruneNegativeAmount(OrderConfirmVo orderConfirmVo) {
    String orderCategory = orderConfirmVo.getOrderCategory();
    boolean isMaterial = orderCategory.equals(OrderCategoryEnum.MATERIAL_ORDER.getDictCode());
    if (isMaterial) {
      return;
    }
    // 本地实际支付应该大于 等于 零
    BigDecimal actualAmountPaid = orderConfirmVo.getActualAmountPaid();
    int compare = BigDecimal.ZERO.compareTo(actualAmountPaid);
    if (compare == 1) {
      // 本方法不处理应付为负的情况。
      return;
    }
    Map<String, BigDecimal> ToNegativeAmountMap = new HashMap<>();
    this.pruneNegativeAmount(orderConfirmVo, ToNegativeAmountMap, 0);
    if (!CollectionUtils.isEmpty(ToNegativeAmountMap)) {
      throw new IllegalArgumentException("订单去负分摊，失败");
    }
  }

  /**
   * - 行应付必须小于或等于总该付
   * - 行应付不能为负数
   *
   * @param orderConfirmVo      订单确认签证官
   * @param toNegativeAmountMap 负金额地图
   * @param recursionTimes      递归次数
   */
  private void pruneNegativeAmount(OrderConfirmVo orderConfirmVo, Map<String, BigDecimal> toNegativeAmountMap, int recursionTimes) {
    List<OrderDetailPreviewVo> orderDetails = orderConfirmVo.getOrderDetails();
    // - 为了避免死循环
    if (recursionTimes > orderDetails.size()) {
      return;
    }
    // - 遍历商品行，逐行调整价格
    for (OrderDetailPreviewVo orderDetail : orderDetails) {
      // 负处理
      this.lessThanZeroOperation(orderDetail, toNegativeAmountMap);
      // 零处理
      this.equalToZeroOperation(orderDetail, toNegativeAmountMap);
      // 正处理
      this.greaterThanZeroOperation(orderDetail, toNegativeAmountMap);
    }
    if (!CollectionUtils.isEmpty(toNegativeAmountMap)) {
      this.pruneNegativeAmount(orderConfirmVo, toNegativeAmountMap, ++recursionTimes);
    }
  }

  /**
   * 大于零操作
   * - 分摊成0，或者小于等于总该付
   *
   * @param orderDetail         订单细节
   * @param toNegativeAmountMap
   */
  private void greaterThanZeroOperation(OrderDetailPreviewVo orderDetail, Map<String, BigDecimal> toNegativeAmountMap) {
    // - 行应付（实际支付金额）
    BigDecimal shouldPaymentAmount = orderDetail.getShouldPaymentAmount();
    // - 判断应付正数
    boolean isPositive = BigDecimal.ZERO.compareTo(shouldPaymentAmount) < 0;
    if (!isPositive) {
      return;
    }
    //-- 分摊成0，或者小于等于总该付
    List<OrderDetailPayPreviewVo> orderDetailPays = orderDetail.getOrderDetailPays();
    if (CollectionUtils.isEmpty(orderDetailPays)) {
      return;
    }
    //-- 分摊缓存,遍历抵扣项目
    for (OrderDetailPayPreviewVo orderDetailPay : orderDetailPays) {
      //-- 行应付从正数调整为0后，跳出调整
      shouldPaymentAmount = orderDetail.getShouldPaymentAmount();
      if (this.isZero(shouldPaymentAmount)) {
        break;
      }
      String itemKey = orderDetailPay.getItemKey();
      String originData = orderDetailPay.getOriginData();
      String mapKey = StringUtils.join(itemKey, originData);
      // 缓存抵扣项目的借出金额
      BigDecimal lendAmount = toNegativeAmountMap.get(mapKey);
      // 此项没有借出
      if (this.isZero(lendAmount)) {
        continue;
      }
      int compare = shouldPaymentAmount.compareTo(lendAmount);
      //--- 行应付 大于 缓存抵扣项目
      if (compare == 1) {
        BigDecimal adjust = lendAmount;
        BigDecimal itemAmount = orderDetailPay.getItemAmount();
        //---- 抵扣项目（增加）
        orderDetailPay.setItemAmount(itemAmount.add(adjust));
        // 清空抵扣项目缓存
        BigDecimal surplus = lendAmount.subtract(adjust);
        if (this.isZero(surplus)) {
          toNegativeAmountMap.remove(mapKey);
        } else {
          toNegativeAmountMap.put(mapKey, surplus);
        }
        // 行应付重置(减少）
        BigDecimal surplusShouldPayment = shouldPaymentAmount.subtract(adjust);
        orderDetail.setShouldPaymentAmount(surplusShouldPayment);
      } else {
        //--- 行应付 小于 缓存抵扣项目
        BigDecimal adjust = shouldPaymentAmount;
        BigDecimal itemAmount = orderDetailPay.getItemAmount();
        //---- 抵扣项目（增加）
        orderDetailPay.setItemAmount(itemAmount.add(adjust));
        // 缓存抵扣项目重置
        BigDecimal surplus = lendAmount.subtract(adjust);
        if (this.isZero(surplus)) {
          toNegativeAmountMap.remove(mapKey);
        } else {
          toNegativeAmountMap.put(mapKey, surplus);
        }
        // 行应付重置(减少）
        BigDecimal surplusShouldPayment = shouldPaymentAmount.subtract(adjust);
        orderDetail.setShouldPaymentAmount(surplusShouldPayment);
      }
    }
  }

  /**
   * 等于零操作
   *
   * @param orderDetail         订单细节
   * @param toNegativeAmountMap 负金额地图
   */
  private void equalToZeroOperation(OrderDetailPreviewVo orderDetail, Map<String, BigDecimal> toNegativeAmountMap) {
    // - 行应付（实际支付金额）
    BigDecimal shouldPaymentAmount = orderDetail.getShouldPaymentAmount();
    // - 零
    if (!this.isZero(shouldPaymentAmount)) {
      return;
    }
    //-- 抵扣项目缓存为空，跳过
    if (CollectionUtils.isEmpty(toNegativeAmountMap)) {
      return;
    }
    // 借出队列，借进队列
    List<OrderDetailPayPreviewVo> lendList = Lists.newLinkedList();
    List<OrderDetailPayPreviewVo> borrowList = Lists.newLinkedList();
    //借出容器余量
    BigDecimal lendTotalAmount = BigDecimal.ZERO;
    // - 遍历抵扣项目;
    List<OrderDetailPayPreviewVo> orderDetailPays = orderDetail.getOrderDetailPays();
    for (OrderDetailPayPreviewVo orderDetailPay : orderDetailPays) {
      BigDecimal itemAmount = orderDetailPay.getItemAmount();
      String itemKey = orderDetailPay.getItemKey();
      String originData = orderDetailPay.getOriginData();
      String mapKey = StringUtils.join(itemKey, originData);
      boolean contains = toNegativeAmountMap.containsKey(mapKey);
      if (contains) {
        // 本行可以借进的（本行存在的项目,且，借出容器含该抵扣项目）
        borrowList.add(orderDetailPay);
        lendTotalAmount = lendTotalAmount.add(toNegativeAmountMap.get(mapKey));
      } else {
        // 本行可以借出的（大于0的项目，且，借出容器不含该抵扣项目）
        int compare = BigDecimal.ZERO.compareTo(itemAmount);
        if (compare == -1) {
          lendList.add(orderDetailPay);
        }
      }
    }
    //行的借出能力 决定 借进量
    BigDecimal reduce = lendList.stream().map(OrderDetailPayPreviewVo::getItemAmount).reduce(BigDecimal.ZERO, BigDecimal::add);
    //借出容器余量 决定 借进量
    //借进量
    BigDecimal borrowAmount;
    if (lendTotalAmount.compareTo(reduce) == 1) {
      borrowAmount = reduce;
    } else {
      borrowAmount = lendTotalAmount;
    }
    //有借进才能借出
    if (this.isZero(borrowAmount)) {
      return;
    }
    //确定总量后，挨个借
    BigDecimal tempLend = borrowAmount.subtract(BigDecimal.ZERO);
    // 遍历借出队列
    for (OrderDetailPayPreviewVo orderDetailPay : lendList) {
      if (this.isZero(tempLend)) {
        break;
      }
      String itemKey = orderDetailPay.getItemKey();
      String originData = orderDetailPay.getOriginData();
      String mapKey = StringUtils.join(itemKey, originData);
      BigDecimal adjust;
      BigDecimal itemAmount = orderDetailPay.getItemAmount();
      int compare = tempLend.compareTo(itemAmount);
      if (compare == 1) {
        adjust = itemAmount;
      } else {
        adjust = tempLend;
      }
      tempLend = tempLend.subtract(adjust);
      //-- 抵扣项目（减少）,
      orderDetailPay.setItemAmount(itemAmount.subtract(adjust));
      //调整借出缓存
      toNegativeAmountMap.put(mapKey, adjust);
    }
    BigDecimal tempBorrow = borrowAmount.subtract(BigDecimal.ZERO);
    // 遍历借进队列
    for (OrderDetailPayPreviewVo orderDetailPay : borrowList) {
      if (this.isZero(tempBorrow)) {
        break;
      }
      String itemKey = orderDetailPay.getItemKey();
      String originData = orderDetailPay.getOriginData();
      String mapKey = StringUtils.join(itemKey, originData);
      BigDecimal adjust;
      BigDecimal lendAmount = toNegativeAmountMap.get(mapKey);
      int compare = tempBorrow.compareTo(lendAmount);
      if (compare == 1) {
        adjust = lendAmount;
      } else {
        adjust = tempBorrow;
      }
      tempBorrow = tempBorrow.subtract(adjust);
      //-- 抵扣项目（增加）,
      BigDecimal itemAmount = orderDetailPay.getItemAmount();
      orderDetailPay.setItemAmount(itemAmount.add(adjust));
      //-- 缓存抵扣项目（借出）
      BigDecimal surplus = lendAmount.subtract(adjust);
      if (this.isZero(surplus)) {
        toNegativeAmountMap.remove(mapKey);
      } else {
        toNegativeAmountMap.put(mapKey, surplus);
      }
    }
  }

  /**
   * 小于零操作
   * - 应付转0
   */
  private void lessThanZeroOperation(OrderDetailPreviewVo orderDetail, Map<String, BigDecimal> toNegativeAmountMap) {
    // - 行应付（实际支付金额）
    BigDecimal shouldPaymentAmount = orderDetail.getShouldPaymentAmount();
    // - 判断应付正负数
    boolean isNegative = BigDecimal.ZERO.compareTo(shouldPaymentAmount) > 0;
    if (!isNegative) {
      return;
    }
    // - 遍历抵扣项目;
    List<OrderDetailPayPreviewVo> orderDetailPays = orderDetail.getOrderDetailPays();
    for (OrderDetailPayPreviewVo orderDetailPay : orderDetailPays) {
      //-- 行应付从负数调整为0后，跳出调整
      shouldPaymentAmount = orderDetail.getShouldPaymentAmount();
      if (this.isZero(shouldPaymentAmount)) {
        break;
      }
      // - 抵扣项目无钱可用,跳过该抵扣项目
      BigDecimal itemAmount = orderDetailPay.getItemAmount();
      if (this.isZero(itemAmount)) {
        continue;
      }
      String itemKey = orderDetailPay.getItemKey();
      String originData = orderDetailPay.getOriginData();
      String mapKey = StringUtils.join(itemKey, originData);
      //-- 抵扣项目只要不为负数，能存一点是一点，分摊应付。
      // 比实际支付金额的绝对值大或相等的扣减项
      boolean ge = itemAmount.compareTo(shouldPaymentAmount.abs()) > -1;
      if (ge) {
        BigDecimal adjust = shouldPaymentAmount.abs();
        //-- 抵扣项目（减少）,
        orderDetailPay.setItemAmount(itemAmount.subtract(adjust));
        //-- 调整应付
        orderDetail.setShouldPaymentAmount(shouldPaymentAmount.add(adjust));
        //-- 缓存抵扣项目（借出）
        BigDecimal amassNegateAbs = toNegativeAmountMap.get(mapKey);
        if (this.isZero(amassNegateAbs)) {
          toNegativeAmountMap.put(mapKey, adjust);
        } else {
          toNegativeAmountMap.put(mapKey, amassNegateAbs.add(adjust));
        }
      } else {
        BigDecimal adjust = itemAmount;
        //-- 抵扣项目（减少）,
        orderDetailPay.setItemAmount(itemAmount.subtract(adjust));
        //-- 调整应付
        orderDetail.setShouldPaymentAmount(shouldPaymentAmount.add(adjust));
        //-- 缓存抵扣项目（借出）
        BigDecimal amassNegateAbs = toNegativeAmountMap.get(mapKey);
        if (this.isZero(amassNegateAbs)) {
          toNegativeAmountMap.put(mapKey, adjust);
        } else {
          toNegativeAmountMap.put(mapKey, amassNegateAbs.add(adjust));
        }
      }
    }
    shouldPaymentAmount = orderDetail.getShouldPaymentAmount();
    if (!this.isZero(shouldPaymentAmount)) {
      // 到这还不是0，就是异常数据了。
      throw new IllegalArgumentException("下单时，商品行的金额始终为负数");
    }
  }

  /**
   * 为空 或 为零
   *
   * @param bigDecimal 项目数量
   * @return boolean
   */
  private boolean isZero(BigDecimal bigDecimal) {
    boolean isZero = bigDecimal == null || bigDecimal.compareTo(BigDecimal.ZERO) == 0;
    return isZero;
  }

  @Autowired(required = false)
  private GenerateCodeService generateCodeService;

  /**
   * 订单确认--不是预览，要生成真的订单
   * --不通过基本业务数据校验，不做落库处理
   * --与【结算预览】的区别只有持久化数据
   *
   * @param order
   * @return
   */
  @Transactional
  @Override
  public OrderConfirmVo handleConfirm(OrderConfirmDto order) {
    /**
     * 行为的操作步骤
     * -- 查询订单配置
     * -- 校验商品关系
     * -- 分摊计算
     * ----分摊流水数据持久化
     * ----分摊service排序后，获取分摊chain
     * --组装返回vo
     * --返回结算信息的部分展示数据
     * -- 后续校验
     * -- 数据持久化
     * -- 占用
     * -- 常购清单记录
     */
    //确保会有订单编号
    Order orderDb = this.setOrderCode(order);
    // 过滤掉赠品信息
    this.orderAssist.filterAndInitializeData(order);
    // 查询订单配置
    OrderConfigVo orderConfigVo = this.findOrderConfig(order);
    OrderDto orderContextDto = this.nebulaToolkitService.copyObjectByBlankList(order, OrderDto.class, HashSet.class, LinkedList.class);
    // 校验商品关系
    OrderVerificationContext orderVerificationContext = this.validateOrder(orderContextDto, orderConfigVo, OrderValidationProcessNodeEnum.ONE);
    // 分摊计算 -- 分摊流水数据持久化
    OrderApportionVo orderApportionVo = this.apportionPromotion(orderConfigVo, orderContextDto, OrderCommitPhaseEnum.CONFIRM);
    // 组装返回vo
    // - 将货补，与本品，赠品 收集在一起
    OrderConfirmVo orderConfirmVo = this.getOrderConfirmVo(order, orderApportionVo);
    // - 特殊订单的数据构造
    this.buildTheSpecialOrderData(orderConfirmVo);
    // - 返回结算信息的部分展示数据
    this.tabulateData(orderConfirmVo);
    // - 特殊订单的数据清洗
    this.cleanTheSpecialOrderData(orderConfirmVo);
    // - 按照实际支付设置支付金额
    this.setOrderPays(orderConfirmVo, order);
    // 后续校验
    this.rearValidate(order, Lists.newLinkedList(), orderVerificationContext, orderConfirmVo);
    // 保存订单数据
    this.saveOrder(order, orderDb, orderConfigVo, orderConfirmVo);
    // 扣减费用，并发起审批流程
    this.orderFlowService.pay(order);
    return orderConfirmVo;
  }

  /**
   * 验证订单
   *
   * @param orderContextDto        订单上下文dto
   * @param orderConfigVo          订单配置签证官
   * @param validationProcessNodes 验证过程节点
   * @return {@link OrderVerificationContext}
   */
  private OrderVerificationContext validateOrder(OrderDto orderContextDto, OrderConfigVo orderConfigVo, OrderValidationProcessNodeEnum... validationProcessNodes) {
    // - 订单校验上下文
    OrderVerificationContext orderVerificationContext = new OrderVerificationContext();
    orderVerificationContext.setOrderDto(orderContextDto);
    orderVerificationContext.setOrderConfigVo(orderConfigVo);
    return this.validateOrderByProcessNodes(orderVerificationContext, validationProcessNodes);
  }

  /**
   * 按照订单流程节点校验订单
   *
   * @param orderVerificationContext 订单验证上下文
   * @param validationProcessNodes   验证过程节点
   * @return {@link OrderVerificationContext}
   */
  private OrderVerificationContext validateOrderByProcessNodes(OrderVerificationContext orderVerificationContext, OrderValidationProcessNodeEnum... validationProcessNodes) {
    if (ArrayUtils.isEmpty(validationProcessNodes)) {
      return orderVerificationContext;
    }
    for (OrderValidationProcessNodeEnum validationProcessNode : validationProcessNodes) {
      orderVerificationContext.setProcessNodeEnum(validationProcessNode);
      this.orderVerificationService.execute(orderVerificationContext);
    }
    return orderVerificationContext;
  }

  /**
   * 分配促销
   *
   * @param orderConfigVo   订单配置签证官
   * @param orderContextDto 订单上下文dto
   * @return {@link OrderApportionVo}
   */
  private OrderApportionVo apportionPromotion(OrderConfigVo orderConfigVo, OrderDto orderContextDto, OrderCommitPhaseEnum phaseEnum) {
    // 分摊计算,组装分摊计算用的上下文,并设置提交阶段枚举
    OrderApportionContext context = new OrderApportionContext();
    context.setOrderConfigVo(orderConfigVo);
    context.setOrderDto(orderContextDto);
    context.setOrderCommitPhaseEnum(phaseEnum);
    // -- 分摊service排序后，获取分摊chain，分摊service内部根据订单提交阶段 自行决定是否执行
    OrderApportionVo orderApportionVo = this.apportion(context);
    return orderApportionVo;
  }

  /**
   * 保存订单
   *
   * @param order          订单
   * @param orderDb        订单数据库
   * @param orderConfigVo  订单配置签证官
   * @param orderConfirmVo 订单确认签证官
   */
  private void saveOrder(OrderConfirmDto order, Order orderDb, OrderConfigVo orderConfigVo, OrderConfirmVo orderConfirmVo) {
    Order orderInsert = this.nebulaToolkitService.copyObjectByBlankList(orderConfirmVo, Order.class, HashSet.class, LinkedList.class);
    if (Objects.nonNull(orderDb)) {
      String orderStatus = orderDb.getOrderStatus();
      boolean isRejected = OrderStatusEnum.REJECTED.getDictCode().equals(orderStatus);
      boolean isCanceled = OrderStatusEnum.CANCELED.getDictCode().equals(orderStatus);
      if (orderDb.getDraft()) {
        //--删除草稿信息,
        orderService.deleteDraftByOrderCode(orderConfirmVo.getOrderCode());
        orderInsert.setId(orderDb.getId());
      } else if (isRejected || isCanceled) {
        // 驳回和撤回的订单，再提交，重新搞个订单
        orderInsert.setId(null);
      } else {
        String message = "[%s]状态的订单不允许提交";
        OrderStatusEnum orderStatusEnum = OrderStatusEnum.findByCode(orderStatus);
        throw new IllegalArgumentException(String.format(message, orderStatusEnum.getValue()));
      }
    }
    orderInsert.setDraft(false);
    if (ObjectUtils.isNotEmpty(orderConfigVo)) {
      orderInsert.setSplitPrice(orderConfigVo.getIsKneadingPrice());
    }
    orderInsert.setQuantityOfCommodity(orderConfirmVo.getTotalCountGoods());
    orderInsert.setOrderStatus(OrderStatusEnum.AWAIT_APPROVE.getDictCode());
    orderInsert.setProcessKey(order.getProcessKey());
    orderInsert.setProcessRemark(order.getProcessRemark());
    this.orderService.create(orderInsert);
    orderConfirmVo.setId(orderInsert.getId());
    if (!order.getOrderCategory().equals(OrderCategoryEnum.MATERIAL_ORDER.getDictCode())) {
      // 记录常购清单
      this.recordPurchaseHistory(orderInsert);
    }
  }

  /**
   * 实例化 OrderConfirmVo
   * - 将货补，与本品，赠品 收集在一起
   *
   * @param order            订单
   * @param orderApportionVo 订单分配签证官
   * @return {@link OrderConfirmVo}
   */
  private OrderConfirmVo getOrderConfirmVo(OrderConfirmDto order, OrderApportionVo orderApportionVo) {
    // --入参中的原始数据拷入vo
    OrderConfirmVo orderConfirmVo = nebulaToolkitService.copyObjectByBlankList(order, OrderConfirmVo.class, HashSet.class, LinkedList.class);
    // --分摊数据拷入另一个vo
    OrderConfirmVo apportionOrderConfirmVo = nebulaToolkitService.copyObjectByBlankList(orderApportionVo, OrderConfirmVo.class, HashSet.class, LinkedList.class);
    // --orderDetails需要替换
    // ----全部入参（含货补）
    List<OrderDetailPreviewVo> inputOrderDetails = orderConfirmVo.getOrderDetails();
    // ----只有本品和赠品
    List<OrderDetailPreviewVo> orderDetails = apportionOrderConfirmVo.getOrderDetails();
    // ----将货补，与本品，赠品 收集在一起
    List<OrderDetailPreviewVo> unionSet = OrderAssist.unionOrderDetails(inputOrderDetails, orderDetails);
    orderConfirmVo.setOrderDetails(unionSet);
    return orderConfirmVo;
  }

  @Transactional
  @Override
  public OrderConfirmVo handleConfirmWithoutResource(OrderConfirmDto order) {
    //确保会有订单编号
    Order orderDb = this.setOrderCode(order);
    // 过滤掉赠品信息
    this.orderAssist.filterAndInitializeData(order);
    // 查询订单配置
    OrderConfigVo orderConfigVo = this.findOrderConfig(order);
    OrderDto orderContextDto = this.nebulaToolkitService.copyObjectByBlankList(order, OrderDto.class, HashSet.class, LinkedList.class);
    // 校验商品关系
    OrderVerificationContext orderVerificationContext = this.validateOrder(orderContextDto, orderConfigVo, OrderValidationProcessNodeEnum.ONE);
    // 分摊计算 -- 分摊流水数据持久化
    OrderApportionVo orderApportionVo = this.apportionPromotion(orderConfigVo, orderContextDto, OrderCommitPhaseEnum.CONFIRM);
    // 组装返回vo
    // - 将货补，与本品，赠品 收集在一起
    OrderConfirmVo orderConfirmVo = this.getOrderConfirmVo(order, orderApportionVo);
    // - 特殊订单的数据构造
    this.buildTheSpecialOrderData(orderConfirmVo);
    // - 返回结算信息的部分展示数据
    this.tabulateData(orderConfirmVo);
    // - 特殊订单的数据清洗
    this.cleanTheSpecialOrderData(orderConfirmVo);
    // - 按照实际支付设置支付金额
    this.setOrderPays(orderConfirmVo, order);
    // 后续校验
    this.rearValidate(order, Lists.newArrayList(OrderValidationProcessNodeEnum.THREE), orderVerificationContext, orderConfirmVo);
    // 保存订单数据
    this.saveOrder(order, orderDb, orderConfigVo, orderConfirmVo);
    // 占用库存
    this.orderFlowService.occupyStock(order);
    return orderConfirmVo;
  }

  /**
   * 建立特殊订单数据
   *
   * @param orderConfirmVo 订单确认签证官
   */
  private void buildTheSpecialOrderData(OrderConfirmVo orderConfirmVo) {
    this.buildTheDiscountOrderData(orderConfirmVo);
  }

  /**
   * 特殊订单的数据清洗
   *
   * @param orderConfirmVo
   */
  private void cleanTheSpecialOrderData(OrderConfirmVo orderConfirmVo) {
    this.cleanTheFreeOrderData(orderConfirmVo);
  }

  @Qualifier("DiscountApportionTallyItemRegister")
  @Autowired(required = false)
  private TallyItemRegister discountTallyItemRegister;

  /**
   * 构建折扣订单数据
   *
   * @param orderConfirmVo 订单确认签证官
   */
  private void buildTheDiscountOrderData(OrderConfirmVo orderConfirmVo) {
    String orderType = orderConfirmVo.getOrderType();
    boolean isDiscount = OrderTypeEnum.DISCOUNT.getDictCode().equals(orderType);
    if (!isDiscount) {
      return;
    }
    TallyItemRegisterModel tallyItemRegisterModel = discountTallyItemRegister.findTallyItemRegisterModel();
    List<OrderDetailPreviewVo> orderDetails = orderConfirmVo.getOrderDetails();
    for (OrderDetailPreviewVo orderDetail : orderDetails) {
      BigDecimal salesAmount = orderDetail.getSalesAmount();
      List<OrderDetailPayPreviewVo> orderDetailPayPreviewVos = Lists.newLinkedList();
      OrderDetailPayPreviewVo orderDetailPayPreviewVo = new OrderDetailPayPreviewVo();
      orderDetailPayPreviewVo.setOrderDetailCode(orderDetail.getOrderDetailCode());
      orderDetailPayPreviewVo.setOrderCode(orderConfirmVo.getOrderCode());
      orderDetailPayPreviewVo.setItemGroupKey(tallyItemRegisterModel.getItemGroupKey());
      orderDetailPayPreviewVo.setItemKey(tallyItemRegisterModel.getItemKey());
      orderDetailPayPreviewVo.setItemName(tallyItemRegisterModel.getItemName());
      orderDetailPayPreviewVo.setItemAmount(salesAmount);
      orderDetailPayPreviewVo.setAvailable(true);
      orderDetailPayPreviewVos.add(orderDetailPayPreviewVo);
      orderDetail.setOrderDetailPays(orderDetailPayPreviewVos);
    }
  }

  /**
   * 清洁免费订单数据
   *
   * @param orderConfirmVo
   */
  private void cleanTheFreeOrderData(OrderConfirmVo orderConfirmVo) {
    String orderType = orderConfirmVo.getOrderType();
    boolean isFree = OrderTypeEnum.FREE.getDictCode().equals(orderType);
    if (!isFree) {
      return;
    }
    List<OrderDetailPreviewVo> orderDetails = orderConfirmVo.getOrderDetails();
    for (OrderDetailPreviewVo orderDetail : orderDetails) {
      orderDetail.setShouldPaymentAmount(BigDecimal.ZERO);
    }
    orderConfirmVo.setActualAmountPaid(BigDecimal.ZERO);
  }

  @Override
  public OrderConfirmVo createOrUpdate(OrderConfirmDto order) {
    Order orderInsert = this.nebulaToolkitService.copyObjectByBlankList(order, Order.class, HashSet.class, LinkedList.class);
    Order orderResult = this.orderService.create(orderInsert);

    return this.nebulaToolkitService.copyObjectByBlankList(orderResult, OrderConfirmVo.class, HashSet.class, LinkedList.class);
  }

  @Qualifier("CapitalTallyItemRegister")
  @Autowired(required = false)
  private TallyItemRegister capitalTallyItemRegister;

  @Qualifier("CreditTallyItemRegister")
  @Autowired(required = false)
  private TallyItemRegister creditTallyItemRegister;

  @Autowired(required = false)
  @Qualifier("costPoolCapitalStrategyImpl")
  private CostPoolStrategy costPoolCapitalStrategy;

  @Autowired(required = false)
  @Qualifier("costPoolCapitalRegisterImpl")
  private CostPoolRegister costPoolCapitalRegister;

  @Autowired(required = false)
  @Qualifier("CostPoolCreditStrategyImpl")
  private CostPoolStrategy costPoolCreditStrategy;

  @Autowired(required = false)
  @Qualifier("CostPoolCreditRegisterImpl")
  private CostPoolRegister costPoolCreditRegister;


  /**
   * 设置订单支付
   *
   * @param order           订单
   * @param orderConfirmDto
   */
  private void setOrderPays(OrderPreviewVo order, OrderConfirmDto orderConfirmDto) {
    List<OrderPayPreviewVo> orderPays = order.getOrderPays();
    if (CollectionUtils.isEmpty(orderPays)) {
      orderPays = Lists.newLinkedList();
      order.setOrderPays(orderPays);
    }
    // - 清空支付项目
    Iterator<OrderPayPreviewVo> iterator = orderPays.iterator();
    while (iterator.hasNext()) {
      OrderPayPreviewVo next = iterator.next();
      Boolean itemGroupType = next.getItemGroupType();
      if (itemGroupType) {
        iterator.remove();
      }
    }
    // 特殊订单不予处理
    String orderType = order.getOrderType();
    boolean isFree = OrderTypeEnum.FREE.getDictCode().equals(orderType);
    if (isFree) {
      return;
    }
    // 资金
    TallyItemRegisterModel capitalTallyItem = capitalTallyItemRegister.findTallyItemRegisterModel();
    OrderPayPreviewVo capital = new OrderPayPreviewVo();
    Boolean pay = this.isPay(capitalTallyItem);
    capital.setItemGroupType(pay);
    capital.setItemGroupKey(capitalTallyItem.getItemGroupKey());
    capital.setItemKey(capitalTallyItem.getItemKey());
    capital.setItemName(capitalTallyItem.getItemName());
    orderPays.add(capital);
    // 授信
    TallyItemRegisterModel creditTallyItem = creditTallyItemRegister.findTallyItemRegisterModel();
    OrderPayPreviewVo credit = new OrderPayPreviewVo();
    Boolean isPay = this.isPay(creditTallyItem);
    credit.setItemGroupType(isPay);
    credit.setItemGroupKey(creditTallyItem.getItemGroupKey());
    credit.setItemKey(creditTallyItem.getItemKey());
    credit.setItemName(creditTallyItem.getItemName());
    orderPays.add(credit);
    // 分钱
    BigDecimal actualAmountPaid = order.getActualAmountPaid();
    // - 可用资金余额
    BigDecimal capitalUsableAmount = this.findUsableAmount(costPoolCapitalRegister, order.getRelateCode(), costPoolCapitalStrategy);
    //- 可用信用余额
    BigDecimal creditUsableAmount = this.findUsableAmount(costPoolCreditRegister, order.getRelateCode(), costPoolCreditStrategy);
    //- 加上退货那部分钱
    List<OrderPayConfirmDto> refundOrderPays = orderConfirmDto.getRefundOrderPays();
    if (!CollectionUtils.isEmpty(refundOrderPays)) {
      for (OrderPayConfirmDto refundOrderPay : refundOrderPays) {
        BigDecimal itemAmount = refundOrderPay.getItemAmount();
        Boolean itemGroupType = refundOrderPay.getItemGroupType();
        if (itemGroupType == null && !itemGroupType) {
          continue;
        }
        boolean isCapital = capitalTallyItem.getItemKey().equals(refundOrderPay.getItemKey());
        if (isCapital) {
          capitalUsableAmount = capitalUsableAmount.add(itemAmount);
        }
        boolean isCredit = creditTallyItem.getItemKey().equals(refundOrderPay.getItemKey());
        if (isCredit) {
          creditUsableAmount = creditUsableAmount.add(itemAmount);
        }
      }
    }
    // - 分钱支付
    BigDecimal totalUsableAmount = capitalUsableAmount.add(creditUsableAmount);
    int compare = actualAmountPaid.compareTo(totalUsableAmount);
    if (compare > 0) {
      throw new IllegalArgumentException("余额不足，无法支付订单");
    } else if (compare < 0) {
      int compareCapital = actualAmountPaid.compareTo(capitalUsableAmount);
      if (compareCapital > 0) {
        capital.setItemAmount(capitalUsableAmount);
        credit.setItemAmount(actualAmountPaid.subtract(capitalUsableAmount));
      } else if (compareCapital < 0) {
        capital.setItemAmount(actualAmountPaid);
        credit.setItemAmount(BigDecimal.ZERO);
      } else {
        capital.setItemAmount(actualAmountPaid);
        credit.setItemAmount(BigDecimal.ZERO);
      }
    } else {
      capital.setItemAmount(capitalUsableAmount);
      credit.setItemAmount(creditUsableAmount);
    }
    /**
     * 入参中的也要覆盖
     */
    List<OrderPayConfirmDto> orderConfirmDtos = (List<OrderPayConfirmDto>) this.nebulaToolkitService.copyCollectionByBlankList(orderPays, OrderPayPreviewVo.class, OrderPayConfirmDto.class, HashSet.class, LinkedList.class);
    orderConfirmDto.setOrderPays(orderConfirmDtos);
  }

  private Boolean isPay(TallyItemRegisterModel creditTallyItem) {
    Integer itemGroupTypeInt = Integer.valueOf(creditTallyItem.getTallyItemGroupType().getDictCode());
    Boolean isPay = itemGroupTypeInt == 0 ? false : true;
    return isPay;
  }

  /**
   * 找到可用数量
   *
   * @param costPoolRegister 成本池注册
   * @param customerCode     客户代码
   * @param costPoolStrategy 成本池策略
   */
  private BigDecimal findUsableAmount(CostPoolRegister costPoolRegister, String customerCode, CostPoolStrategy costPoolStrategy) {
    // - 查询客户的资金信息
    CostPoolDto costPoolDto = new CostPoolDto();
    costPoolDto.setPoolType(costPoolRegister.getKey());
    costPoolDto.setCustomerCode(customerCode);
    List<CostPoolVo> costPoolVos = costPoolStrategy.onRequestCostPoolVos(costPoolDto);
    if (CollectionUtils.isEmpty(costPoolVos)) {
      return BigDecimal.ZERO;
    }
    //汇总剩余可使用金额
    BigDecimal usableAmount = costPoolVos.stream()
        .map(CostPoolVo::getUsableAmount)
        .reduce(BigDecimal.ZERO, BigDecimal::add);
    return usableAmount;
  }

  /**
   * 设置订单编码
   *
   * @param order 订单
   * @return {@link Order}
   */
  private Order setOrderCode(OrderConfirmDto order) {
    // 确保会有订单编号
    this.generateOrderCode(order);
    // 驳回和撤回的订单，再提交，重新搞个订单
    Order orderDb = orderService.findByOrderCode(order.getOrderCode());
    if (Objects.nonNull(orderDb)) {
      String orderStatus = orderDb.getOrderStatus();
      boolean isRejected = OrderStatusEnum.REJECTED.getDictCode().equals(orderStatus);
      boolean isCanceled = OrderStatusEnum.CANCELED.getDictCode().equals(orderStatus);
      if (isRejected || isCanceled) {
        order.setOrderCode(null);
      }
      this.generateOrderCode(order);
    }
    return orderDb;
  }

  /**
   * 补偿付款准确性
   *
   * @param orderConfirmVo 订单确认签证官
   */
  private void compensationPaymentAccuracy(OrderConfirmVo orderConfirmVo) {
    // - 不处理物料订单
    String orderCategory = orderConfirmVo.getOrderCategory();
    if (orderCategory.equals(OrderCategoryEnum.MATERIAL_ORDER.getDictCode())) {
      return;
    }
    String orderType = orderConfirmVo.getOrderType();
    boolean isFree = OrderTypeEnum.FREE.getDictCode().equals(orderType);
    if (isFree) {
      return;
    }
    // -
    List<OrderPayPreviewVo> orderPays = orderConfirmVo.getOrderPays();
    //订单总金额
    BigDecimal totalOrderAmount = orderConfirmVo.getTotalOrderAmount();
    //政策优惠金额
    BigDecimal policyDiscountAmount = orderConfirmVo.getPolicyDiscountAmount();
    //货补金额
    BigDecimal totalCompensatedAmount = orderConfirmVo.getTotalCompensatedAmount();
    //折扣金额--其实是所有该扣项目的汇总
    BigDecimal totalDiscountAmount = BigDecimal.ZERO;
    for (OrderPayPreviewVo orderPay : orderPays) {
      Boolean itemGroupType = orderPay.getItemGroupType();
      if (!itemGroupType) {
        BigDecimal itemAmount = orderPay.getItemAmount();
        itemAmount = Objects.nonNull(itemAmount) ? itemAmount : BigDecimal.ZERO;
        totalDiscountAmount = totalDiscountAmount.add(itemAmount);
      }
    }
    //授信金额 + 资金金额--其实是所有支付项目的汇总
    BigDecimal totalPaymentAmount = BigDecimal.ZERO;
    for (OrderPayPreviewVo orderPay : orderPays) {
      Boolean itemGroupType = orderPay.getItemGroupType();
      if (itemGroupType) {
        BigDecimal itemAmount = orderPay.getItemAmount();
        itemAmount = Objects.nonNull(itemAmount) ? itemAmount : BigDecimal.ZERO;
        totalPaymentAmount = totalPaymentAmount.add(itemAmount);
      }
    }
    // 校验订单金额-政策优惠金额-货补-折扣=X（应付金额）
    BigDecimal amountPayable = totalOrderAmount.subtract(policyDiscountAmount).subtract(totalCompensatedAmount).subtract(totalDiscountAmount);
    int compare = amountPayable.compareTo(totalPaymentAmount);
    // 尾差1毛以内的才管
    BigDecimal threshold = new BigDecimal("0.1");
    if (compare == 0) {
      return;
    } else if (compare > 0) {
      // 在某一个支付项目上加上补偿值
      BigDecimal compensationValue = amountPayable.subtract(totalPaymentAmount);
      boolean canCompensate = threshold.compareTo(compensationValue) > -1;
      if (!canCompensate) {
        return;
      }
      for (OrderPayPreviewVo orderPay : orderPays) {
        Boolean itemGroupType = orderPay.getItemGroupType();
        if (!itemGroupType) {
          continue;
        }
        BigDecimal itemAmount = orderPay.getItemAmount();
        itemAmount = Objects.nonNull(itemAmount) ? itemAmount : BigDecimal.ZERO;
        orderPay.setItemAmount(itemAmount.add(compensationValue));
        break;
      }
    } else {
      // 在某一个支付项目上减去补偿值
      BigDecimal compensationValue = totalPaymentAmount.subtract(amountPayable);
      boolean canCompensate = threshold.compareTo(compensationValue) > -1;
      if (!canCompensate) {
        return;
      }
      for (OrderPayPreviewVo orderPay : orderPays) {
        Boolean itemGroupType = orderPay.getItemGroupType();
        if (!itemGroupType) {
          continue;
        }
        BigDecimal itemAmount = orderPay.getItemAmount();
        itemAmount = Objects.nonNull(itemAmount) ? itemAmount : BigDecimal.ZERO;
        boolean ge = itemAmount.compareTo(compensationValue) > -1;
        if (ge) {
          orderPay.setItemAmount(itemAmount.subtract(compensationValue));
          break;
        }
      }
    }
  }

  @Autowired(required = false)
  private LoginUserService loginUserService;

  @Autowired(required = false)
  private DictDataVoService dictDataVoService;

  @Override
  @Transactional
  public OrderConfirmVo handleConfirmByCurrentCustomer(OrderConfirmDto order) {
    LoginUserDetailsForEMS loginDetails = loginUserService.getLoginDetails(LoginUserDetailsForEMS.class);
    order.setRelateCode(loginDetails.getCustomerCode());
    order.setRelateName(loginDetails.getCustomerName());
    // - 经销商下单的时候，由于页面没有选择流程的地方，这里直接指定默认流程。当然，后续可以商量是用字典还是配置文件。fixme
    List<DictDataVo> dmsWorkFlow = dictDataVoService.findByDictTypeCode("dms_work_flow");
    if (!CollectionUtils.isEmpty(dmsWorkFlow)) {
      for (DictDataVo dictDataVo : dmsWorkFlow) {
        String dictCode = dictDataVo.getDictCode();
        if ("customerOrderKey".equals(dictCode)) {
          order.setProcessKey(dictDataVo.getDictValue());
        } else if ("customerOrderForm".equals(dictCode)) {
          order.setCompetenceCode(dictDataVo.getDictValue());
        }
      }
    } else {
      order.setProcessKey("orderapprove");
      order.setCompetenceCode("CRM202101050000000425");
    }
    return this.handleConfirmWithPaymentMethod(order);
  }

  @Override
  public OrderConfirmVo handlePartSubmitByCurrentCustomer(OrderConfirmDto order) {
    LoginUserDetailsForEMS loginDetails = loginUserService.getLoginDetails(LoginUserDetailsForEMS.class);
    order.setRelateCode(loginDetails.getCustomerCode());
    order.setRelateName(loginDetails.getCustomerName());
    List<OrderValidationProcessNodeEnum> exclude = Lists.newArrayList(OrderValidationProcessNodeEnum.TEN);
    return this.handleSubmit(order, exclude);
  }

  /**
   * 购买历史记录
   *
   * @param orderInsert 订单
   */
  private void recordPurchaseHistory(Order orderInsert) {
    List<OrderDetail> details = orderInsert.getOrderDetails();
    List<PurchaseHistory> purchaseHistories = Lists.newArrayListWithCapacity(details.size());
    for (OrderDetail detail : details) {
      PurchaseHistory purchaseHistory = new PurchaseHistory();
      purchaseHistory.setGoodsCode(detail.getGoodsCode());
      purchaseHistory.setGoodsName(detail.getGoodsName());
      purchaseHistory.setSpec(detail.getSpec());
      purchaseHistory.setUnite(detail.getUnite());
      purchaseHistory.setRelateCode(orderInsert.getRelateCode());
      purchaseHistory.setTenantCode(TenantUtils.getTenantCode());
      purchaseHistory.setAppCode(TenantUtils.getAppCode());
      purchaseHistories.add(purchaseHistory);
    }
    this.purchaseHistoryService.handleIncrement(purchaseHistories);
  }

  /**
   * 确保会有订单编号
   *
   * @param order
   */
  private void generateOrderCode(OrderConfirmDto order) {
    // 填充编码 PS:已经存在编码就没必要再生成了。
    if (StringUtils.isBlank(order.getOrderCode())) {
      String ruleCode = StringUtils.join(OrderConstant.ORDER_RULE_CODE, DateFormatUtils.format(new Date(), "yyyyMMdd"));
      String orderCode = this.generateCodeService.generateCode(ruleCode, 1, 10, 2, TimeUnit.DAYS).get(0);
      order.setOrderCode(orderCode);
    }
  }

  /**
   * 分摊service排序后，获取分摊chain，分摊service内部根据订单提交阶段 自行决定是否执行
   *
   * @param context
   * @return
   */
  private OrderApportionVo apportion(OrderApportionContext context) {
    Validate.isTrue(!CollectionUtils.isEmpty(orderApportionVoServiceList), "还没有分摊实现，请确认项目是否正确构建");
    Collections.sort(orderApportionVoServiceList);
    OrderApportionVoServiceChain chain = this.orderApportionVoServiceChainInstance.getChain(orderApportionVoServiceList);
    chain.execute(context);
    return context.getOrderApportionVo();
  }

}
