package com.biz.crm.dms.business.order.local.service.internal;


import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.dms.business.order.local.entity.OrderDetail;
import com.biz.crm.dms.business.order.local.entity.OrderDetailPay;
import com.biz.crm.dms.business.order.local.repository.OrderDetailPayRepository;
import com.biz.crm.dms.business.order.local.service.OrderDetailPayService;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.math.BigDecimal;
import java.util.List;
import java.util.Objects;

/**
 * 订单明细支付明细(OrderDetailPay)表服务实现类
 *
 * @author makejava
 * @since 2022-01-08 15:13:16
 */
@Service("orderDetailPayService")
public class OrderDetailPayServiceImpl implements OrderDetailPayService {

  @Autowired(required = false)
  private OrderDetailPayRepository orderDetailPayRepository;

  /**
   * 分页查询数据
   *
   * @param pageable       分页对象
   * @param orderDetailPay 实体对象
   * @return
   */
  @Override
  public Page<OrderDetailPay> findByConditions(Pageable pageable, OrderDetailPay orderDetailPay) {
    ObjectUtils.defaultIfNull(pageable, PageRequest.of(0, 50));
    if (Objects.isNull(orderDetailPay)) {
      orderDetailPay = new OrderDetailPay();
    }
    return this.orderDetailPayRepository.findByConditions(pageable, orderDetailPay);
  }

  /**
   * 通过主键查询单条数据
   *
   * @param id 主键
   * @return 单条数据
   */
  @Override
  public OrderDetailPay findById(String id) {
    if (StringUtils.isBlank(id)) {
      return null;
    }
    return this.orderDetailPayRepository.getById(id);
  }

  /**
   * 新增数据
   *
   * @param orderDetailPay 实体对象
   * @return 新增结果
   */
  @Transactional
  @Override
  public OrderDetailPay create(OrderDetailPay orderDetailPay) {
    this.createValidate(orderDetailPay);
    this.orderDetailPayRepository.saveOrUpdate(orderDetailPay);
    return orderDetailPay;
  }

  /**
   * 修改新据
   *
   * @param orderDetailPay 实体对象
   * @return 修改结果
   */
  @Transactional
  @Override
  public OrderDetailPay update(OrderDetailPay orderDetailPay) {
    this.updateValidate(orderDetailPay);
    this.orderDetailPayRepository.saveOrUpdate(orderDetailPay);
    return orderDetailPay;
  }

  /**
   * 删除数据
   *
   * @param idList 主键结合
   * @return 删除结果
   */
  @Transactional
  @Override
  public void delete(List<String> idList) {
    Validate.isTrue(!CollectionUtils.isEmpty(idList), "删除数据时，主键集合不能为空！");
    this.orderDetailPayRepository.removeByIds(idList);
  }
  
  @Transactional
  @Override
  public void createBatch(List<OrderDetail> orderDetails) {
    Validate.isTrue(!CollectionUtils.isEmpty(orderDetails), "新增时，没有信息传入！");
    for (OrderDetail orderDetail : orderDetails) {
      this.saveOrUpdateBatch(orderDetail);
    }
  }

  @Transactional
  @Override
  public void deleteByOrderCode(String orderCode) {
    Validate.notBlank(orderCode, "删除时，要传入订单编码");
    this.orderDetailPayRepository.deleteByOrderCode(orderCode);
  }

  /**
   * 持久化 行上该扣项目 （批量）
   * @param orderDetail
   */
  private void saveOrUpdateBatch(OrderDetail orderDetail) {
    List<OrderDetailPay> orderDetailPays = orderDetail.getOrderDetailPays();
    if (CollectionUtils.isEmpty(orderDetailPays)) {
      return;
    }
    for (OrderDetailPay orderDetailPay : orderDetailPays) {
      orderDetailPay.setTenantCode(orderDetail.getTenantCode());
      orderDetailPay.setAppCode(orderDetail.getAppCode());
      orderDetailPay.setOrderCode(orderDetail.getOrderCode());
      orderDetailPay.setOrderDetailCode(orderDetail.getOrderDetailCode());
      this.createValidate(orderDetailPay);
    }
    this.orderDetailPayRepository.saveOrUpdateBatch(orderDetailPays);
  }

  /**
   * 创建验证
   *
   * @param orderDetailPay
   */
  private void createValidate(OrderDetailPay orderDetailPay) {
    Validate.notNull(orderDetailPay, "新增时，对象信息不能为空！");
    Validate.notBlank(orderDetailPay.getTenantCode(), "新增数据时，租户编号不能为空！");
    Validate.notBlank(orderDetailPay.getAppCode(), "新增数据时，品牌商租户编号不能为空！");
    Validate.notBlank(orderDetailPay.getOrderDetailCode(), "明细编码,不能为空！");
    Validate.notBlank(orderDetailPay.getOrderCode(), "订单编码,不能为空！");
    String itemGroupKey = orderDetailPay.getItemGroupKey();
    String itemKey = orderDetailPay.getItemKey();
    String itemName = orderDetailPay.getItemName();
    BigDecimal itemAmount = orderDetailPay.getItemAmount();
    String originData = orderDetailPay.getOriginData();
    Validate.notBlank(itemGroupKey, "该扣项目组的key,不能为空！");
    Validate.notBlank(itemKey, "该扣项目key,不能为空！");
    Validate.notBlank(itemName, "该扣项目名称,不能为空！");
    Validate.notNull(itemAmount, "该扣项目金额（组内分摊）,不能为空！");
    Validate.isTrue(itemGroupKey.length() < 32, "该扣项目组的key,限定长度(32)，请检查！");
    Validate.isTrue(itemKey.length() < 32, "该扣项目key,限定长度(32)，请检查！");
    Validate.isTrue(itemName.length() < 32, "该扣项目名称,限定长度(32)，请检查！");
    Validate.isTrue(StringUtils.isEmpty(originData) || originData.length() < 255, "该扣项目金额数据的由来,限定长度(255)，请检查！");
  }

  /**
   * 修改验证
   *
   * @param orderDetailPay
   */
  private void updateValidate(OrderDetailPay orderDetailPay) {
    Validate.notNull(orderDetailPay, "修改时，对象信息不能为空！");
    Validate.notBlank(orderDetailPay.getId(), "修改时，不能为空！");
    Validate.notBlank(orderDetailPay.getTenantCode(), "修改时，租户编号不能为空！");
    Validate.notBlank(orderDetailPay.getAppCode(), "修改时，品牌商租户编号不能为空！");

  }
}

