package com.biz.crm.dms.business.order.local.service.internal;


import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.dms.business.order.local.entity.Order;
import com.biz.crm.dms.business.order.local.entity.OrderFile;
import com.biz.crm.dms.business.order.local.repository.OrderFileRepository;
import com.biz.crm.dms.business.order.local.service.OrderFileService;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.util.Iterator;
import java.util.List;
import java.util.Objects;

/**
 * 订单附件信息(OrderFile)表服务实现类
 *
 * @author makejava
 * @since 2022-01-08 15:13:20
 */
@Service("orderFileService")
public class OrderFileServiceImpl implements OrderFileService {

  @Autowired(required = false)
  private OrderFileRepository orderFileRepository;

  /**
   * 分页查询数据
   *
   * @param pageable  分页对象
   * @param orderFile 实体对象
   * @return
   */
  @Override
  public Page<OrderFile> findByConditions(Pageable pageable, OrderFile orderFile) {
    ObjectUtils.defaultIfNull(pageable, PageRequest.of(0, 50));
    if (Objects.isNull(orderFile)) {
      orderFile = new OrderFile();
    }
    return this.orderFileRepository.findByConditions(pageable, orderFile);
  }

  /**
   * 通过主键查询单条数据
   *
   * @param id 主键
   * @return 单条数据
   */
  @Override
  public OrderFile findById(String id) {
    if (StringUtils.isBlank(id)) {
      return null;
    }
    return this.orderFileRepository.getById(id);
  }

  /**
   * 新增数据
   *
   * @param orderFile 实体对象
   * @return 新增结果
   */
  @Transactional
  @Override
  public OrderFile create(OrderFile orderFile) {
    this.createValidate(orderFile);
    this.orderFileRepository.saveOrUpdate(orderFile);
    return orderFile;
  }

  /**
   * 修改新据
   *
   * @param orderFile 实体对象
   * @return 修改结果
   */
  @Transactional
  @Override
  public OrderFile update(OrderFile orderFile) {
    this.updateValidate(orderFile);
    this.orderFileRepository.saveOrUpdate(orderFile);
    return orderFile;
  }

  /**
   * 删除数据
   *
   * @param idList 主键结合
   * @return 删除结果
   */
  @Transactional
  @Override
  public void delete(List<String> idList) {
    Validate.isTrue(!CollectionUtils.isEmpty(idList), "删除数据时，主键集合不能为空！");
    this.orderFileRepository.removeByIds(idList);
  }

  @Transactional
  @Override
  public void createBatch(Order order) {
    /**
     * 即使是草稿，没有文件的关键信息也是不存的
     */
    Validate.notNull(order, "批量持久化订单附件时，订单主信息不能为空");
    List<OrderFile> orderFiles = order.getOrderFiles();
    if (CollectionUtils.isEmpty(orderFiles)) {
      return;
    }
    Iterator<OrderFile> iterator = orderFiles.iterator();
    while (iterator.hasNext()) {
      OrderFile next = iterator.next();
      if (Objects.isNull(next)) {
        iterator.remove();
        continue;
      }
      next.setTenantCode(order.getTenantCode());
      next.setAppCode(order.getAppCode());
      next.setOrderCode(order.getOrderCode());
      this.createValidate(next);
    }
    this.orderFileRepository.saveOrUpdateBatch(orderFiles);
  }

  @Transactional
  @Override
  public void deleteByOrderCode(String orderCode) {
    Validate.notBlank(orderCode, "删除时，要传入订单编码");
    this.orderFileRepository.deleteByOrderCode(orderCode);
  }

  /**
   * 创建验证
   *
   * @param orderFile
   */
  private void createValidate(OrderFile orderFile) {
    Validate.notNull(orderFile, "新增时，对象信息不能为空！");
    Validate.notBlank(orderFile.getTenantCode(), "新增数据时，租户编号不能为空！");
    Validate.notBlank(orderFile.getAppCode(), "新增数据时，品牌商租户编号不能为空！");
    Validate.notBlank(orderFile.getOrderCode(), "新增数据时，订单编码不能为空！");
    Validate.notBlank(orderFile.getFileName(), "新增数据时， 文件名 不能为空！");
    Validate.notBlank(orderFile.getRelativePath(), "新增数据时， 相对路径 不能为空！");
  }

  /**
   * 修改验证
   *
   * @param orderFile
   */
  private void updateValidate(OrderFile orderFile) {
    Validate.notNull(orderFile, "修改时，对象信息不能为空！");
    Validate.notBlank(orderFile.getId(), "新增数据时，不能为空！");
    Validate.notBlank(orderFile.getTenantCode(), "新增数据时，租户编号不能为空！");
    Validate.notBlank(orderFile.getAppCode(), "新增数据时，品牌商租户编号不能为空！");
    Validate.notBlank(orderFile.getFileName(), "新增数据时， 文件名 不能为空！");
    Validate.notBlank(orderFile.getRelativePath(), "新增数据时， 相对路径 不能为空！");

  }
}

