package com.biz.crm.dms.business.order.local.service.internal;


import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.dms.business.order.common.sdk.model.TallyItemRegisterModel;
import com.biz.crm.dms.business.order.common.sdk.service.TallyItemRegisterService;
import com.biz.crm.dms.business.order.local.entity.Order;
import com.biz.crm.dms.business.order.local.entity.OrderPay;
import com.biz.crm.dms.business.order.local.repository.OrderPayRepository;
import com.biz.crm.dms.business.order.local.service.OrderPayService;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.math.BigDecimal;
import java.util.List;
import java.util.Objects;

/**
 * 订单支付明细(OrderPay)表服务实现类
 *
 * @author makejava
 * @since 2022-01-08 15:13:26
 */
@Service("orderPayService")
public class OrderPayServiceImpl implements OrderPayService {

  @Autowired(required = false)
  private OrderPayRepository orderPayRepository;

  @Autowired(required = false)
  private TallyItemRegisterService tallyItemRegisterService;

  /**
   * 分页查询数据
   *
   * @param pageable 分页对象
   * @param orderPay 实体对象
   * @return
   */
  @Override
  public Page<OrderPay> findByConditions(Pageable pageable, OrderPay orderPay) {
    ObjectUtils.defaultIfNull(pageable, PageRequest.of(0, 50));
    if (Objects.isNull(orderPay)) {
      orderPay = new OrderPay();
    }
    return this.orderPayRepository.findByConditions(pageable, orderPay);
  }

  /**
   * 通过主键查询单条数据
   *
   * @param id 主键
   * @return 单条数据
   */
  @Override
  public OrderPay findById(String id) {
    if (StringUtils.isBlank(id)) {
      return null;
    }
    return this.orderPayRepository.getById(id);
  }

  /**
   * 新增数据
   *
   * @param orderPay 实体对象
   * @return 新增结果
   */
  @Transactional
  @Override
  public OrderPay create(OrderPay orderPay) {
    this.createValidate(orderPay);
    this.orderPayRepository.saveOrUpdate(orderPay);
    return orderPay;
  }

  /**
   * 修改新据
   *
   * @param orderPay 实体对象
   * @return 修改结果
   */
  @Transactional
  @Override
  public OrderPay update(OrderPay orderPay) {
    this.updateValidate(orderPay);
    this.orderPayRepository.saveOrUpdate(orderPay);
    return orderPay;
  }

  /**
   * 删除数据
   *
   * @param idList 主键结合
   * @return 删除结果
   */
  @Transactional
  @Override
  public void delete(List<String> idList) {
    Validate.isTrue(!CollectionUtils.isEmpty(idList), "删除数据时，主键集合不能为空！");
    this.orderPayRepository.removeByIds(idList);
  }
  
  @Transactional
  @Override
  public void createBatch(Order order) {
    /**
     *
     */
    Validate.notNull(order, "批量持久化订单【记账项目】时，订单主信息不能为空");
    Boolean draft = order.getDraft();
    List<OrderPay> orderPays = order.getOrderPays();
    if (CollectionUtils.isEmpty(orderPays)) {
      return;
    }
    for (OrderPay orderPay : orderPays) {
      orderPay.setOrderCode(order.getOrderCode());
      orderPay.setTenantCode(order.getTenantCode());
      orderPay.setAppCode(order.getAppCode());
      this.createValidate(orderPay);
      if (!draft) {
        this.requiredValidate(orderPay);
      }
    }
    this.orderPayRepository.saveOrUpdateBatch(orderPays);
  }

  @Transactional
  @Override
  public void deleteByOrderCode(String orderCode) {
    Validate.notBlank(orderCode, "删除时，要传入订单编码");
    this.orderPayRepository.deleteByOrderCode(orderCode);
  }

  /**
   * 必填校验
   * @param orderPay
   */
  private void requiredValidate(OrderPay orderPay) {
    Validate.notBlank(orderPay.getItemGroupKey(), "项目组的key，必须传入！");
    Validate.notBlank(orderPay.getItemKey(), "项目key，必须传入！");
    Validate.notBlank(orderPay.getItemName(), "项目名称，必须传入！");
    Validate.notNull(orderPay.getItemAmount(), "项目金额，必须传入！");
  }

  /**
   * 创建验证
   * --单条信息的数据完整性的基本验证
   * --不含业务验证
   *
   * @param orderPay
   */
  private void createValidate(OrderPay orderPay) {
    Validate.notNull(orderPay, "新增时，对象信息不能为空！");
    Validate.notBlank(orderPay.getTenantCode(), "新增数据时，租户编号不能为空！");
    Validate.notBlank(orderPay.getAppCode(), "新增数据时，品牌商租户编号不能为空！");
    Validate.notBlank(orderPay.getOrderCode(), "新增数据时，订单编号不能为空！");
    /**
     * 项目组类型在系统中是否存在
     * --boolean就不用校验系统中是否存在了，不为null就行
     * 项目组的key在系统中是否存在
     * 项目key在系统中是否存在
     * 项目金额不能为负数
     * 长度校验
     */
    Boolean itemGroupType = orderPay.getItemGroupType();
    String itemGroupKey = orderPay.getItemGroupKey();
    String itemKey = orderPay.getItemKey();
    String itemName = orderPay.getItemName();
    BigDecimal itemAmount = orderPay.getItemAmount();
    // 项目组的key在系统中是否存在
    Validate.notNull(itemGroupType, "项目组类型，必须传入！");
    Validate.notNull(itemGroupKey, "项目组的key，必须传入！");
    Validate.notNull(itemKey, "项目key，必须传入！");
    Validate.notNull(itemName, "项目名称，必须传入！");
    Validate.notNull(itemAmount, "项目金额，必须传入！");
    // 项目组的key在系统中是否存在
    List<TallyItemRegisterModel> models = this.tallyItemRegisterService.findByItemGroupKey(itemGroupKey);
    Validate.isTrue(!CollectionUtils.isEmpty(models), "项目组的key，在系统中不存在，请检查！");
    // 项目key在系统中是否存在
    TallyItemRegisterModel model = this.tallyItemRegisterService.findByItemKey(itemKey);
    Validate.notNull(model, "项目key，在系统中不存在，请检查！");
    // 项目组的key 和 项目key 是否是正确的匹配关系
    TallyItemRegisterModel model1 = this.tallyItemRegisterService.findByItemGroupKeyAndItemKey(itemGroupKey, itemKey);
    // --其实 model 与 model1 不是同一个对象的话，那么项目key和项目组的key就不匹配
    Validate.notNull(model1, "项目key与项目组的key，在系统中不匹配，请检查！");
    // 项目金额不能为负数
    boolean gtZero = itemAmount.compareTo(BigDecimal.ZERO) > -1;
    Validate.isTrue(gtZero, "项目金额不能为负数，请检查！");
    // 长度校验
    Validate.isTrue(itemGroupKey.length() < 32, "项目组的key,限定长度(32)，请检查！");
    Validate.isTrue(itemKey.length() < 32, "项目key,限定长度(32)，请检查！");
    Validate.isTrue(itemName.length() < 32, "项目名称,限定长度(32)，请检查！");
  }

  /**
   * 修改验证
   *
   * @param orderPay
   */
  private void updateValidate(OrderPay orderPay) {
    Validate.notNull(orderPay, "修改时，对象信息不能为空！");
    Validate.notBlank(orderPay.getId(), "新增数据时，不能为空！");
    Validate.notBlank(orderPay.getTenantCode(), "新增数据时，租户编号不能为空！");
    Validate.notBlank(orderPay.getAppCode(), "新增数据时，品牌商租户编号不能为空！");

  }
}

