package com.biz.crm.dms.business.order.local.service.internal;


import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.sdk.model.LoginUserDetailsForEMS;
import com.biz.crm.business.common.sdk.service.GenerateCodeService;
import com.biz.crm.business.common.sdk.service.LoginUserService;
import com.biz.crm.dms.business.order.common.sdk.enums.OrderStatusEnum;
import com.biz.crm.dms.business.order.local.entity.Order;
import com.biz.crm.dms.business.order.local.entity.OrderDetail;
import com.biz.crm.dms.business.order.local.repository.OrderRepository;
import com.biz.crm.dms.business.order.local.service.OrderDetailService;
import com.biz.crm.dms.business.order.local.service.OrderFileService;
import com.biz.crm.dms.business.order.local.service.OrderPayService;
import com.biz.crm.dms.business.order.local.service.OrderService;
import com.biz.crm.dms.business.order.sdk.constant.OrderConstant;
import com.biz.crm.dms.business.order.sdk.dto.CustomerOrderPageDto;
import com.biz.crm.dms.business.order.sdk.dto.OrderEventDto;
import com.biz.crm.dms.business.order.sdk.dto.OrderPaginationDto;
import com.biz.crm.dms.business.order.sdk.event.OrderLogEventListener;
import com.biz.crm.dms.business.order.sdk.vo.OrderVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.bizunited.nebula.event.sdk.function.SerializableBiConsumer;
import com.bizunited.nebula.event.sdk.service.NebulaNetEventClient;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.apache.commons.lang3.time.DateFormatUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.math.BigDecimal;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

/**
 * 订单主信息(Order)表服务实现类
 *
 * @author makejava
 * @since 2022-01-08 15:13:08
 */
@Service("orderService")
public class OrderServiceImpl implements OrderService {

  @Autowired(required = false)
  private OrderRepository orderRepository;

  @Autowired(required = false)
  private GenerateCodeService generateCodeService;

  @Autowired(required = false)
  private NebulaNetEventClient nebulaNetEventClient;

  @Autowired(required = false)
  private NebulaToolkitService nebulaToolkitService;

  /**
   * 分页查询数据
   *
   * @param pageable 分页对象
   * @param order    实体对象
   * @return
   */
  @Override
  public Page<Order> findByConditions(Pageable pageable, OrderPaginationDto order) {
    ObjectUtils.defaultIfNull(pageable, PageRequest.of(0, 50));
    if (Objects.isNull(order)) {
      order = new OrderPaginationDto();
    }
    order.setTenantCode(TenantUtils.getTenantCode());
    order.setAppCode(TenantUtils.getAppCode());
    order.setIsShow(true);
    return this.orderRepository.findByConditions(pageable, order);
  }

  /**
   * 通过主键查询单条数据
   *
   * @param id 主键
   * @return 单条数据
   */
  @Override
  public Order findById(String id) {
    if (StringUtils.isBlank(id)) {
      return null;
    }
    return this.orderRepository.findDetailById(id);
  }

  @Autowired(required = false)
  private OrderPayService orderPayService;

  @Autowired(required = false)
  private OrderFileService orderFileService;

  @Autowired(required = false)
  private OrderDetailService orderDetailService;

  /**
   * 新增数据
   * --这里只做数据完整校验不做业务校验。
   * --业务校验，在【订单提交】处执行
   * --由于草稿设定的存在，导致业务数据在表中都没有了非空约束
   *
   * @param order 实体对象
   * @return 新增结果
   */
  @Transactional
  @Override
  public Order create(Order order) {
    /**
     * 填充数据
     * - 默认显示
     * 校验
     * 保存数据
     */
    this.paddingData(order);
    Boolean draft = order.getDraft();
    Validate.notNull(draft, "暂存标识，不能为空！");
    if (draft) {
      // 是草稿，说明是暂存信息。
      order.setOrderStatus(OrderStatusEnum.STAGING.getDictCode());
      this.draftValidate(order);
      Order orderDb = this.findByOrderCode(order.getOrderCode());
      if (orderDb != null) {
        order.setId(orderDb.getId());
      }
      //--删除之前的草稿信息,
      this.deleteDraftByOrderCode(order.getOrderCode());
      //--草稿统计订单
      this.tabulateData(order);
    } else {
      // 不是草稿，就要做最严厉的数据完整性校验。
      this.submitValidate(order);
    }
    // 因为暂存的设定，id可能有值
    this.orderRepository.saveOrUpdate(order);
    // 从表信息的保存
    this.saveDetails(order);

    OrderEventDto orderEventDto = new OrderEventDto();
    orderEventDto.setOriginal(null);
    OrderVo orderVo = this.nebulaToolkitService.copyObjectByBlankList(order, OrderVo.class, HashSet.class, LinkedList.class);
    orderEventDto.setNewest(orderVo);
    // 订单创建事件
    SerializableBiConsumer<OrderLogEventListener, OrderEventDto> onCreate = OrderLogEventListener::onCreate;
    this.nebulaNetEventClient.publish(orderEventDto, OrderLogEventListener.class, onCreate);
    return order;
  }

  /**
   * 汇总数据
   *
   * @param order 订单
   */
  private void tabulateData(Order order) {
    //--商品总数
    BigDecimal totalCountGoods = BigDecimal.ZERO;
    // --订单总金额 = 所有商品行销售价*数量汇总
    BigDecimal totalOrderAmount = BigDecimal.ZERO;

    List<OrderDetail> orderDetails = order.getOrderDetails();
    if (CollectionUtils.isEmpty(orderDetails)) {
      return;
    }
    for (OrderDetail orderDetail : orderDetails) {
      // --行上的销售总金额
      BigDecimal salesAmount = orderDetail.getSalesAmount();
      totalOrderAmount = totalOrderAmount.add(salesAmount);
      BigDecimal quantity = orderDetail.getQuantity();
      if (ObjectUtils.isEmpty(quantity)) {
        continue;
      }
      totalCountGoods = totalCountGoods.add(quantity);
    }
    order.setQuantityOfCommodity(totalCountGoods);
    order.setTotalOrderAmount(totalOrderAmount);

  }

  /**
   * 从表信息的保存
   *
   * @param order
   */
  private void saveDetails(Order order) {
    this.orderFileService.createBatch(order);
    this.orderPayService.createBatch(order);
    this.orderDetailService.createBatch(order);
  }

  /**
   * 为订单提交业务提供的数据完整性校验方法
   *
   * @param order
   */
  private void submitValidate(Order order) {
    /**
     * 暂未发现需要校验什么
     */
  }

  /**
   * 为暂存业务提供的数据完整性校验方法
   *
   * @param order
   */
  private void draftValidate(Order order) {
    /**
     * 租户
     * 订单编码
     */
    Validate.notNull(order, "订单信息不能为空");
    String tenantCode = order.getTenantCode();
    Validate.notBlank(tenantCode, "租户信息不能为空");
    String orderCode = order.getOrderCode();
    Validate.notBlank(orderCode, "订单编码不能为空");
  }

  /**
   * 填充数据
   *
   * @param order
   */
  private void paddingData(Order order) {
    String appCode = TenantUtils.getAppCode();
    String tenantCode = TenantUtils.getTenantCode();
    Validate.notBlank(appCode, "当前登录账户没有appCode，请联系系统管理员");
    Validate.notBlank(tenantCode, "当前登录账户没有tenantCode，请联系系统管理员");
    order.setAppCode(appCode);
    order.setTenantCode(tenantCode);
    // 填充编码 PS:已经存在编码就没必要再生成了。
    if (StringUtils.isBlank(order.getOrderCode())) {
      String ruleCode = StringUtils.join(OrderConstant.ORDER_RULE_CODE, DateFormatUtils.format(new Date(), "yyyyMMdd"));
      String orderCode = this.generateCodeService.generateCode(ruleCode, 1, 5, 2, TimeUnit.DAYS).get(0);
      order.setOrderCode(orderCode);
    }
    String externalSources = order.getExternalSources();
    Boolean isShow = order.getIsShow();
    if (isShow == null) {
      // 默认显示
      order.setIsShow(true);
    }
    if (StringUtils.isBlank(externalSources)) {
      // 订单管理本身的业务，默认显示
      order.setIsShow(true);
    }

  }

  /**
   * 修改新据
   *
   * @param order 实体对象
   * @return 修改结果
   */
  @Transactional
  @Override
  public Order update(Order order) {
    order.setTenantCode(TenantUtils.getTenantCode());
    this.updateValidate(order);
    this.orderRepository.saveOrUpdate(order);
    return order;
  }

  /**
   * 删除数据
   *
   * @param idList 主键结合
   * @return 删除结果
   */
  @Transactional
  @Override
  public void delete(List<String> idList) {
    Validate.isTrue(!CollectionUtils.isEmpty(idList), "删除数据时，主键集合不能为空！");
    List<Order> orders = this.orderRepository.listByIds(idList);
    Set<String> set = Sets.newHashSet(
        OrderStatusEnum.STAGING.getDictCode(),
        OrderStatusEnum.REJECTED.getDictCode(),
        OrderStatusEnum.CANCELED.getDictCode()
    );
    for (Order order : orders) {
      boolean contains = set.contains(order.getOrderStatus());
      Validate.isTrue(contains, "含有不能删除的订单状态");
      this.deleteDraftByOrderCode(order.getOrderCode());
      this.orderRepository.removeById(order.getId());
    }
  }

  @Override
  public Order findByOrderCode(String orderCode) {
    if (StringUtils.isBlank(orderCode)) {
      return null;
    }
    return this.orderRepository.findByOrderCode(orderCode);
  }

  @Transactional
  @Override
  public void deleteDraftByOrderCode(String orderCode) {
    Validate.notBlank(orderCode, "删除信息时，要传入订单编码");
    // 删除子表信息
    this.orderDetailService.deleteByOrderCode(orderCode);
    this.orderPayService.deleteByOrderCode(orderCode);
    this.orderFileService.deleteByOrderCode(orderCode);
  }

  @Autowired(required = false)
  private LoginUserService loginUserService;

  @Override
  public Page<Order> findByCustomerOrderPageDto(Pageable pageable, CustomerOrderPageDto customerOrderPageDto) {
    ObjectUtils.defaultIfNull(pageable, PageRequest.of(0, 50));
    if (Objects.isNull(customerOrderPageDto)) {
      customerOrderPageDto = new CustomerOrderPageDto();
    }
    LoginUserDetailsForEMS loginDetails = this.loginUserService.getLoginDetails(LoginUserDetailsForEMS.class);
    customerOrderPageDto.setRelateCode(loginDetails.getCustomerCode());
    customerOrderPageDto.setAppCode(TenantUtils.getAppCode());
    Page<Order> page = this.orderRepository.findByCustomerOrderPageDto(pageable, customerOrderPageDto);
    // 因为连表太多，导致分页问题
    List<Order> records = page.getRecords();
    if (!CollectionUtils.isEmpty(records)) {
      List<String> orderCodes = records.stream().map(Order::getOrderCode).collect(Collectors.toList());
      List<Order> orders = this.orderRepository.findDetailByOrderCodes(orderCodes);
      Map<String, Order> orderCodeToOrderMap = new HashMap<>(orders.size());
      for (Order order : orders) {
        orderCodeToOrderMap.put(order.getOrderCode(), order);
      }
      for (Order record : records) {
        String orderCode = record.getOrderCode();
        Order orderDb = orderCodeToOrderMap.get(orderCode);
        record.setOrderDetails(orderDb.getOrderDetails());
        record.setOrderPays(orderDb.getOrderPays());
      }
    }
    return page;
  }

  @Override
  public List<Order> findRefundableByCustomerCode(String relateCode) {
    Validate.isTrue(StringUtils.isNotBlank(relateCode), "客户编码不能为空");
    List<String> orderStatusList = Lists.newArrayList(
        OrderStatusEnum.PART_SHIPPED.getDictCode(),
        OrderStatusEnum.COMPLETED.getDictCode()
    );
    return orderRepository.findByRelateCodeAndOrderStatusIn(relateCode, orderStatusList);
  }

  // TODO 创建时的createValidate方法还没有完成编写

  /**
   * 修改验证
   *
   * @param order
   */
  private void updateValidate(Order order) {
    Validate.notNull(order, "修改时，对象信息不能为空！");
    order.setTenantCode(TenantUtils.getTenantCode());
    order.setAppCode(TenantUtils.getAppCode());
    Validate.notBlank(order.getId(), "修改时，不能为空！");
    Validate.notBlank(order.getTenantCode(), "修改时，租户编号不能为空！");
    Validate.notBlank(order.getAppCode(), "修改时，品牌商租户编号不能为空！");

  }
}

