package com.biz.crm.dms.business.order.local.repository;


import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.biz.crm.dms.business.order.common.sdk.enums.OrderStatusEnum;
import com.biz.crm.dms.business.order.local.entity.Order;
import com.biz.crm.dms.business.order.local.mapper.OrderMapper;
import com.biz.crm.dms.business.order.sdk.dto.CustomerOrderPageDto;
import com.biz.crm.dms.business.order.sdk.dto.OrderPaginationDto;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;

import java.math.BigDecimal;
import java.util.Date;
import java.util.List;


/**
 * 订单主信息(Order)表数据库访问层
 *
 * @author makejava
 * @since 2022-01-08 15:13:09
 */
@Component
public class OrderRepository extends ServiceImpl<OrderMapper, Order> {

  @Autowired(required = false)
  private OrderMapper orderMapper;

  /**
   * 分页查询数据
   *
   * @param pageable           分页对象
   * @param orderPaginationDto 实体对象
   * @return
   */
  public Page<Order> findByConditions(Pageable pageable, OrderPaginationDto orderPaginationDto) {
    Page<Order> page = new Page<>(pageable.getPageNumber(), pageable.getPageSize());
    Page<Order> pageList = orderMapper.findByConditions(page, orderPaginationDto);
    return pageList;
  }

  /**
   * 根据订单编码查询
   *
   * @param orderCode
   * @return
   */
  public Order findByOrderCode(String orderCode) {
    return this.lambdaQuery()
        .eq(Order::getAppCode, TenantUtils.getAppCode())
        .eq(Order::getOrderCode, orderCode)
        .one();
  }

  /**
   * 根据订单编码集合查询订单详情
   *
   * @param orderCodes
   * @return
   */
  public List<Order> findDetailByOrderCodes(List<String> orderCodes) {
    if (CollectionUtils.isEmpty(orderCodes)) {
      return null;
    }
    return this.baseMapper.findDetailByOrderCodes(orderCodes, TenantUtils.getAppCode());
  }

  /**
   * 根据 id 修改 orderStatus
   *
   * @param orderStatus
   * @param id
   */
  public void updateOrderStatusById(String orderStatus, String id) {
    this.lambdaUpdate()
        .set(Order::getOrderStatus, orderStatus)
        .eq(Order::getId, id)
        .update();
  }

  /**
   * 根据 id 修改 orderStatus
   *
   * @param orderStatus
   * @param id
   * @param isShow
   */
  public void updateOrderStatusAndIsShowById(String orderStatus, String id, Boolean isShow) {
    this.lambdaUpdate()
        .set(Order::getOrderStatus, orderStatus)
        .set(Order::getIsShow, isShow)
        .eq(Order::getId, id)
        .update();
  }

  /**
   * 根据 id 修改 orderStatus
   *
   * @param orderStatus
   * @param id
   */
  public void updateOrderStatusAndProcessNumberById(String orderStatus, String processNumber, String id) {
    this.lambdaUpdate()
        .set(Order::getOrderStatus, orderStatus)
        .set(Order::getProcessNumber, processNumber)
        .eq(Order::getId, id)
        .update();
  }

  /**
   * 更新订单状态和发货数量通过订单id
   *
   * @param orderStatus      订单状态
   * @param deliveryQuantity 发货数量
   * @param id               订单id
   */
  public void updateOrderStatusAndDeliveryQuantityById(String orderStatus, BigDecimal deliveryQuantity, String id) {
    this.lambdaUpdate()
        .set(Order::getOrderStatus, orderStatus)
        .set(Order::getDeliveryQuantity, deliveryQuantity)
        .eq(Order::getId, id)
        .update();
  }

  /**
   * 根据ID查询，包括子表信息
   *
   * @param id id
   * @return {@link Order}
   */
  public Order findDetailById(String id) {

    return this.baseMapper.findDetailById(id);
  }

  /**
   * 根据客户编码和订单状态集合查询
   *
   * @param relateCode      关联编码 【客户编码】
   * @param orderStatusList 订单状态列表
   * @return {@link List}<{@link Order}>
   */
  public List<Order> findByRelateCodeAndOrderStatusIn(String relateCode, List<String> orderStatusList) {
    return this.lambdaQuery()
        .eq(Order::getRelateCode, relateCode)
        .eq(Order::getAppCode, TenantUtils.getAppCode())
        .in(Order::getOrderStatus, orderStatusList)
        .list();
  }

  /**
   * 分页查询当前客户的数据
   *
   * @param pageable             分页
   * @param customerOrderPageDto 客户订单分页dto
   * @return {@link Page}<{@link Order}>
   */
  public Page<Order> findByCustomerOrderPageDto(Pageable pageable, CustomerOrderPageDto customerOrderPageDto) {
    Page<Order> page = new Page<>(pageable.getPageNumber(), pageable.getPageSize());
    Page<Order> pageList = orderMapper.findByCustomerOrderPageDto(page, customerOrderPageDto);
    return pageList;
  }

  /**
   * 根据订单状态和 提交时间小于指定时间查询
   *
   * @param orderStatusEnum 订单状态枚举
   * @param date            日期
   * @return {@link List}<{@link Order}>
   */
  public List<Order> findByOrderStatusAndCreateTimeLe(OrderStatusEnum orderStatusEnum, Date date) {
    List<Order> list = this.lambdaQuery()
        .eq(Order::getAppCode, TenantUtils.getAppCode())
        .eq(Order::getOrderStatus, orderStatusEnum.getDictCode())
        .le(Order::getCreateTime, date)
        .list();
    return list;

  }
}

