package com.biz.crm.dms.business.order.local.service.assist;

import com.biz.crm.dms.business.costpool.capital.sdk.dto.CostPoolCapitalDto;
import com.biz.crm.dms.business.costpool.capital.sdk.enums.CapitalAdjustTypeEnum;
import com.biz.crm.dms.business.costpool.capital.sdk.enums.PoolTypeEnum;
import com.biz.crm.dms.business.costpool.credit.sdk.dto.CreditPayDto;
import com.biz.crm.dms.business.costpool.credit.sdk.enums.CashAdjustOperateEnum;
import com.biz.crm.dms.business.costpool.credit.sdk.enums.CashAdjustTypeEnum;
import com.biz.crm.dms.business.costpool.credit.sdk.enums.CreditPoolTypeEnum;
import com.biz.crm.dms.business.costpool.discount.sdk.dto.CostPoolDiscountDto;
import com.biz.crm.dms.business.costpool.discount.sdk.enums.PoolGroupEnum;
import com.biz.crm.dms.business.costpool.discount.sdk.enums.PoolOperationTypeEnum;
import com.biz.crm.dms.business.costpool.discount.sdk.enums.PoolPayTypeEnum;
import com.biz.crm.dms.business.costpool.discount.sdk.enums.PoolUseTypeEnum;
import com.biz.crm.dms.business.costpool.replenishment.sdk.dto.CostPoolReplenishmentDto;
import com.biz.crm.dms.business.order.common.sdk.enums.ItemTypeEnum;
import com.biz.crm.dms.business.order.common.sdk.enums.OrderCategoryEnum;
import com.biz.crm.dms.business.order.common.sdk.model.TallyItemRegisterModel;
import com.biz.crm.dms.business.order.common.sdk.register.TallyItemRegister;
import com.biz.crm.dms.business.order.local.entity.Order;
import com.biz.crm.dms.business.order.local.entity.OrderDetail;
import com.biz.crm.dms.business.order.local.entity.OrderPay;
import com.biz.crm.dms.business.order.sdk.dto.OrderDetailPayPreviewDto;
import com.biz.crm.dms.business.order.sdk.dto.OrderDetailPreviewDto;
import com.biz.crm.dms.business.order.sdk.dto.OrderPreviewDto;
import com.biz.crm.dms.business.order.sdk.vo.OrderDetailPreviewVo;
import com.biz.crm.dms.business.order.sdk.vo.OrderFileVo;
import com.biz.crm.dms.business.psi.product.sdk.dto.productstock.ProductStockOperationDto;
import com.biz.crm.dms.business.psi.product.sdk.enums.productstock.ProductStockOperationType;
import com.biz.crm.mdm.business.material.sdk.service.MaterialVoService;
import com.biz.crm.mdm.business.material.sdk.vo.MaterialVo;
import com.biz.crm.mdm.business.price.sdk.dto.FindPriceDto;
import com.biz.crm.mdm.business.price.sdk.enums.FindPriceUserTypeEnum;
import com.biz.crm.mdm.business.price.sdk.service.PriceModelVoService;
import com.biz.crm.mdm.business.price.sdk.vo.PriceModelVo;
import com.biz.crm.mdm.business.product.sdk.vo.ProductMediaVo;
import com.biz.crm.mdm.business.warehouse.sdk.service.WarehouseVoService;
import com.biz.crm.mdm.business.warehouse.sdk.vo.WarehouseVo;
import com.google.common.collect.Lists;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * 订单业务辅助类
 * - 用与消除冗余代码
 * - 统一部分业务流程
 *
 * @author hefan
 * @date 2022/04/15
 */
@Component
public class OrderAssist {

  @Autowired(required = false)
  private PriceModelVoService priceModelVoService;

  @Autowired(required = false)
  private MaterialVoService materialVoService;

  /**
   * 过滤货物
   *
   * @param inputOrderDetails
   * @param unionMap
   */
  public static void filterOrderDetails(List<OrderDetailPreviewVo> inputOrderDetails, Map<String, OrderDetailPreviewVo> unionMap) {
    if (!CollectionUtils.isEmpty(inputOrderDetails)) {
      for (OrderDetailPreviewVo inputOrderDetail : inputOrderDetails) {
        String orderDetailCode = inputOrderDetail.getOrderDetailCode();
        orderDetailCode = StringUtils.isBlank(orderDetailCode) ? "" : orderDetailCode;
        String goodsCode = inputOrderDetail.getGoodsCode();
        String itemType = inputOrderDetail.getItemType();
        // itemCode  特殊行存在项目编号（例如货补行）
        String itemCode = StringUtils.isBlank(inputOrderDetail.getItemCode()) ? "" : inputOrderDetail.getItemCode();
        String key = StringUtils.join(itemType, "-", goodsCode, "-", orderDetailCode,"-",itemCode);
        if (!unionMap.containsKey(key)) {
          unionMap.put(key, inputOrderDetail);
        }
      }
    }
  }

  /**
   * 过滤和初始化数据
   *
   * @param order 订单
   */
  public void filterAndInitializeData(OrderPreviewDto order) {
    // -- 过滤掉赠品信息
    List<OrderDetailPreviewDto> orderDetailsList = order.getOrderDetails();
    Validate.isTrue(!CollectionUtils.isEmpty(orderDetailsList), "没有货物传入");
    List<OrderDetailPreviewDto> nonComplimentaryGoods = orderDetailsList.stream()
        .filter(orderDetail -> {
          Validate.notBlank(orderDetail.getItemType(), "行项目类型必传！");
          return !orderDetail.getItemType().equals(ItemTypeEnum.COMPLIMENTARY_GOODS.getDictCode());
        })
        .collect(Collectors.toList());
    Validate.isTrue(!CollectionUtils.isEmpty(nonComplimentaryGoods), "没有非赠品货物传入");
    order.setOrderDetails(nonComplimentaryGoods);
    // -- 货物是否享受政策优惠
    for (OrderDetailPreviewDto nonComplimentaryGood : nonComplimentaryGoods) {
      Boolean isAutoHit = nonComplimentaryGood.getIsAutoHit();
      if (!isAutoHit) {
        nonComplimentaryGood.setOrderDetailPays(Lists.newArrayList());
      } else {
        List<OrderDetailPayPreviewDto> orderDetailPays = nonComplimentaryGood.getOrderDetailPays();
        if (CollectionUtils.isEmpty(orderDetailPays)) {
          nonComplimentaryGood.setOrderDetailPays(null);
        }
      }
    }
    // -- 获取订单最新价格（不信任入参价格）
    Set<String> productCodeSet = orderDetailsList.stream()
        .filter(a -> StringUtils.isNotBlank(a.getGoodsCode()))
        .map(OrderDetailPreviewDto::getGoodsCode)
        .collect(Collectors.toSet());
    Map<String, BigDecimal> priceMap = this.getPriceMap(order, productCodeSet);
    for (OrderDetailPreviewDto previewDto : orderDetailsList) {
      BigDecimal price = priceMap.get(previewDto.getGoodsCode());
      Validate.notNull(price, StringUtils.join("没有查询到商品价格:", previewDto.getGoodsCode()));
      previewDto.setPresetUnitPrice(price);
      BigDecimal quantity = previewDto.getQuantity();
      Validate.notNull(quantity, StringUtils.join("购买数量必须传:", previewDto.getGoodsCode()));
      previewDto.setSalesAmount(quantity.multiply(price));
    }
  }

  /**
   * key：货物编码
   * value：价格模型
   *
   * @param order          订单
   * @param goodsCodeSet 货物编码集合
   * @return {@link Map}<{@link String}, {@link PriceModelVo}>
   */
  private Map<String, BigDecimal> getPriceMap(OrderPreviewDto order, Set<String> goodsCodeSet) {
    Map<String, BigDecimal> priceMap = new HashMap<>(goodsCodeSet.size());
    // 销售订单与物料订单 询价的逻辑区分
    String orderCategory = order.getOrderCategory();
    if (orderCategory.equals(OrderCategoryEnum.SALES_ORDER.getDictCode())) {
      FindPriceDto findPriceDto = new FindPriceDto();
      findPriceDto.setUserType(FindPriceUserTypeEnum.CUSTOMER.getDictCode());
      findPriceDto.setUserCode(order.getRelateCode());
      findPriceDto.setProductCodeSet(goodsCodeSet);
      Map<String, PriceModelVo> mapPriceModelVo = this.priceModelVoService.findPrice(findPriceDto);
      Validate.notNull(mapPriceModelVo, "没有查询到价格：%s", findPriceDto.toString());
      for (Map.Entry<String, PriceModelVo> entry : mapPriceModelVo.entrySet()) {
        priceMap.put(entry.getKey(), entry.getValue().getPrice());
      }
    } else if (orderCategory.equals(OrderCategoryEnum.MATERIAL_ORDER.getDictCode())) {
      List<MaterialVo> materialList = materialVoService.findDetailByMaterialCodes(goodsCodeSet);
      Validate.isTrue(!CollectionUtils.isEmpty(materialList), "没有查询到物料价格！");
      for (MaterialVo materialVo : materialList) {
        String costPrice = materialVo.getCostPrice();
        if (StringUtils.isNotBlank(costPrice)) {
          priceMap.put(materialVo.getMaterialCode(), new BigDecimal(costPrice));
        }
      }
    } else {
      throw new IllegalArgumentException("系统中不存在该订单类别");
    }
    return priceMap;
  }

  /**
   * 合并，入参货物 和 分摊货物
   *
   * @param inputOrderDetails
   * @param orderDetails
   * @return
   */
  public static List<OrderDetailPreviewVo> unionOrderDetails(List<OrderDetailPreviewVo> inputOrderDetails, List<OrderDetailPreviewVo> orderDetails) {
    if (CollectionUtils.isEmpty(inputOrderDetails)) {
      return orderDetails;
    }
    if (CollectionUtils.isEmpty(orderDetails)) {
      return inputOrderDetails;
    }
    Map<String, OrderDetailPreviewVo> unionMap = new HashMap<>(inputOrderDetails.size() + orderDetails.size());
    OrderAssist.filterOrderDetails(orderDetails, unionMap);
    OrderAssist.filterOrderDetails(inputOrderDetails, unionMap);
    Collection<OrderDetailPreviewVo> values = unionMap.values();
    return Lists.newArrayList(values);
  }


  /**
   * 从订单中获取记账金额
   *
   * @param order
   * @param creditTallyItemRegister
   * @return
   */
  public static BigDecimal getTallyItemAmount(Order order, TallyItemRegister creditTallyItemRegister) {
    List<OrderPay> orderPays = order.getOrderPays();
    BigDecimal itemAmount = BigDecimal.ZERO;
    if (CollectionUtils.isEmpty(orderPays)) {
      return itemAmount;
    }
    for (OrderPay orderPay : orderPays) {
      String itemKey = orderPay.getItemKey();
      if (StringUtils.isBlank(itemKey)) {
        continue;
      }
      TallyItemRegisterModel tallyItemRegisterModel = creditTallyItemRegister.findTallyItemRegisterModel();
      String itemKey1 = tallyItemRegisterModel.getItemKey();
      if (itemKey.equals(itemKey1)) {
        itemAmount = orderPay.getItemAmount();
        break;
      }
    }
    return itemAmount;
  }

  @Autowired(required = false)
  private WarehouseVoService warehouseVoService;

  /**
   * 查询仓库
   *
   * @param order 订单
   * @return {@link WarehouseVo}
   */
  public WarehouseVo findWarehouseVo(Order order) {
    WarehouseVo warehouse = null;
    if (order.getOrderCategory().equals(OrderCategoryEnum.MATERIAL_ORDER.getDictCode())) {
      String warehouseCode = order.getWarehouseCode();
      warehouse = warehouseVoService.findDetailsByCode(warehouseCode);
    } else {
      warehouse = warehouseVoService.findDetailsByCityCode(order.getCityCode());
      if (warehouse == null) {
        warehouse = warehouseVoService.findDetailsByWarehouseDefault(true);
      }
    }
    return warehouse;
  }

  /**
   * 得到库存操作dto
   *
   * @param order     订单
   * @param warehouse 仓库
   * @return {@link List}<{@link ProductStockOperationDto}>
   */
  public List<ProductStockOperationDto> getProductStockOperationDtos(Order order, WarehouseVo warehouse) {
    List<OrderDetail> orderDetails = order.getOrderDetails();
    Validate.isTrue(!CollectionUtils.isEmpty(orderDetails), "订单占用库存的时候，没有货物信息");
    List<ProductStockOperationDto> list = Lists.newArrayListWithCapacity(orderDetails.size());
    for (OrderDetail orderDetail : orderDetails) {
      ProductStockOperationDto productStockOperationDto = new ProductStockOperationDto();
      productStockOperationDto.setWarehouseCode(warehouse.getWarehouseCode());
      productStockOperationDto.setWarehouseName(warehouse.getWarehouseName());
      productStockOperationDto.setProductCode(orderDetail.getGoodsCode());
      productStockOperationDto.setProductName(orderDetail.getGoodsName());
      productStockOperationDto.setProductStockOperationType(ProductStockOperationType.SALE_DELIVER.getDictCode());
      productStockOperationDto.setCustomerCode(order.getRelateCode());
      productStockOperationDto.setCustomerName(order.getRelateName());
      productStockOperationDto.setOriginalOrderCode(order.getOriginalOrderCode());
      productStockOperationDto.setOrderType(order.getOrderType());
      productStockOperationDto.setOrderCode(order.getOrderCode());
      productStockOperationDto.setOrderItemCode(orderDetail.getOrderDetailCode());
      productStockOperationDto.setQuantity(orderDetail.getQuantity());
      list.add(productStockOperationDto);
    }
    return list;
  }

  /**
   * 资金dto
   *
   * @param order                 订单
   * @param itemAmount            项目数量
   * @param capitalAdjustTypeEnum 资本调整枚举类型
   * @return {@link CostPoolCapitalDto}
   */
  public static CostPoolCapitalDto getCostPoolCapitalDto(Order order, BigDecimal itemAmount, CapitalAdjustTypeEnum capitalAdjustTypeEnum) {
    CostPoolCapitalDto dto = new CostPoolCapitalDto();
    dto.setPoolType(PoolTypeEnum.CAPITAL.getKey());
    dto.setCustomerCode(order.getRelateCode());
    dto.setFromCode(order.getOrderCode());
    dto.setFromDesc(capitalAdjustTypeEnum.getValue());
    dto.setAmount(itemAmount);
    dto.setAdjustMoney(itemAmount);
    dto.setAdjustType(capitalAdjustTypeEnum.getKey());
    dto.setAdjustTypeName(capitalAdjustTypeEnum.getValue());
    dto.setBillNo(order.getOrderCode());
    dto.setBillType(order.getOrderType());
    return dto;
  }

  /**
   * 得到授信支付dto
   *
   * @param order      订单
   * @param itemAmount 项目数量
   * @return {@link CreditPayDto}
   */
  public static CreditPayDto getCreditPayDto(Order order, BigDecimal itemAmount, CashAdjustOperateEnum cashAdjustOperateEnum, CashAdjustTypeEnum cashAdjustTypeEnum) {
    CreditPayDto dto = new CreditPayDto();
    dto.setPoolType(CreditPoolTypeEnum.CREDIT.getKey());
    dto.setCustomerCode(order.getRelateCode());
    dto.setFromCode(order.getOrderCode());
    dto.setFromDesc(cashAdjustOperateEnum.getValue());
    dto.setAmount(itemAmount);
    dto.setAdjustTypeCode(cashAdjustTypeEnum.getDictCode());
    dto.setAdjustOperateCode(cashAdjustOperateEnum.getDictCode());
    return dto;
  }

  /**
   * 得到折扣DTO
   *
   * @param order      订单
   * @param itemAmount 项目数量
   * @return {@link CostPoolDiscountDto}
   */
  public static CostPoolDiscountDto getCostPoolDiscountDto(Order order, BigDecimal itemAmount, PoolOperationTypeEnum poolOperationTypeEnum) {
    CostPoolDiscountDto dto = new CostPoolDiscountDto();
    dto.setPoolType(com.biz.crm.dms.business.costpool.discount.sdk.enums.PoolTypeEnum.DISCOUNT.getKey());
    dto.setCustomerCode(order.getRelateCode());
    dto.setFromCode(order.getOrderCode());
    dto.setFromDesc(poolOperationTypeEnum.getValue());
    dto.setAmount(itemAmount);
    dto.setPoolGroup(PoolGroupEnum.DEFAULT.getDictCode());
    dto.setPayType(PoolPayTypeEnum.DISCOUNT.getDictCode());
    dto.setUseType(PoolUseTypeEnum.DEFAULT.getDictCode());
    dto.setCustomerName(order.getRelateName());
    dto.setOperationType(poolOperationTypeEnum.getDictCode());
    return dto;
  }

  /**
   * 得到货补 DTO
   *
   * @param order       订单
   * @param orderDetail 订单细节
   * @param salesAmount 销售金额
   * @return {@link CostPoolReplenishmentDto}
   */
  public static CostPoolReplenishmentDto getCostPoolReplenishmentDto(Order order, OrderDetail orderDetail, BigDecimal salesAmount, com.biz.crm.dms.business.costpool.replenishment.sdk.enums.PoolOperationTypeEnum poolOperationTypeEnum) {
    CostPoolReplenishmentDto dto = new CostPoolReplenishmentDto();
    dto.setPoolType(com.biz.crm.dms.business.costpool.replenishment.sdk.enums.PoolTypeEnum.Replenishment.getKey());
    dto.setCustomerCode(order.getRelateCode());
    dto.setFromCode(order.getOrderCode());
    dto.setFromDesc(poolOperationTypeEnum.getValue());
    dto.setAmount(salesAmount);
    dto.setPoolGroup(com.biz.crm.dms.business.costpool.replenishment.sdk.enums.PoolGroupEnum.DEFAULT.getDictCode());
    dto.setPoolCode(orderDetail.getItemCode());
    dto.setPayType(com.biz.crm.dms.business.costpool.replenishment.sdk.enums.PoolPayTypeEnum.Replenishment.getDictCode());
    dto.setUseType(com.biz.crm.dms.business.costpool.replenishment.sdk.enums.PoolUseTypeEnum.DEFAULT.getDictCode());
    dto.setCustomerName(order.getRelateName());
    dto.setOperationType(poolOperationTypeEnum.getDictCode());
    return dto;
  }

  /**
   * 得到文件vos
   *
   * @param productVo 产品签证官
   * @return {@link List}<{@link ProductMediaVo}>
   */
  public static List<OrderFileVo> getFileVos(List<ProductMediaVo> productVo) {
    List<ProductMediaVo> productMediaVos = productVo;
    if (CollectionUtils.isEmpty(productMediaVos)) {
      return null;
    }
    List<OrderFileVo> pictureFileList = new ArrayList<>(productMediaVos.size());
    for (ProductMediaVo item : productMediaVos) {
      OrderFileVo fileVo = new OrderFileVo();
      fileVo.setFileCode(item.getId());
      pictureFileList.add(fileVo);
    }
    return pictureFileList;
  }
}
