package com.biz.crm.dms.business.order.local.service.internal;


import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.sdk.service.GenerateCodeService;
import com.biz.crm.dms.business.order.common.sdk.enums.OrderCategoryEnum;
import com.biz.crm.dms.business.order.local.entity.Order;
import com.biz.crm.dms.business.order.local.entity.OrderDetail;
import com.biz.crm.dms.business.order.local.repository.OrderDetailRepository;
import com.biz.crm.dms.business.order.local.service.OrderDetailPayService;
import com.biz.crm.dms.business.order.local.service.OrderDetailService;
import com.biz.crm.dms.business.order.sdk.constant.OrderConstant;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.apache.commons.lang3.time.DateFormatUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.math.BigDecimal;
import java.util.Date;
import java.util.List;
import java.util.Objects;
import java.util.concurrent.TimeUnit;

/**
 * 订单明细(OrderDetail)表服务实现类
 *
 * @author makejava
 * @since 2022-01-08 15:13:12
 */
@Service("orderDetailService")
public class OrderDetailServiceImpl implements OrderDetailService {

  @Autowired(required = false)
  private OrderDetailRepository orderDetailRepository;

  @Autowired(required = false)
  private OrderDetailPayService orderDetailPayService;

  @Autowired(required = false)
  private GenerateCodeService generateCodeService;

  /**
   * 分页查询数据
   *
   * @param pageable    分页对象
   * @param orderDetail 实体对象
   * @return
   */
  @Override
  public Page<OrderDetail> findByConditions(Pageable pageable, OrderDetail orderDetail) {
    ObjectUtils.defaultIfNull(pageable, PageRequest.of(0, 50));
    if (Objects.isNull(orderDetail)) {
      orderDetail = new OrderDetail();
    }
    return this.orderDetailRepository.findByConditions(pageable, orderDetail);
  }

  /**
   * 通过主键查询单条数据
   *
   * @param id 主键
   * @return 单条数据
   */
  @Override
  public OrderDetail findById(String id) {
    if (StringUtils.isBlank(id)) {
      return null;
    }
    return this.orderDetailRepository.getById(id);
  }

  /**
   * 新增数据
   *
   * @param orderDetail 实体对象
   * @return 新增结果
   */
  @Transactional
  @Override
  public OrderDetail create(OrderDetail orderDetail) {
    this.createValidate(orderDetail);
    this.orderDetailRepository.saveOrUpdate(orderDetail);
    return orderDetail;
  }

  /**
   * 修改新据
   *
   * @param orderDetail 实体对象
   * @return 修改结果
   */
  @Transactional
  @Override
  public OrderDetail update(OrderDetail orderDetail) {
    this.updateValidate(orderDetail);
    this.orderDetailRepository.saveOrUpdate(orderDetail);
    return orderDetail;
  }

  /**
   * 删除数据
   *
   * @param idList 主键结合
   * @return 删除结果
   */
  @Transactional
  @Override
  public void delete(List<String> idList) {
    Validate.isTrue(!CollectionUtils.isEmpty(idList), "删除数据时，主键集合不能为空！");
    this.orderDetailRepository.removeByIds(idList);
  }

  @Transactional
  @Override
  public void createBatch(Order order) {
    Validate.notNull(order, "批量持久化订单明细时，订单主信息不能为空");
    Boolean draft = order.getDraft();
    List<OrderDetail> orderDetails = order.getOrderDetails();
    if (draft && CollectionUtils.isEmpty(orderDetails)) {
      return;
    }
    Validate.isTrue(!CollectionUtils.isEmpty(orderDetails), "批量持久化订单明细时,订单明细信息，不能为空！");
    for (OrderDetail orderDetail : orderDetails) {
      orderDetail.setTenantCode(order.getTenantCode());
      orderDetail.setAppCode(order.getAppCode());
      orderDetail.setOrderCode(order.getOrderCode());
      this.setOrderDetailCode(orderDetail);
      this.createValidate(orderDetail);
      if (!draft) {
        this.requiredValidate(orderDetail);
      }
      if (order.getOrderCategory().equals(OrderCategoryEnum.SALES_ORDER.getDictCode())) {
        Validate.notBlank(orderDetail.getSpec(), "规格,不能为空！");
      }
    }
    this.orderDetailRepository.saveOrUpdateBatch(orderDetails);
    // 从表信息的保存
    this.saveOrderDetailPay(orderDetails);
  }

  @Transactional
  @Override
  public void deleteByOrderCode(String orderCode) {
    Validate.notBlank(orderCode, "删除时，要传入订单编码");
    this.orderDetailPayService.deleteByOrderCode(orderCode);
    this.orderDetailRepository.deleteByOrderCode(orderCode);
  }

  /**
   * 填充编码 PS:已经存在编码就没必要再生成了。
   * @param orderDetail
   */
  private void setOrderDetailCode(OrderDetail orderDetail) {
    /**
     * 新增时，不信任一切code
     */
    String ruleCode = StringUtils.join(OrderConstant.ORDER_DETAIL_RULE_CODE, DateFormatUtils.format(new Date(), "yyyyMMdd"));
    String orderDetailCode = this.generateCodeService.generateCode(ruleCode, 1, 5, 2, TimeUnit.DAYS).get(0);
    orderDetail.setOrderDetailCode(orderDetailCode);
  }

  /**
   * 必填校验
   * @param orderDetail
   */
  private void requiredValidate(OrderDetail orderDetail) {
    /**
     * 成本价
     * 实际单价 = 组内分摊后的单价（正向）
     * 平均价
     * 销售金额
     * 实际支付金额（销售金额 - 优惠金额 - 抵扣金额）
     *
     */
    BigDecimal presetUnitPrice = orderDetail.getPresetUnitPrice();
    BigDecimal averageUnitPrice = orderDetail.getAverageUnitPrice();
    BigDecimal salesAmount = orderDetail.getSalesAmount();
    BigDecimal shouldPaymentAmount = orderDetail.getShouldPaymentAmount();
    Validate.notNull(presetUnitPrice, "成本单价,不能为空");
    Validate.notNull(averageUnitPrice, "平均价,不能为空");
    Validate.notNull(salesAmount, "销售金额,不能为空");
    Validate.notNull(shouldPaymentAmount, "实际支付金额（销售金额 - 优惠金额 - 抵扣金额）,不能为空");
  }

  /**
   * 持久化 行上该扣项目
   *
   * @param orderDetails
   */
  private void saveOrderDetailPay(List<OrderDetail> orderDetails) {
    if (CollectionUtils.isEmpty(orderDetails)) {
      return;
    }
    this.orderDetailPayService.createBatch(orderDetails);
  }

  /**
   * 创建验证
   * --单条信息的数据完整性的基本验证
   * --不含业务验证
   *
   * @param orderDetail
   */
  private void createValidate(OrderDetail orderDetail) {
    Validate.notNull(orderDetail, "新增时，对象信息不能为空！");
    Validate.notBlank(orderDetail.getTenantCode(), "新增数据时，租户编号不能为空！");
    Validate.notBlank(orderDetail.getAppCode(), "新增数据时，品牌商租户编号不能为空！");
    Validate.notBlank(orderDetail.getOrderCode(), "新增数据时，订单编号不能为空！");
    /**
     * 只是想想，不用校验 {
     * 商品信息的校验
     * --商品在系统中是否存在
     * --商品的价格是否正确
     * --商品+行项目类型是否可行
     * ----商品是否可为赠品
     * ----商品是否可为货补
     * ----商品是否可为免费
     * }
     * 数量不能小于0
     * 长度校验
     */
    String orderDetailCode = orderDetail.getOrderDetailCode();
    String itemType = orderDetail.getItemType();
    String itemCode = orderDetail.getItemCode();
    String goodsCode = orderDetail.getGoodsCode();
    String goodsName = orderDetail.getGoodsName();
    String spec = orderDetail.getSpec();
    String unite = orderDetail.getUnite();
    BigDecimal quantity = orderDetail.getQuantity();
    if (ObjectUtils.isEmpty(quantity)) {
      quantity = BigDecimal.ZERO;
    }
    boolean gtZero = quantity.compareTo(BigDecimal.ZERO) > -1;
    Validate.isTrue(gtZero, "订购数量，不能小于0！");

    Validate.notBlank(orderDetailCode, "明细编码,不能为空！");
    Validate.notBlank(itemType, "行项目类型,不能为空！");
    Validate.notBlank(goodsCode, "商品/物料编码,不能为空！");
    Validate.notBlank(goodsName, "商品/物料名称,不能为空！");
    Validate.notBlank(unite, "销售单位,不能为空！");
    Validate.isTrue(orderDetailCode.length() < 255, "明细编码,限定长度(255)，请检查！");
    Validate.isTrue(itemType.length() < 64, "行项目类型,限定长度(64)，请检查！");
    Validate.isTrue(StringUtils.isBlank(itemCode) || itemCode.length() < 64, "行项目类型的特有业务编码,限定长度(64)，请检查！");
    Validate.isTrue(goodsCode.length() < 64, "商品/物料编码,限定长度(64)，请检查！");
    Validate.isTrue(goodsName.length() < 100, "商品/物料名称,限定长度(100)，请检查！");
    Validate.isTrue(StringUtils.isBlank(spec) || spec.length() < 64, "规格,限定长度(64)，请检查！");
    Validate.isTrue(unite.length() < 64, "销售单位,限定长度(64)，请检查！");
  }

  /**
   * 修改验证
   *
   * @param orderDetail
   */
  private void updateValidate(OrderDetail orderDetail) {
    Validate.notNull(orderDetail, "修改时，对象信息不能为空！");
    Validate.notBlank(orderDetail.getId(), "修改时，不能为空！");
    Validate.notBlank(orderDetail.getTenantCode(), "修改时，租户编号不能为空！");
    Validate.notBlank(orderDetail.getAppCode(), "修改时，品牌商租户编号不能为空！");

  }
}

