package com.biz.crm.dms.business.order.local.service.internal;

import com.alibaba.fastjson.JSONArray;
import com.biz.crm.dms.business.costpool.capital.sdk.dto.CostPoolCapitalDto;
import com.biz.crm.dms.business.costpool.capital.sdk.enums.CapitalAdjustTypeEnum;
import com.biz.crm.dms.business.costpool.credit.sdk.dto.CreditPayDto;
import com.biz.crm.dms.business.costpool.credit.sdk.enums.CashAdjustOperateEnum;
import com.biz.crm.dms.business.costpool.credit.sdk.enums.CashAdjustTypeEnum;
import com.biz.crm.dms.business.costpool.discount.sdk.dto.CostPoolDiscountDto;
import com.biz.crm.dms.business.costpool.discount.sdk.enums.PoolOperationTypeEnum;
import com.biz.crm.dms.business.costpool.replenishment.sdk.dto.CostPoolReplenishmentDto;
import com.biz.crm.dms.business.costpool.sdk.service.CostPoolVoService;
import com.biz.crm.dms.business.order.common.sdk.enums.ItemTypeEnum;
import com.biz.crm.dms.business.order.common.sdk.enums.OrderCategoryEnum;
import com.biz.crm.dms.business.order.common.sdk.enums.OrderStatusEnum;
import com.biz.crm.dms.business.order.common.sdk.enums.OrderTypeEnum;
import com.biz.crm.dms.business.order.common.sdk.register.TallyItemRegister;
import com.biz.crm.dms.business.order.local.entity.Order;
import com.biz.crm.dms.business.order.local.entity.OrderDetail;
import com.biz.crm.dms.business.order.local.repository.OrderRepository;
import com.biz.crm.dms.business.order.local.service.OrderFlowService;
import com.biz.crm.dms.business.order.local.service.assist.OrderAssist;
import com.biz.crm.dms.business.order.sdk.constant.OrderConstant;
import com.biz.crm.dms.business.order.sdk.dto.OrderConfirmDto;
import com.biz.crm.dms.business.psi.product.sdk.dto.productstock.ProductStockOperationDto;
import com.biz.crm.dms.business.psi.product.sdk.service.productstock.ProductStockVoService;
import com.biz.crm.mdm.business.warehouse.sdk.vo.WarehouseVo;
import com.biz.crm.workflow.sdk.dto.ProcessBusinessDto;
import com.biz.crm.workflow.sdk.service.ProcessBusinessService;
import com.biz.crm.workflow.sdk.vo.ProcessBusinessVo;
import com.bizunited.nebula.common.util.JsonUtils;
import com.bizunited.nebula.event.sdk.service.NebulaNetEventClient;
import com.google.common.collect.Lists;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.math.BigDecimal;
import java.util.List;
import java.util.Objects;

/**
 * @author hefan
 */
@Service
public class OrderFlowServiceImpl implements OrderFlowService {

  @Autowired(required = false)
  private OrderRepository orderRepository;

  @Autowired(required = false)
  private ProductStockVoService productStockVoService;

  @Autowired(required = false)
  private CostPoolVoService costPoolVoService;
  
  @Autowired(required = false)
  private NebulaNetEventClient nebulaNetEventClient;

  @Transactional
  @Override
  public void pay(OrderConfirmDto orderConfirmDto) {
    String orderCode = orderConfirmDto.getOrderCode();
    /**
     * - 查询出订单信息
     * - 应当时处于某种状态的订单才可以去支付
     * - 支付费用
     */
    Validate.notBlank(orderCode, "支付订单时，订单编码不能为空");
    List<Order> orders = orderRepository.findDetailByOrderCodes(Lists.newArrayList(orderCode));
    // 因为只传了一个code，所以只需要一条信息
    Validate.isTrue(!CollectionUtils.isEmpty(orders), "支付订单时，没有查询到订单信息");
    for (Order order : orders) {
      String orderType = order.getOrderType();
      boolean isFree = OrderTypeEnum.FREE.getDictCode().equals(orderType);
      String orderCategory = order.getOrderCategory();
      boolean isMaterial = OrderCategoryEnum.MATERIAL_ORDER.getDictCode().equals(orderCategory);
      if (!isFree && !isMaterial) {
        this.occupyOrder(order);
      }
      this.occupyStock(order);
      this.commitProcess(orderConfirmDto, order);
    }
  }

  @Transactional
  @Override
  public void occupyResource(OrderConfirmDto orderConfirmDto, Boolean excludeCost) {
    String orderCode = orderConfirmDto.getOrderCode();
    /**
     * - 查询出订单信息
     * - 应当时处于某种状态的订单才可以去支付
     * - 支付费用
     */
    Validate.notBlank(orderCode, "支付订单时，订单编码不能为空");
    List<Order> orders = orderRepository.findDetailByOrderCodes(Lists.newArrayList(orderCode));
    // 因为只传了一个code，所以只需要一条信息
    Validate.isTrue(!CollectionUtils.isEmpty(orders), "支付订单时，没有查询到订单信息");
    for (Order order : orders) {
      if (!excludeCost) {
        this.occupyOrder(order);
      }
      this.occupyStock(order);
      this.commitProcess(orderConfirmDto, order);
    }
  }

  @Transactional
  @Override
  public void occupyStock(OrderConfirmDto orderConfirmDto) {
    String orderCode = orderConfirmDto.getOrderCode();
    Validate.notBlank(orderCode, "支付订单时，订单编码不能为空");
    List<Order> orders = orderRepository.findDetailByOrderCodes(Lists.newArrayList(orderCode));
    Validate.isTrue(!CollectionUtils.isEmpty(orders), "支付订单时，没有查询到订单信息");
    for (Order order : orders) {
      this.occupyStock(order);
    }
  }

  @Autowired(required = false)
  private OrderAssist orderAssist;

  /**
   * 占用库存
   *
   * @param order
   */
  private void occupyStock(Order order) {
    WarehouseVo warehouse = orderAssist.findWarehouseVo(order);
    Validate.notNull(warehouse, "订单占用库存的时候，没有找到仓库信息");
    List<ProductStockOperationDto> list = orderAssist.getProductStockOperationDtos(order, warehouse);
    this.productStockVoService.frozenBatch(list);
  }

  /**
   * 单个订单的占用
   *
   * @param order
   */
  private void occupyOrder(Order order) {
    JSONArray array = new JSONArray();
    // 货补、折扣、资金、授信都要去费用池扣减。
    CostPoolCapitalDto costPoolCapitalDto = this.getCapitalDto(order);
    if (Objects.nonNull(costPoolCapitalDto)) {
      array.add(costPoolCapitalDto);
    }
    CreditPayDto creditPayDto = this.getCreditPayDto(order);
    if (Objects.nonNull(creditPayDto)) {
      array.add(creditPayDto);
    }
    CostPoolDiscountDto discountDto = this.getDiscountDto(order);
    if (Objects.nonNull(discountDto)) {
      array.add(discountDto);
    }
    List<CostPoolReplenishmentDto> replenishmentDto = this.getReplenishmentDto(order);
    if (!CollectionUtils.isEmpty(replenishmentDto)) {
      array.addAll(replenishmentDto);
    }
    this.costPoolVoService.handleAdjust(array);
  }

  /**
   * 创建 货补扣减需要的对象
   *
   * @param order
   * @return
   */
  private List<CostPoolReplenishmentDto> getReplenishmentDto(Order order) {
    List<OrderDetail> orderDetails = order.getOrderDetails();
    if (CollectionUtils.isEmpty(orderDetails)) {
      return null;
    }
    List<CostPoolReplenishmentDto> list = Lists.newArrayListWithCapacity(orderDetails.size());
    // 只考虑货补商品
    for (OrderDetail orderDetail : orderDetails) {
      // --行上的销售总金额
      BigDecimal salesAmount = orderDetail.getSalesAmount();
      String itemType = orderDetail.getItemType();
      if (!itemType.equals(ItemTypeEnum.COMPENSATED_GOODS.getDictCode())) {
        continue;
      }
      CostPoolReplenishmentDto dto = OrderAssist.getCostPoolReplenishmentDto(order, orderDetail, salesAmount, com.biz.crm.dms.business.costpool.replenishment.sdk.enums.PoolOperationTypeEnum.OCCUPY_USE);
      list.add(dto);
    }
    return list;
  }


  @Qualifier("DiscountApportionTallyItemRegister")
  @Autowired(required = false)
  private TallyItemRegister discountApportionTallyItemRegister;

  /**
   * 创建 折扣扣减需要的对象
   *
   * @param order
   * @return
   */
  private CostPoolDiscountDto getDiscountDto(Order order) {
    BigDecimal itemAmount = OrderAssist.getTallyItemAmount(order, discountApportionTallyItemRegister);
    if (itemAmount.compareTo(BigDecimal.ZERO) == 0) {
      return null;
    }
    return OrderAssist.getCostPoolDiscountDto(order, itemAmount, PoolOperationTypeEnum.OCCUPY_USE);
  }

  @Qualifier("CreditTallyItemRegister")
  @Autowired(required = false)
  private TallyItemRegister creditTallyItemRegister;

  /**
   * 创建 信用扣减需要的对象
   * CreditRefundDto
   *
   * @param order
   * @return
   */
  private CreditPayDto getCreditPayDto(Order order) {
    BigDecimal itemAmount = OrderAssist.getTallyItemAmount(order, creditTallyItemRegister);
    if (itemAmount.compareTo(BigDecimal.ZERO) == 0) {
      return null;
    }
    return OrderAssist.getCreditPayDto(order, itemAmount, CashAdjustOperateEnum.OCCUPY_USE, CashAdjustTypeEnum.OCCUPY_USE);
  }

  @Qualifier("CapitalTallyItemRegister")
  @Autowired(required = false)
  private TallyItemRegister capitalTallyItemRegister;

  /**
   * 创建 资金扣减需要的对象
   *
   * @return
   */
  private CostPoolCapitalDto getCapitalDto(Order order) {
    BigDecimal itemAmount = OrderAssist.getTallyItemAmount(order, capitalTallyItemRegister);
    if (itemAmount.compareTo(BigDecimal.ZERO) == 0) {
      return null;
    }
    /**
     * 资金
     * CapitalAdjustTypeEnum.getKey()
     */
    return OrderAssist.getCostPoolCapitalDto(order, itemAmount, CapitalAdjustTypeEnum.OCCUPY_USE);
  }

  @Autowired(required = false)
  private ProcessBusinessService processBusinessService;

  private void commitProcess(OrderConfirmDto orderDto, Order order) {
    ProcessBusinessDto businessProcessDto = orderDto.getProcessBusiness();
    businessProcessDto.setProcessTitle(StringUtils.join("订单审批", "：", order.getOrderCode()));
    businessProcessDto.setBusinessNo(order.getOrderCode());
    businessProcessDto.setBusinessFormJson(JsonUtils.obj2JsonString(orderDto));
    businessProcessDto.setBusinessCode(OrderConstant.BUSINESS_CODE);
    ProcessBusinessVo processBusinessVo = processBusinessService.processStart(businessProcessDto);
    this.orderRepository.updateOrderStatusAndProcessNumberById(OrderStatusEnum.AWAIT_APPROVE.getDictCode(), processBusinessVo.getProcessNo(), order.getId());
  }
}
