package com.biz.crm.dms.business.order.local.service.internal;

import com.biz.crm.business.common.sdk.service.LoginUserService;
import com.biz.crm.dms.business.costpool.sdk.dto.CostPoolDto;
import com.biz.crm.dms.business.costpool.sdk.strategy.CostPoolRegister;
import com.biz.crm.dms.business.costpool.sdk.strategy.CostPoolStrategy;
import com.biz.crm.dms.business.costpool.sdk.vo.CostPoolVo;
import com.biz.crm.dms.business.order.apportion.sdk.model.OrderApportionContext;
import com.biz.crm.dms.business.order.apportion.sdk.service.OrderApportionVoService;
import com.biz.crm.dms.business.order.apportion.sdk.service.OrderApportionVoServiceChain;
import com.biz.crm.dms.business.order.apportion.sdk.service.OrderApportionVoServiceChainInstance;
import com.biz.crm.dms.business.order.apportion.sdk.vo.OrderApportionVo;
import com.biz.crm.dms.business.order.common.sdk.dto.OrderDto;
import com.biz.crm.dms.business.order.common.sdk.enums.ItemTypeEnum;
import com.biz.crm.dms.business.order.common.sdk.enums.OrderCommitPhaseEnum;
import com.biz.crm.dms.business.order.common.sdk.enums.OrderTypeEnum;
import com.biz.crm.dms.business.order.common.sdk.enums.OrderValidationProcessNodeEnum;
import com.biz.crm.dms.business.order.common.sdk.model.TallyItemRegisterModel;
import com.biz.crm.dms.business.order.common.sdk.register.TallyItemRegister;
import com.biz.crm.dms.business.order.common.sdk.vo.OrderConfigVo;
import com.biz.crm.dms.business.order.config.sdk.service.OrderValidateVoService;
import com.biz.crm.dms.business.order.local.entity.Order;
import com.biz.crm.dms.business.order.local.service.OrderService;
import com.biz.crm.dms.business.order.local.service.assist.OrderAssist;
import com.biz.crm.dms.business.order.sdk.dto.OrderPayConfirmDto;
import com.biz.crm.dms.business.order.sdk.dto.OrderPreviewDto;
import com.biz.crm.dms.business.order.sdk.service.OrderPreviewService;
import com.biz.crm.dms.business.order.sdk.vo.OrderDetailPayPreviewVo;
import com.biz.crm.dms.business.order.sdk.vo.OrderDetailPreviewVo;
import com.biz.crm.dms.business.order.sdk.vo.OrderFileVo;
import com.biz.crm.dms.business.order.sdk.vo.OrderPayPreviewVo;
import com.biz.crm.dms.business.order.sdk.vo.OrderPolicyDetailVo;
import com.biz.crm.dms.business.order.sdk.vo.OrderPreviewVo;
import com.biz.crm.dms.business.order.verification.sdk.model.OrderVerificationContext;
import com.biz.crm.dms.business.order.verification.sdk.service.OrderVerificationService;
import com.biz.crm.mdm.business.product.sdk.service.ProductVoService;
import com.biz.crm.mdm.business.product.sdk.vo.ProductVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.google.common.collect.Lists;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import java.math.BigDecimal;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.stream.Collectors;

/**
 * @author he fan
 */
@Service
public class OrderPreviewServiceImpl implements OrderPreviewService {

  @Autowired(required = false)
  private OrderValidateVoService orderValidateVoService;

  @Autowired(required = false)
  private NebulaToolkitService nebulaToolkitService;

  @Autowired(required = false)
  private OrderVerificationService orderVerificationService;

  @Autowired(required = false)
  private OrderApportionVoServiceChainInstance orderApportionVoServiceChainInstance;

  @Autowired(required = false)
  private List<OrderApportionVoService> orderApportionVoServiceList;

  @Autowired(required = false)
  private OrderAssist orderAssist;
  /**
   * 订单预览--预计算、不加整单分摊：
   * 前端将基本信息、商品信息传入，获取命中的政策，查看优惠情况。
   * -- 费用池的优惠渠道不会在这个阶段执行分摊
   * -- 不做任何数据记录处理
   */
  @Override
  public OrderPreviewVo handlePreview(OrderPreviewDto order) {
    /**
     * 行为的操作步骤
     * --查询订单配置
     * --业务校验
     * --不做任何数据记录处理
     * --分摊service排序后，获取分摊chain
     * --组装返回vo
     * --返回结算信息的部分展示数据
     */
    // 清洗数据
    this.orderAssist.filterAndInitializeData(order);
    // 查询订单配置
    OrderConfigVo orderValidateDto = this.orderValidateVoService.findByOrderTypeAndCustomerCode(order.getOrderType(), order.getRelateCode());
    Validate.notNull(orderValidateDto, String.format("订单类型%s，客户编码%s，没有订单配置，请检查！", OrderTypeEnum.findByCode(order.getOrderType()).getValue(), order.getRelateCode()));
    OrderDto orderContextDto = this.nebulaToolkitService.copyObjectByBlankList(order, OrderDto.class, HashSet.class, LinkedList.class);
    // 订单校验上下文
    OrderVerificationContext orderVerificationContext = new OrderVerificationContext();
    orderVerificationContext.setOrderDto(orderContextDto);
    orderVerificationContext.setOrderConfigVo(orderValidateDto);
    orderVerificationContext.setProcessNodeEnum(OrderValidationProcessNodeEnum.ONE);
    this.orderVerificationService.execute(orderVerificationContext);
    // 不做任何数据记录处理-- 组装分摊计算用的上下文,并设置提交阶段枚举
    OrderApportionContext context = new OrderApportionContext();
    context.setOrderConfigVo(orderValidateDto);
    context.setOrderDto(orderContextDto);
    context.setOrderCommitPhaseEnum(OrderCommitPhaseEnum.PREVIEW);
    // 分摊service排序后，获取分摊chain，分摊service内部根据订单提交阶段 自行决定是否执行
    Validate.isTrue(!CollectionUtils.isEmpty(orderApportionVoServiceList), "还没有分摊实现，请确认项目是否正确构建");
    Collections.sort(orderApportionVoServiceList);
    OrderApportionVoServiceChain chain = this.orderApportionVoServiceChainInstance.getChain(orderApportionVoServiceList);
    chain.execute(context);
    OrderApportionVo orderApportionVo = context.getOrderApportionVo();
    // 组装返回vo
    // --入参中的原始数据拷入vo
    OrderPreviewVo orderPreviewVo = nebulaToolkitService.copyObjectByBlankList(order, OrderPreviewVo.class, HashSet.class, LinkedList.class);
    // --分摊数据拷入另一个vo
    OrderPreviewVo apportionOrderPreviewVo = nebulaToolkitService.copyObjectByBlankList(orderApportionVo, OrderPreviewVo.class, HashSet.class, LinkedList.class);
    // --orderDetails需要替换
    // ----全部入参（含货补）
    List<OrderDetailPreviewVo> inputOrderDetails = orderPreviewVo.getOrderDetails();
    // ----只有本品和赠品
    List<OrderDetailPreviewVo> orderDetails = apportionOrderPreviewVo.getOrderDetails();
    // ----将货补，与本品，赠品 收集在一起
    List<OrderDetailPreviewVo> unionSet = OrderAssist.unionOrderDetails(inputOrderDetails, orderDetails);
    orderPreviewVo.setOrderDetails(unionSet);
    // 返回结算信息的部分展示数据
    this.tabulateData(orderPreviewVo);
    // - 按照实际支付设置支付金额
    this.setOrderPays(orderPreviewVo, order);
    // 补sku图片
    this.findSkuMedia(orderPreviewVo);
    return orderPreviewVo;
  }

  @Qualifier("CapitalTallyItemRegister")
  @Autowired(required = false)
  private TallyItemRegister capitalTallyItemRegister;

  @Qualifier("CreditTallyItemRegister")
  @Autowired(required = false)
  private TallyItemRegister creditTallyItemRegister;

  @Autowired(required = false)
  @Qualifier("costPoolCapitalStrategyImpl")
  private CostPoolStrategy costPoolCapitalStrategy;

  @Autowired(required = false)
  @Qualifier("costPoolCapitalRegisterImpl")
  private CostPoolRegister costPoolCapitalRegister;

  @Autowired(required = false)
  @Qualifier("CostPoolCreditStrategyImpl")
  private CostPoolStrategy costPoolCreditStrategy;

  @Autowired(required = false)
  @Qualifier("CostPoolCreditRegisterImpl")
  private CostPoolRegister costPoolCreditRegister;


  /**
   * 设置订单支付
   *
   * @param order 订单
   * @param orderConfirmDto
   */
  private void setOrderPays(OrderPreviewVo order, OrderPreviewDto orderConfirmDto) {
    List<OrderPayPreviewVo> orderPays = order.getOrderPays();
    if (CollectionUtils.isEmpty(orderPays)) {
      orderPays = Lists.newLinkedList();
      order.setOrderPays(orderPays);
    }
    // - 清空支付项目
    Iterator<OrderPayPreviewVo> iterator = orderPays.iterator();
    while (iterator.hasNext()) {
      OrderPayPreviewVo next = iterator.next();
      Boolean itemGroupType = next.getItemGroupType();
      if (itemGroupType) {
        iterator.remove();
      }
    }
    // 换货才需要校验
    if (CollectionUtils.isEmpty(orderConfirmDto.getRefundOrderPays())) {
      return;
    }
    // 特殊订单不予处理
    String orderType = order.getOrderType();
    boolean isFree = OrderTypeEnum.FREE.getDictCode().equals(orderType);
    if (isFree) {
      return;
    }
    // 资金
    TallyItemRegisterModel capitalTallyItem = capitalTallyItemRegister.findTallyItemRegisterModel();
    OrderPayPreviewVo capital = new OrderPayPreviewVo();
    Boolean pay = this.isPay(capitalTallyItem);
    capital.setItemGroupType(pay);
    capital.setItemGroupKey(capitalTallyItem.getItemGroupKey());
    capital.setItemKey(capitalTallyItem.getItemKey());
    capital.setItemName(capitalTallyItem.getItemName());
    orderPays.add(capital);
    // 授信
    TallyItemRegisterModel creditTallyItem = creditTallyItemRegister.findTallyItemRegisterModel();
    OrderPayPreviewVo credit = new OrderPayPreviewVo();
    Boolean isPay = this.isPay(creditTallyItem);
    credit.setItemGroupType(isPay);
    credit.setItemGroupKey(creditTallyItem.getItemGroupKey());
    credit.setItemKey(creditTallyItem.getItemKey());
    credit.setItemName(creditTallyItem.getItemName());
    orderPays.add(credit);
    // 分钱
    BigDecimal actualAmountPaid = order.getActualAmountPaid();
    // - 可用资金余额
    BigDecimal capitalUsableAmount = this.findUsableAmount(costPoolCapitalRegister, order.getRelateCode(), costPoolCapitalStrategy);
    //- 可用信用余额
    BigDecimal creditUsableAmount = this.findUsableAmount(costPoolCreditRegister, order.getRelateCode(), costPoolCreditStrategy);
    //- 加上退货那部分钱
    List<OrderPayConfirmDto> refundOrderPays = orderConfirmDto.getRefundOrderPays();
    if (!CollectionUtils.isEmpty(refundOrderPays)) {
      for (OrderPayConfirmDto refundOrderPay : refundOrderPays) {
        BigDecimal itemAmount = refundOrderPay.getItemAmount();
        Boolean itemGroupType = refundOrderPay.getItemGroupType();
        if (itemGroupType == null && !itemGroupType) {
          continue;
        }
        boolean isCapital = capitalTallyItem.getItemKey().equals(refundOrderPay.getItemKey());
        if (isCapital) {
          capitalUsableAmount = capitalUsableAmount.add(itemAmount);
        }
        boolean isCredit = creditTallyItem.getItemKey().equals(refundOrderPay.getItemKey());
        if (isCredit) {
          creditUsableAmount = creditUsableAmount.add(itemAmount);
        }
      }
    }
    // - 分钱支付
    BigDecimal totalUsableAmount = capitalUsableAmount.add(creditUsableAmount);
    int compare = actualAmountPaid.compareTo(totalUsableAmount);
    if (compare > 0) {
      throw new IllegalArgumentException("余额不足，无法支付订单");
    } else if (compare < 0) {
      int compareCapital = actualAmountPaid.compareTo(capitalUsableAmount);
      if (compareCapital > 0) {
        capital.setItemAmount(capitalUsableAmount);
        credit.setItemAmount(actualAmountPaid.subtract(capitalUsableAmount));
      } else if (compareCapital < 0) {
        capital.setItemAmount(actualAmountPaid);
        credit.setItemAmount(BigDecimal.ZERO);
      } else{
        capital.setItemAmount(actualAmountPaid);
        credit.setItemAmount(BigDecimal.ZERO);
      }
    } else {
      capital.setItemAmount(capitalUsableAmount);
      credit.setItemAmount(creditUsableAmount);
    }
  }

  private Boolean isPay(TallyItemRegisterModel creditTallyItem) {
    Integer itemGroupTypeInt = Integer.valueOf(creditTallyItem.getTallyItemGroupType().getDictCode());
    Boolean isPay = itemGroupTypeInt == 0 ? false : true;
    return isPay;
  }

  /**
   * 找到可用数量
   *
   * @param costPoolRegister 成本池注册
   * @param customerCode     客户代码
   * @param costPoolStrategy 成本池策略
   */
  private BigDecimal findUsableAmount(CostPoolRegister costPoolRegister, String customerCode, CostPoolStrategy costPoolStrategy) {
    // - 查询客户的资金信息
    CostPoolDto costPoolDto = new CostPoolDto();
    costPoolDto.setPoolType(costPoolRegister.getKey());
    costPoolDto.setCustomerCode(customerCode);
    List<CostPoolVo> costPoolVos = costPoolStrategy.onRequestCostPoolVos(costPoolDto);
    if (CollectionUtils.isEmpty(costPoolVos)) {
      return BigDecimal.ZERO;
    }
    //汇总剩余可使用金额
    BigDecimal usableAmount = costPoolVos.stream()
        .map(CostPoolVo::getUsableAmount)
        .reduce(BigDecimal.ZERO, BigDecimal::add);
    return usableAmount;
  }



  /**
   * 找到sku媒体
   *
   * @param orderPreviewVo 订单预览签证官
   */
  private void findSkuMedia(OrderPreviewVo orderPreviewVo) {
    List<OrderDetailPreviewVo> orderDetails = orderPreviewVo.getOrderDetails();
    List<String> productCodeList = orderDetails.stream().map(OrderDetailPreviewVo::getGoodsCode).collect(Collectors.toList());
    // 获取sku的信息
    List<ProductVo> productList = this.productVoService.findDetailsByIdsOrProductCodes(null, productCodeList);
    Map<String,ProductVo> productCodeMap = new HashMap<>(productList.size());
    for (ProductVo productVo : productList) {
      productCodeMap.put(productVo.getProductCode(), productVo);
    }
    for (OrderDetailPreviewVo orderDetail : orderDetails) {
      String goodsCode = orderDetail.getGoodsCode();
      ProductVo productVo = productCodeMap.get(goodsCode);
      if (ObjectUtils.isEmpty(productVo)) {
        continue;
      }
      List<OrderFileVo> pictureFileList = OrderAssist.getFileVos(productVo.getPictureMediaList());
      List<OrderFileVo> videoFileList = OrderAssist.getFileVos(productVo.getVideoMediaList());
      orderDetail.setPictureMediaList(pictureFileList);
      orderDetail.setVideoMediaList(videoFileList);
    }
  }

  @Autowired(required = false)
  private LoginUserService loginUserService;

  @Autowired(required = false)
  private ProductVoService productVoService;

  @Override
  public OrderPreviewVo handlePreviewByCurrentCustomer(OrderPreviewDto order) {
    String customerCode = order.getRelateCode();
    String customerName = order.getRelateName();
    order.setRelateCode(customerCode);
    order.setRelateName(customerName);
    OrderPreviewVo orderPreviewVo = this.handlePreview(order);

    return orderPreviewVo;
  }



  @Autowired(required = false)
  private OrderService orderService;

  @Override
  public OrderPreviewVo handlePreviewByOrderId(String orderId) {
    Validate.notBlank(orderId, "订单ID必须传");
    Order order = this.orderService.findById(orderId);
    OrderPreviewDto orderPreviewDto = this.nebulaToolkitService.copyObjectByBlankList(order, OrderPreviewDto.class, HashSet.class, LinkedList.class);
    Validate.isTrue(!CollectionUtils.isEmpty(orderPreviewDto.getOrderDetails()), "该订单不含商品不能再来一单");
    orderPreviewDto.getOrderDetails().forEach(item -> item.setIsAutoHit(true));
    return this.handlePreview(orderPreviewDto);
  }

  /**
   * 返回结算信息的部分展示数据
   *
   * @param orderPreviewVo
   */
  private void tabulateData(OrderPreviewVo orderPreviewVo) {
    // --订单总金额 = 所有商品行销售价*数量汇总
    BigDecimal totalOrderAmount = BigDecimal.ZERO;
    // --政策优惠金额 = 本单所有优惠金额汇总，商品优惠政策和整单优惠政策都包含
    BigDecimal policyDiscountAmount = BigDecimal.ZERO;
    // --使用货补金额 = 本单所有货补项金额汇总
    BigDecimal totalCompensatedAmount = BigDecimal.ZERO;
    // --使用折扣费用
    BigDecimal totalDiscountAmount = BigDecimal.ZERO;
    List<OrderPayPreviewVo> orderPays = orderPreviewVo.getOrderPays();
    if (!CollectionUtils.isEmpty(orderPays)) {
      for (OrderPayPreviewVo orderPay : orderPays) {
        Boolean itemGroupType = orderPay.getItemGroupType();
        if (!itemGroupType) {
          BigDecimal itemAmount = orderPay.getItemAmount();
          itemAmount = Objects.nonNull(itemAmount) ? itemAmount : BigDecimal.ZERO;
          totalDiscountAmount = totalDiscountAmount.add(itemAmount);
        }
      }
    }
    // --遍历订单明细
    List<OrderDetailPreviewVo> orderDetails = orderPreviewVo.getOrderDetails();
    if (!CollectionUtils.isEmpty(orderDetails)) {
      Map<String, OrderPolicyDetailVo> policyDetailMap = new HashMap<>(orderDetails.size());
      for (OrderDetailPreviewVo orderDetail : orderDetails) {
        // --行上的销售总金额
        BigDecimal salesAmount = orderDetail.getSalesAmount();
        // --订单明细的扣款项目
        // ----订单明细行上总的优惠(所有该扣 货补、折扣、政策)金额
        BigDecimal totalItemAmount = BigDecimal.ZERO;
        // ----订单明细行上总的政策优惠金额
        BigDecimal totalPolicyItemAmount = BigDecimal.ZERO;
        List<OrderDetailPayPreviewVo> orderDetailPays = orderDetail.getOrderDetailPays();
        if (!CollectionUtils.isEmpty(orderDetailPays)) {
          for (OrderDetailPayPreviewVo orderDetailPay : orderDetailPays) {
            Boolean available = orderDetailPay.getAvailable();
            if (ObjectUtils.isEmpty(available)) {
              continue;
            }
            if (available) {
              BigDecimal itemAmount = orderDetailPay.getItemAmount();
              totalItemAmount = totalItemAmount.add(itemAmount);
              // 优惠政策记账模块注册的项目组key
              String itemGroupKey = "salepolicyGroup";
              boolean equals = itemGroupKey.equals(orderDetailPay.getItemGroupKey());
              if (equals) {
                totalPolicyItemAmount = totalPolicyItemAmount.add(itemAmount);
              }
            }
            String originData = orderDetailPay.getOriginData();
            String originDataAlias = orderDetailPay.getOriginDataAlias();
            OrderPolicyDetailVo orderPolicyDetailVo = policyDetailMap.get(originData);
            if (Objects.isNull(orderPolicyDetailVo)) {
              orderPolicyDetailVo = new OrderPolicyDetailVo();
              policyDetailMap.put(originData, orderPolicyDetailVo);
              orderPolicyDetailVo.setOriginDataAlias(originDataAlias);
            }
            String itemType = orderDetail.getItemType();
            if (ItemTypeEnum.NORMAL_GOODS.getDictCode().equals(itemType)) {
              List<String> goodsName = orderPolicyDetailVo.getGoodsName();
              goodsName.add(orderDetail.getGoodsName());
            }
            if (ItemTypeEnum.COMPLIMENTARY_GOODS.getDictCode().equals(itemType)) {
              List<String> goodsName = orderPolicyDetailVo.getComplimentaryGoodsName();
              goodsName.add(orderDetail.getGoodsName());
            }
            List<String> reason = orderPolicyDetailVo.getReason();
            reason.add(orderDetailPay.getReason());
            BigDecimal policyAmount = orderPolicyDetailVo.getItemAmount();
            policyAmount = policyAmount.add(orderDetailPay.getItemAmount());
            orderPolicyDetailVo.setItemAmount(policyAmount);
          }
        }
        // --实际支付金额（组内分摊后）== 销售金额 - 优惠金额 （即，本应该支付金额）
        BigDecimal shouldPaymentAmount = salesAmount.subtract(totalItemAmount);
        orderDetail.setShouldPaymentAmount(shouldPaymentAmount);
        policyDiscountAmount = policyDiscountAmount.add(totalPolicyItemAmount);
        totalOrderAmount = totalOrderAmount.add(salesAmount);
        // 总得优惠详情
        if (!CollectionUtils.isEmpty(policyDetailMap)) {
          List<OrderPolicyDetailVo> orderPolicyDetailList = Lists.newLinkedList();
          for (Map.Entry<String, OrderPolicyDetailVo> entry : policyDetailMap.entrySet()) {
            orderPolicyDetailList.add(entry.getValue());
          }
          orderPreviewVo.setOrderPolicyDetail(orderPolicyDetailList);
        }
        String itemType = orderDetail.getItemType();
        if (itemType.equals(ItemTypeEnum.COMPENSATED_GOODS.getDictCode())) {
          totalCompensatedAmount = totalCompensatedAmount.add(salesAmount);
        }
      }
    }
    orderPreviewVo.setTotalOrderAmount(totalOrderAmount);
    // --本单实际支付 ==  订单总金额（元）-政策优惠金额（元）-使用货补金额-使用折扣费用（元） 备注：这里不管货补和折扣
    BigDecimal actualAmountPaid = totalOrderAmount.subtract(policyDiscountAmount).subtract(totalCompensatedAmount).subtract(totalDiscountAmount);
    // 免费单，应付为零
    String orderType = orderPreviewVo.getOrderType();
    boolean isFree = OrderTypeEnum.FREE.getDictCode().equals(orderType);
    orderPreviewVo.setActualAmountPaid(isFree ? BigDecimal.ZERO : actualAmountPaid);
    orderPreviewVo.setPolicyDiscountAmount(policyDiscountAmount);
    // --订单的总的该扣项目: 这个接口不用管

  }
}
