package com.biz.crm.dms.business.order.local.service.internal;

import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.dms.business.order.common.sdk.enums.OrderCategoryEnum;
import com.biz.crm.dms.business.order.common.sdk.enums.OrderStatusEnum;
import com.biz.crm.dms.business.order.local.entity.Order;
import com.biz.crm.dms.business.order.local.repository.OrderRepository;
import com.biz.crm.dms.business.order.local.service.OrderService;
import com.biz.crm.dms.business.order.local.service.assist.OrderAssist;
import com.biz.crm.dms.business.order.sdk.dto.CustomerOrderPageDto;
import com.biz.crm.dms.business.order.sdk.service.OrderVoService;
import com.biz.crm.dms.business.order.sdk.vo.OrderDetailVo;
import com.biz.crm.dms.business.order.sdk.vo.OrderFileVo;
import com.biz.crm.dms.business.order.sdk.vo.OrderVo;
import com.biz.crm.dms.business.psi.product.sdk.service.productstock.ProductStockVoService;
import com.biz.crm.dms.business.psi.product.sdk.vo.productstock.ProductStockVo;
import com.biz.crm.mdm.business.product.sdk.service.ProductVoService;
import com.biz.crm.mdm.business.product.sdk.vo.ProductVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.math.BigDecimal;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * @author he fan
 */
@Service
public class OrderVoServiceImpl implements OrderVoService {

  @Autowired(required = false)
  private OrderRepository orderRepository;

  @Autowired(required = false)
  private NebulaToolkitService nebulaToolkitService;

  @Override
  public List<OrderVo> findByOrderCodes(List<String> orderCodes) {
    List<Order> detailByOrderCodes = orderRepository.findDetailByOrderCodes(orderCodes);
    List<OrderVo> orderVos = (List<OrderVo>) this.nebulaToolkitService.copyCollectionByBlankList(detailByOrderCodes, Order.class, OrderVo.class, HashSet.class, LinkedList.class);
    return orderVos;
  }

  @Transactional
  @Override
  public OrderVo updateOrderStatusByDeliveryQuantity(String orderCode, BigDecimal deliveryQuantity, Boolean isAdd) {
    Validate.notBlank(orderCode, "根据发货数量修改订单状态时，没有传入订单编码");
    Validate.notNull(isAdd, "根据发货数量修改订单状态时，业务标识没传");
    Boolean isPositive = Objects.nonNull(deliveryQuantity) && deliveryQuantity.compareTo(BigDecimal.ZERO) == 1;
    Validate.isTrue(isPositive, "根据发货数量修改订单状态时，发货数量不能不传，不能为0，为负数");
    Order order = orderRepository.findByOrderCode(orderCode);
    Validate.notNull(order, "根据发货数量修改订单状态时，没有查询到订单信息");
    BigDecimal deliveryQuantityDb = order.getDeliveryQuantity();
    deliveryQuantityDb = Objects.isNull(deliveryQuantityDb) ? BigDecimal.ZERO : deliveryQuantityDb;
    if (isAdd) {
      deliveryQuantity = deliveryQuantityDb.add(deliveryQuantity);
    } else {
      deliveryQuantity = deliveryQuantityDb.subtract(deliveryQuantity);
    }
    int compareToZero = deliveryQuantity.compareTo(BigDecimal.ZERO);
    Validate.isTrue(compareToZero > -1, "根据发货数量修改订单状态时，已发货数量不能为负");
    if (compareToZero == 0) {
      // 回到待发货状态
      this.orderRepository.updateOrderStatusAndDeliveryQuantityById(OrderStatusEnum.WAIT_SHIPPED.getDictCode(), deliveryQuantity, order.getId());
    } else {
      int compareToQuantity = deliveryQuantity.compareTo(order.getQuantityOfCommodity());
      Validate.isTrue(compareToQuantity <= 1, "根据发货数量修改订单状态时，已发货数量不能超过订单商品总量");
      if (compareToQuantity == 0) {
        // 订单发货完成
        this.orderRepository.updateOrderStatusAndDeliveryQuantityById(OrderStatusEnum.COMPLETED.getDictCode(), deliveryQuantity, order.getId());
      } else {
        // 部分发货
        this.orderRepository.updateOrderStatusAndDeliveryQuantityById(OrderStatusEnum.PART_SHIPPED.getDictCode(), deliveryQuantity, order.getId());
      }

    }
    return this.nebulaToolkitService.copyObjectByBlankList(order, OrderVo.class, HashSet.class, LinkedList.class);

  }

  @Autowired(required = false)
  private OrderService orderService;

  @Override
  public Page<OrderVo> findByCustomerOrderPageDto(Pageable pageable, CustomerOrderPageDto customerOrderPageDto) {
    Page<Order> page = orderService.findByCustomerOrderPageDto(pageable, customerOrderPageDto);
    Page<OrderVo> voPage = new Page<>(page.getCurrent(), page.getSize(), page.getTotal());
    List<Order> records = page.getRecords();
    if (CollectionUtils.isEmpty(records)) {
      voPage.setRecords(Lists.newArrayList());
    } else {
      List<OrderVo> orderVos = (List<OrderVo>)this.nebulaToolkitService.copyCollectionByBlankList(records, Order.class, OrderVo.class, HashSet.class, LinkedList.class);
      voPage.setRecords(orderVos);
    }
    return voPage;
  }
  @Autowired(required = false)
  private ProductStockVoService productStockVoService;

  @Autowired(required = false)
  private ProductVoService productVoService;

  @Override
  public OrderVo findById(String id) {
    Order order = orderService.findById(id);
    OrderVo orderVo = this.nebulaToolkitService.copyObjectByBlankList(order, OrderVo.class, HashSet.class, LinkedList.class);
    String orderCategory = orderVo.getOrderCategory();
    boolean isMaterial = OrderCategoryEnum.MATERIAL_ORDER.getDictCode().equals(orderCategory);
    if (isMaterial) {
      // 查询物料库存
      String warehouseCode = orderVo.getWarehouseCode();
      List<OrderDetailVo> orderDetails = orderVo.getOrderDetails();
      Set<String> goodsCodes = orderDetails.stream().map(OrderDetailVo::getGoodsCode).collect(Collectors.toSet());
      List<ProductStockVo> productStockVos = productStockVoService.findByWarehouseAndProducts(warehouseCode, goodsCodes);
      Map<String, BigDecimal> skuToAvailableMap = new HashMap<>(productStockVos.size());
      for (ProductStockVo stockVo : productStockVos) {
        skuToAvailableMap.put(stockVo.getProductCode(), stockVo.getAvailableStock());
      }
      for (OrderDetailVo orderDetail : orderDetails) {
        orderDetail.setAvailableStock(skuToAvailableMap.get(orderDetail.getGoodsCode()));
      }
    }
    // 获取sku的信息
    List<OrderDetailVo> orderDetails = orderVo.getOrderDetails();
    if (!CollectionUtils.isEmpty(orderDetails)) {
      List<String> productCodeList = orderDetails.stream().map(OrderDetailVo::getGoodsCode).collect(Collectors.toList());
      List<ProductVo> productList = this.productVoService.findDetailsByIdsOrProductCodes(null, productCodeList);
      Map<String,ProductVo> productCodeMap = new HashMap<>(productList.size());
      for (ProductVo productVo : productList) {
        productCodeMap.put(productVo.getProductCode(), productVo);
      }
      for (OrderDetailVo orderDetail : orderDetails) {
        String goodsCode = orderDetail.getGoodsCode();
        ProductVo productVo = productCodeMap.get(goodsCode);
        if (ObjectUtils.isEmpty(productVo)) {
          continue;
        }
        List<OrderFileVo> pictureFileList = OrderAssist.getFileVos(productVo.getPictureMediaList());
        List<OrderFileVo> videoFileList = OrderAssist.getFileVos(productVo.getVideoMediaList());
        orderDetail.setPictureMediaList(pictureFileList);
        orderDetail.setVideoMediaList(videoFileList);
      }
    }
    return orderVo;
  }

  @Override
  public void updateOrderStatusAndIsShowByOrderCode(String orderCode, OrderStatusEnum orderStatusEnum, Boolean isShow) {
    /**
     * 订单是否存在
     * 订单现在的状态是否允许改变
     * 修改订单状态
     */
    Order order = orderRepository.findByOrderCode(orderCode);
    Validate.notNull(order, "订单信息在系统中不存在");
    Set<String> set = Sets.newHashSet(
        OrderStatusEnum.REJECTED.getDictCode(),
        OrderStatusEnum.CLOSE.getDictCode(),
        OrderStatusEnum.CANCELED.getDictCode(),
        OrderStatusEnum.COMPLETED.getDictCode()
    );
    Validate.isTrue(!set.contains(order.getOrderStatus()), "当前订单状态不允许修改");
    orderRepository.updateOrderStatusAndIsShowById(orderStatusEnum.getDictCode(), order.getId(), isShow);
  }

  @Override
  public OrderVo findByOrderCode(String orderCode) {
    Order order = orderRepository.findByOrderCode(orderCode);
    if (ObjectUtils.isEmpty(order)) {
      return null;
    }
    return this.findById(order.getId());
  }
}
