package com.biz.crm.dms.business.order.local.service.internal;


import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.sdk.constant.LoginUserConstant;
import com.biz.crm.business.common.sdk.enums.BooleanEnum;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.model.AbstractCrmUserIdentity;
import com.biz.crm.business.common.sdk.service.LoginUserService;
import com.biz.crm.dms.business.order.local.entity.HistoryViewQueryConfig;
import com.biz.crm.dms.business.order.local.entity.PurchaseHistory;
import com.biz.crm.dms.business.order.local.repository.PurchaseHistoryRepository;
import com.biz.crm.dms.business.order.local.service.HistoryViewQueryConfigService;
import com.biz.crm.dms.business.order.local.service.PurchaseHistoryService;
import com.biz.crm.dms.business.order.sdk.constant.PurchaseHistoryConstant;
import com.biz.crm.dms.business.order.sdk.dto.PurchaseHistoryPageDto;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.util.Date;
import java.util.List;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * 购买历史(PurchaseHistory)表服务实现类
 *
 * @author makejava
 * @since 2022-04-07 17:14:33
 */
@Service("purchaseHistoryService")
public class PurchaseHistoryServiceImpl implements PurchaseHistoryService {

  @Autowired(required = false)
  private PurchaseHistoryRepository purchaseHistoryRepository;

  @Autowired(required = false)
  private LoginUserService loginUserService;

  @Autowired(required = false)
  private HistoryViewQueryConfigService historyViewQueryConfigService;

  /**
   * 分页查询数据
   *
   * @param pageable        分页对象
   * @param purchaseHistory 实体对象
   * @return
   */
  @Override
  public Page<PurchaseHistory> findByConditions(Pageable pageable, PurchaseHistoryPageDto purchaseHistory) {
    String customerCode = purchaseHistory.getRelateCode();
    ObjectUtils.defaultIfNull(pageable, PageRequest.of(0, 50));
    if (Objects.isNull(purchaseHistory)) {
      purchaseHistory = new PurchaseHistoryPageDto();
    }
    HistoryViewQueryConfig config = historyViewQueryConfigService.findByCurrentCustomerCode(customerCode);
    purchaseHistory.setTimes(ObjectUtils.isNotEmpty(config) ? config.getTimes() : PurchaseHistoryConstant.defaultTimes);
    purchaseHistory.setRelateCode(customerCode);
    purchaseHistory.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
    purchaseHistory.setAppCode(TenantUtils.getAppCode());
    //只展示可用
    purchaseHistory.setIsUsable(BooleanEnum.TRUE.getCapital());
    return this.purchaseHistoryRepository.findByConditions(pageable, purchaseHistory);
  }

  /**
   * 通过主键查询单条数据
   *
   * @param id 主键
   * @return 单条数据
   */
  @Override
  public PurchaseHistory findById(String id) {
    if (StringUtils.isBlank(id)) {
      return null;
    }
    return this.purchaseHistoryRepository.getById(id);
  }

  /**
   * 新增数据
   *
   * @param purchaseHistory 实体对象
   * @return 新增结果
   */
  @Transactional
  @Override
  public PurchaseHistory create(PurchaseHistory purchaseHistory) {
    this.createValidate(purchaseHistory);
    this.purchaseHistoryRepository.saveOrUpdate(purchaseHistory);
    return purchaseHistory;
  }

  /**
   * 修改新据
   *
   * @param purchaseHistory 实体对象
   * @return 修改结果
   */
  @Transactional
  @Override
  public PurchaseHistory update(PurchaseHistory purchaseHistory) {
    this.updateValidate(purchaseHistory);
    this.purchaseHistoryRepository.saveOrUpdate(purchaseHistory);
    return purchaseHistory;
  }

  /**
   * 删除数据
   *
   * @param idList 主键结合
   * @return 删除结果
   */
  @Transactional
  @Override
  public void delete(List<String> idList) {
    Validate.isTrue(!CollectionUtils.isEmpty(idList), "删除数据时，主键集合不能为空！");
    this.purchaseHistoryRepository.deleteByIdIn(idList);
  }

  /**
   * 通过商品编码查询
   * @param code
   */
  @Override
  public List<PurchaseHistory> findByGoodsCode(String code) {
    if (StringUtils.isBlank(code)) {
      return null;
    }
    return this.purchaseHistoryRepository.findByGoodsCode(code);
  }

  /**
   * 通过商品编码和可用状态，改变可用状态
   * @param goodsCode
   * @param isUsable
   */
  @Override
  public void updateUsableStatus(String goodsCode, String isUsable) {
    Validate.isTrue(StringUtils.isNoneBlank(goodsCode,isUsable),"缺失商品编码与状态");
    this.purchaseHistoryRepository.updateUsableStatus(goodsCode,isUsable);
  }

  @SuppressWarnings("unchecked")
  @Transactional
  @Override
  public List<PurchaseHistory> handleIncrement(List<PurchaseHistory> purchaseHistory) {
    /**
     * 购买了，就用次方法，记录一次。
     * - 入参中存在第一次购买的商品
     * - 已经有数据记录的购买次数+1
     * - 需要在数据库里保证幂等( 租户+货物编码 唯一）
     */
    Validate.isTrue(!CollectionUtils.isEmpty(purchaseHistory), "保存商品购买记录时，商品信息不能为空");
    List<String> goodsCodes = Lists.newArrayListWithCapacity(purchaseHistory.size());
    for (PurchaseHistory history : purchaseHistory) {
      goodsCodes.add(history.getGoodsCode());
    }
    // 查询到的数据跟入参数据不一定一样数目。
    String relateCode = purchaseHistory.get(0).getRelateCode();
    List<PurchaseHistory> dbList = this.purchaseHistoryRepository.findByGoodsCodeInAndRelateCode(goodsCodes, relateCode);
    if (CollectionUtils.isEmpty(dbList)) {
      Set<PurchaseHistory> purchaseHistorySet = Sets.newHashSet(purchaseHistory);
      // 全部插入
      for (PurchaseHistory history : purchaseHistorySet) {
        this.createValidate(history);
      }
      this.purchaseHistoryRepository.saveBatch(purchaseHistorySet);
      return purchaseHistory;
    }
    // 修改和插入分两边操作
    Set<PurchaseHistory> purchaseHistorySet = Sets.newHashSet(purchaseHistory);
    List<PurchaseHistory> newList = (List<PurchaseHistory>) org.apache.commons.collections.CollectionUtils.subtract(purchaseHistorySet, dbList);
    // 没有id的数据全部插入
    if (!CollectionUtils.isEmpty(newList)) {
      for (PurchaseHistory history : newList) {
        this.createValidate(history);
      }
      this.purchaseHistoryRepository.saveBatch(newList);
    }
    // 有id的数据全部增量修改
    List<String> ids = dbList.stream().map(PurchaseHistory::getId).collect(Collectors.toList());
    this.purchaseHistoryRepository.updateTimesIncrementByIdIn(ids);
    // 把最新的数据返回去
    return this.purchaseHistoryRepository.findByGoodsCodeInAndRelateCode(goodsCodes, relateCode);
  }

  /**
   *  通过商品编码更新商品基本信息
   * @param purchaseHistory
   */
  @Override
  public void updateByProductCode(PurchaseHistory purchaseHistory) {
    Validate.notNull(purchaseHistory,"更新时，数据不能为空");
    Validate.notBlank(purchaseHistory.getGoodsCode(), "新增时，货物编码不能为空！");
    this.purchaseHistoryRepository.updateByProductCode(purchaseHistory);
  }

  /**
   * 创建验证
   *
   * @param purchaseHistory
   */
  private void createValidate(PurchaseHistory purchaseHistory) {
    Validate.notNull(purchaseHistory, "新增时，对象信息不能为空！");
    purchaseHistory.setId(null);
    purchaseHistory.setTenantCode(TenantUtils.getTenantCode());
    Validate.notBlank(purchaseHistory.getGoodsCode(), "新增时，货物编码不能为空！");
    Validate.notBlank(purchaseHistory.getGoodsName(), "新增时，货物名称不能为空！");
    Validate.notBlank(purchaseHistory.getRelateCode(), "新增时，关联编码不能为空！");
    Validate.notBlank(purchaseHistory.getSpec(), "新增时，规格不能为空！");
    Validate.notBlank(purchaseHistory.getUnite(), "新增时，销售单位不能为空！");
    purchaseHistory.setTimes(Integer.valueOf(1));
    purchaseHistory.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
    purchaseHistory.setCreateTime(new Date());
    AbstractCrmUserIdentity loginUser = loginUserService.getAbstractLoginUser();
    purchaseHistory.setCreateAccount(loginUser.getAccount());
    purchaseHistory.setCreateName(loginUser.getUsername());
    purchaseHistory.setAppCode(TenantUtils.getAppCode());
    Validate.isTrue(purchaseHistory.getGoodsCode().length() <= 64, "货物编码 长度不能超过64");
    Validate.isTrue(purchaseHistory.getGoodsName().length() <= 100, "货物名称 长度不能超过100");
    Validate.isTrue(purchaseHistory.getSpec().length() <= 64, "规格 长度不能超过64");
    Validate.isTrue(purchaseHistory.getUnite().length() <= 64, "销售单位 长度不能超过64");
    Validate.isTrue(purchaseHistory.getRelateCode().length() <= 255, "关联编码 长度不能超过255");
  }

  /**
   * 修改验证
   *
   * @param purchaseHistory
   */
  private void updateValidate(PurchaseHistory purchaseHistory) {
    Validate.notNull(purchaseHistory, "修改时，对象信息不能为空！");
    Validate.notBlank(purchaseHistory.getId(), "新增数据时，不能为空！");
    Validate.notBlank(purchaseHistory.getTenantCode(), "新增数据时，租户编号不能为空！");
    Validate.notBlank(purchaseHistory.getAppCode(), "新增数据时，品牌商租户编号不能为空！");

  }
}

