package com.biz.crm.dms.business.order.minimum.local.service.strategy;

import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.dms.business.order.common.sdk.dto.OrderDetailDto;
import com.biz.crm.dms.business.order.common.sdk.dto.OrderDto;
import com.biz.crm.dms.business.order.common.sdk.dto.OrderTabulateDto;
import com.biz.crm.dms.business.order.common.sdk.enums.OrderTypeEnum;
import com.biz.crm.dms.business.order.common.sdk.enums.OrderValidationProcessNodeEnum;
import com.biz.crm.dms.business.order.feerate.fee.statistics.sdk.register.OrderStatisticsElementRegister;
import com.biz.crm.dms.business.order.feerate.fee.statistics.sdk.register.OrderStatisticsService;
import com.biz.crm.dms.business.order.minimum.local.entity.DmsOrderMinimum;
import com.biz.crm.dms.business.order.minimum.local.service.DmsOrderMinimumService;
import com.biz.crm.dms.business.order.minimum.local.service.MinimumValidateMountVoService;
import com.biz.crm.dms.business.order.minimum.sdk.ObjectTypeEnum;
import com.biz.crm.dms.business.order.minimum.sdk.vo.MinimumValidateMountVo;
import com.biz.crm.dms.business.order.verification.sdk.model.OrderVerificationContext;
import com.biz.crm.dms.business.order.verification.sdk.strategy.OrderVerificationStrategy;
import com.biz.crm.mdm.business.customer.sdk.service.CustomerVoService;
import com.biz.crm.mdm.business.customer.sdk.vo.CustomerVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * 起订量校验
 *
 * @Author: zengxingwang
 * @Date: 2021/12/29 17:58
 */
@Service
public class MinimumOrderVerificationStrategyImpl implements OrderVerificationStrategy {

  @Autowired(required = false)
  private List<OrderStatisticsElementRegister> orderStatisticsElementRegisterList;
  @Autowired(required = false)
  private DmsOrderMinimumService dmsOrderMinimumService;
  @Autowired(required = false)
  private OrderStatisticsService orderStatisticsService;
  @Autowired(required = false)
  private MinimumValidateMountVoService minimumValidateMountVoService;
  @Autowired(required = false)
  private CustomerVoService customerVoService;

  @Override
  public List<OrderValidationProcessNodeEnum> processNodeEnums() {
    return Lists.newArrayList(OrderValidationProcessNodeEnum.TWO);
  }

  @Override
  public List<OrderTypeEnum> orderTypeEnums() {
    return Lists.newArrayList(OrderTypeEnum.values());
  }

  @Override
  public Boolean required() {
    return Boolean.FALSE;
  }

  @Override
  public String title() {
    return "起订量校验";
  }

  @Autowired(required = false)
  private NebulaToolkitService nebulaToolkitService;

  @Override
  public void execute(OrderVerificationContext context) {
    OrderTabulateDto orderTabulateDto = context.getOrderTabulateDto();
    OrderDto orderDtoIn = context.getOrderDto();
    if (ObjectUtils.isEmpty(orderTabulateDto)) {
      this.execute(orderDtoIn);
    } else {
      OrderDto orderDtoOut = nebulaToolkitService.copyObjectByBlankList(orderDtoIn, OrderDto.class, HashSet.class, LinkedList.class);
      OrderDto orderDtoTemp = nebulaToolkitService.copyObjectByBlankList(orderTabulateDto, OrderDto.class, HashSet.class, LinkedList.class);
      orderDtoOut.setOrderDetails(orderDtoTemp.getOrderDetails());
      this.execute(orderDtoOut);
    }
  }

  @Override
  public void execute(OrderDto orderDto) {
    Set<String> codes = new HashSet<>();
    for (OrderDetailDto vo : orderDto.getOrderDetails()) {
      codes.add(vo.getGoodsCode());
    }
    CustomerVo customerVo = customerVoService
        .findDetailsByIdOrCode(null, orderDto.getRelateCode());
    Validate.notNull(customerVo, "起订量查询，客户不存在！");
    //获取范围内起订量配置（包含优先级的重复）
    Map<String, DmsOrderMinimum> minimumMap = new HashMap<>();
    List<DmsOrderMinimum> minimums = dmsOrderMinimumService.findByCustomerCodeOrOrgCodeAndStatusInProductCodes(orderDto.getRelateCode(), customerVo.getOrgCode(), EnableStatusEnum.ENABLE.getCode(), codes);
    //优先级过滤
    for (DmsOrderMinimum minimum : minimums) {
      if (minimumMap.containsKey(minimum.getProductCode())) {
        if (ObjectTypeEnum.CUSTOMER.getDictCode().equals(minimum.getType())) {
          minimumMap.put(minimum.getProductCode(), minimum);
        }
      } else {
        minimumMap.put(minimum.getProductCode(), minimum);
      }
    }

    List<DmsOrderMinimum> result = new ArrayList<>();
    minimumMap.forEach((key, value) -> {
      result.add(value);
    });

    if (CollectionUtils.isEmpty(result)) {
      return;
    }
    this.handleMinimumValidate(result, orderDto);
  }

  /**
   * 验证数量
   *
   * @param result
   * @param orderDto
   */
  private void handleMinimumValidate(List<DmsOrderMinimum> result, OrderDto orderDto) {
    Map<String, List<OrderDetailDto>> productMap = orderDto.getOrderDetails().stream()
        .filter(a -> StringUtils.isNoneBlank(a.getGoodsCode()))
        .collect(Collectors.groupingBy(OrderDetailDto::getGoodsCode));

    for (DmsOrderMinimum minimum : result) {
      List<OrderDetailDto> products = productMap.get(minimum.getProductCode());
      if (CollectionUtils.isEmpty(products)) {
        continue;
      }
      Set<String> checkTypes = Sets.newHashSet(StringUtils.split(minimum.getCheckType(), "+"));
      List<BigDecimal> feeQuantity = this.getFeeQuantity(checkTypes, products);
      //统计值
      BigDecimal quantity = orderStatisticsService.getFeeCount(feeQuantity);

      List<MinimumValidateMountVo> all = minimumValidateMountVoService.findAll();
      String name = "";
      if (!CollectionUtils.isEmpty(all)) {
        for (MinimumValidateMountVo vo : all) {
          if (vo.getKey().equals(minimum.getCheckType())) {
            name = vo.getName();
          }
        }
      }
      Validate.isTrue(minimum.getMinimum().compareTo(quantity) < 1, "[%s]商品[%s]起订量不够，请检查！", name, minimum.getProductName());
    }
  }

  /**
   * 订单行编码获取行数量
   *
   * @param checkTypes
   * @param products
   * @date
   */
  private List<BigDecimal> getFeeQuantity(Set<String> checkTypes, List<OrderDetailDto> products) {
    Validate.isTrue(!CollectionUtils.isEmpty(checkTypes), "传入订单统计范围要素为空");
    List<BigDecimal> feeQuantity = new ArrayList<>();
    for (String checkType : checkTypes) {
      orderStatisticsElementRegisterList.forEach(orderStatisticsElementRegister -> {
        if (orderStatisticsElementRegister.getOrderStatisticsElementCode().equals(checkType)) {

          Map<String, List<OrderDetailDto>> map = products.stream()
              .filter(a -> StringUtils.isNoneBlank(a.getItemType()))
              .collect(Collectors.groupingBy(OrderDetailDto::getItemType));
          List<OrderDetailDto> checkTypeProducts = map.get(checkType);
          if (!CollectionUtils.isEmpty(checkTypeProducts)) {
            BigDecimal quantity = orderStatisticsElementRegister.getFeeQuantity(checkTypeProducts);
            feeQuantity.add(quantity);
          }
        }
      });
    }
    return feeQuantity;
  }
}
