package com.biz.crm.dms.business.order.verification.local.strategy.internal;

import com.biz.crm.business.common.sdk.enums.BooleanEnum;
import com.biz.crm.dms.business.costpool.sdk.dto.CostPoolDto;
import com.biz.crm.dms.business.costpool.sdk.strategy.CostPoolRegister;
import com.biz.crm.dms.business.costpool.sdk.strategy.CostPoolStrategy;
import com.biz.crm.dms.business.costpool.sdk.vo.CostPoolReplenishmentVerifyProductVo;
import com.biz.crm.dms.business.costpool.sdk.vo.CostPoolReplenishmentVerifyVo;
import com.biz.crm.dms.business.costpool.sdk.vo.CostPoolVo;
import com.biz.crm.dms.business.order.common.sdk.dto.OrderDetailDto;
import com.biz.crm.dms.business.order.common.sdk.dto.OrderDto;
import com.biz.crm.dms.business.order.common.sdk.enums.ItemTypeEnum;
import com.biz.crm.dms.business.order.common.sdk.enums.OrderTypeEnum;
import com.biz.crm.dms.business.order.common.sdk.enums.OrderValidationProcessNodeEnum;
import com.biz.crm.dms.business.order.verification.sdk.strategy.OrderMustVerificationStrategy;
import com.biz.crm.mdm.business.product.sdk.service.ProductVoService;
import com.biz.crm.mdm.business.product.sdk.vo.ProductVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.google.common.collect.Lists;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * @author HanJiaJun
 * @describe:添加货补商品校验：可选商品为经销商关联的货补池中的且为上架状态的商品
 * @createTime 2022年03月03日 10:47:00
 */
@Service
public class CostPoolReplenishmentProductVerificationStrategyImpl implements OrderMustVerificationStrategy {

  /**
   * 错误信息
   */
  private static final String ERROR_MESSAGE_SHELF = "商品已下架";
  private static final String ERROR_MESSAGE_EXIST = "产品编码不属于当前客户的货补池";

  @Autowired(required = false)
  @Qualifier("costPoolReplenishmentStrategyImpl")
  private CostPoolStrategy costPoolStrategy;

  @Autowired(required = false)
  @Qualifier("CostPoolReplenishmentRegisterImpl")
  private CostPoolRegister costPoolRegister;

  @Autowired(required = false)
  private NebulaToolkitService nebulaToolkitService;

  @Autowired(required = false)
  private ProductVoService productVoService;

  @Override
  public List<OrderValidationProcessNodeEnum> processNodeEnums() {
    return Lists.newArrayList(OrderValidationProcessNodeEnum.ONE);
  }

  @Override
  public List<OrderTypeEnum> orderTypeEnums() {
    return Lists.newArrayList(OrderTypeEnum.values());
  }

  @Override
  public Boolean required() {
    return true;
  }

  @Override
  public String title() {
    return "新增货补商品校验";
  }

  @Override
  public void execute(OrderDto orderContextDto) {
    Validate.notNull(orderContextDto, "订单上下文信息缺失");
    String customerCode = orderContextDto.getRelateCode();
    Validate.notBlank(customerCode, "订单客户编码不能为空");
    //获取明细
    List<OrderDetailDto> detailList = orderContextDto.getOrderDetails();
    //筛选货补商品
    List<OrderDetailDto> compensatedGoods = detailList.stream()
        .filter(s -> s.getItemType().equals(ItemTypeEnum.COMPENSATED_GOODS.getDictCode()))
        .collect(Collectors.toList());

    List<String> poolCodes = compensatedGoods.stream()
        .filter(s -> StringUtils.isNotBlank(s.getItemCode()))
        .map(OrderDetailDto::getItemCode).collect(Collectors.toList());

    if (CollectionUtils.isEmpty(compensatedGoods)) {
      return;
    }
    Validate.isTrue(compensatedGoods.size() == poolCodes.size(), "存在货补商品没有传入货补池编号");
    CostPoolDto costPoolDto = new CostPoolDto();
    costPoolDto.setPoolCodeList(poolCodes);
    costPoolDto.setCustomerCode(customerCode);
    costPoolDto.setPoolType(this.costPoolRegister.getKey());
    //获得货补池信息
    List<CostPoolVo> costPoolVoList = this.costPoolStrategy.onRequestCostPoolVos(costPoolDto);
    Validate.isTrue(!CollectionUtils.isEmpty(costPoolVoList), "没有获取到货补池信息");
    //复制子类集合
    List<CostPoolReplenishmentVerifyVo> costPoolReplenishmentVerifyVos = new ArrayList<>();
    for (CostPoolVo vo : costPoolVoList) {
      CostPoolReplenishmentVerifyVo costPoolReplenishmentProduct = new CostPoolReplenishmentVerifyVo();
      costPoolReplenishmentProduct = this.nebulaToolkitService
          .copyObjectByWhiteList(vo, CostPoolReplenishmentVerifyVo.class
              , HashSet.class, ArrayList.class
              , "costPoolReplenishmentProduct");
      costPoolReplenishmentVerifyVos.add(costPoolReplenishmentProduct);
    }
    //与产品层级绑定的货补池的编码集合
    List<String> noProductPoolCodes = new ArrayList<>();
    //与产品绑定的货补池的产品编码集合
    List<String> productVoCodes = new ArrayList<>();
    for (CostPoolReplenishmentVerifyVo vo : costPoolReplenishmentVerifyVos) {
      List<CostPoolReplenishmentVerifyProductVo> product = vo.getCostPoolReplenishmentProduct();
      if (CollectionUtils.isEmpty(product)) {
        //如果货补池商品集合是空的则代表当前货补池与产品层级绑定了
        noProductPoolCodes.add(vo.getPoolCode());
      } else {
        productVoCodes.addAll(product.stream().map(CostPoolReplenishmentVerifyProductVo::getGoodsProductCode).collect(Collectors.toList()));
      }
    }
    List<String> goodsCodes = new ArrayList<>();
    //绑定产品层级
    List<OrderDetailDto> levelList = new ArrayList<>();
    //绑定产品
    List<OrderDetailDto> productList = new ArrayList<>();
    if (CollectionUtils.isEmpty(noProductPoolCodes)) {
      goodsCodes = compensatedGoods.stream().map(OrderDetailDto::getGoodsCode).collect(Collectors.toList());
    } else {
      //只用明细行上的货补商品
      Map<String, List<OrderDetailDto>> map = compensatedGoods.stream().collect(Collectors.groupingBy(OrderDetailDto::getItemCode));
      for (String code : noProductPoolCodes) {
        List<OrderDetailDto> orderDetailDtos = map.get(code);
        levelList.addAll(orderDetailDtos);
      }
      noProductPoolCodes.forEach(e->{
        map.remove(e);
      });
      map.forEach((k, v) -> {
        productList.addAll(v);
      });
      if (!CollectionUtils.isEmpty(productList)) {
        goodsCodes = productList.stream().map(OrderDetailDto::getGoodsCode).collect(Collectors.toList());
      }
    }
    /** 校验商品商品是否属于当前客户货补池商品 **/
    Map<String, List<String>> errorMap = new HashMap<>();
    //创建副本
    List<String> copyFirstProCodes = new ArrayList<>();
    copyFirstProCodes.addAll(goodsCodes);
    //校验
    List<String> validateList = Lists.newArrayList();
    if (!CollectionUtils.isEmpty(goodsCodes)) {
      for (String goodsCode : goodsCodes) {
        if (!productVoCodes.contains(goodsCode)) {
          validateList.add(goodsCode);
          //移除
          copyFirstProCodes.remove(goodsCode);
        }
      }
    }
    errorMap.put(ERROR_MESSAGE_EXIST, validateList);

    /** 校验商品商品是否已下架 **/
    //
    if (!CollectionUtils.isEmpty(levelList)){
      copyFirstProCodes.addAll(levelList.stream().map(OrderDetailDto::getGoodsCode).collect(Collectors.toList()));
    }
    List<ProductVo> voLists = productVoService
        .findDetailsByIdsOrProductCodes(new ArrayList<>(0), copyFirstProCodes);
    Validate.isTrue(!CollectionUtils.isEmpty(voLists), "没有获取到货补商品信息");
    //筛选 保留上架商品
    Set<String> codes = voLists.stream()
        .filter(o -> o.getIsShelf().equals(BooleanEnum.TRUE.getCapital()))
        .map(ProductVo::getProductCode)
        .collect(Collectors.toSet());
    //创建副本
    Set<String> copySecondProCodes = new HashSet<>();
    copySecondProCodes.addAll(copyFirstProCodes);
    List<String> validateShellList = Lists.newArrayList();
    for (String copyFirstProCode : copyFirstProCodes) {
      //筛选下架
      if (!codes.contains(copyFirstProCode)) {
        validateShellList.add(copyFirstProCode);
        //移除
        copySecondProCodes.remove(copyFirstProCode);
      }
    }
    errorMap.put(ERROR_MESSAGE_SHELF, validateShellList);
    /**
     *  注：
     *  errorMap为校验未通过的数据，K 是错误Msg ，V 是商品编码集合
     *  copySecondProCodes是校验成功的商品编码集合。
     */
    if (!(CollectionUtils.isEmpty(errorMap.get(ERROR_MESSAGE_SHELF))
        && CollectionUtils.isEmpty(errorMap.get(ERROR_MESSAGE_EXIST)))) {
      String errorMessage = "";
      for (Map.Entry<String, List<String>> entry : errorMap.entrySet()) {
        String key = entry.getKey();
        List<String> value = entry.getValue();
        errorMessage = StringUtils.join(errorMessage, key);
        String join = StringUtils.join(value, "、");
        errorMessage = StringUtils.join(errorMessage, join, "\n");
      }
      Validate.isTrue(CollectionUtils.isEmpty(errorMap), errorMessage);
    }

  }
}
