package com.biz.crm.dms.business.order.verification.local.strategy.internal;

import com.biz.crm.dms.business.order.common.sdk.dto.OrderDetailDto;
import com.biz.crm.dms.business.order.common.sdk.dto.OrderDto;
import com.biz.crm.dms.business.order.common.sdk.enums.OrderCategoryEnum;
import com.biz.crm.dms.business.order.common.sdk.enums.OrderTypeEnum;
import com.biz.crm.dms.business.order.common.sdk.enums.OrderValidationProcessNodeEnum;
import com.biz.crm.dms.business.order.verification.sdk.model.OrderVerificationContext;
import com.biz.crm.dms.business.order.verification.sdk.strategy.OrderMustVerificationStrategy;
import com.biz.crm.mdm.business.price.sdk.constant.PriceConstant;
import com.biz.crm.mdm.business.price.sdk.dto.FindPriceDto;
import com.biz.crm.mdm.business.price.sdk.enums.FindPriceUserTypeEnum;
import com.biz.crm.mdm.business.price.sdk.service.PriceModelVoService;
import com.biz.crm.mdm.business.price.sdk.vo.PriceModelVo;
import com.google.common.collect.Lists;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import java.math.BigDecimal;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * 最新价格验证策略
 *
 * @author hefan
 * @date 2022/04/11
 */
@Service
public class LatestPriceVerificationStrategyImpl implements OrderMustVerificationStrategy {

  @Autowired(required = false)
  private PriceModelVoService priceModelVoService;

  @Override
  public List<OrderValidationProcessNodeEnum> processNodeEnums() {
    return Lists.newArrayList(OrderValidationProcessNodeEnum.TEN);
  }

  @Override
  public List<OrderTypeEnum> orderTypeEnums() {
    return Lists.newArrayList(OrderTypeEnum.values());
  }

  @Override
  public Boolean required() {
    return Boolean.TRUE;
  }

  @Override
  public String title() {
    return "最新价格验证";
  }

  @Override
  public void execute(OrderVerificationContext context) {
    /**
     * 校验传参中的价格与数据库内的价格是否一致
     * -- 查询数据库里的价格
     * -- 与入参中的价格比较
     */
    OrderDto orderDto = context.getOrderDto();
    // - 不处理物料订单
    String orderCategory = orderDto.getOrderCategory();
    if (orderCategory.equals(OrderCategoryEnum.MATERIAL_ORDER.getDictCode())) {
      return;
    }
    List<OrderDetailDto> orderDetails = orderDto.getOrderDetails();
    Set<String> productCodeSet = orderDetails.stream()
        .filter(a -> StringUtils.isNotBlank(a.getGoodsCode()))
        .map(OrderDetailDto::getGoodsCode)
        .collect(Collectors.toSet());
    // 获取单价 com.biz.crm.mdm.business.price.sdk.service.PriceModelVoService#findPrice
    FindPriceDto findPriceDto = new FindPriceDto();
    findPriceDto.setUserType(FindPriceUserTypeEnum.CUSTOMER.getDictCode());
    findPriceDto.setUserCode(orderDto.getRelateCode());
    findPriceDto.setProductCodeSet(productCodeSet);
    Map<String, PriceModelVo> priceMap = this.priceModelVoService.findPrice(findPriceDto);
    Validate.isTrue(!CollectionUtils.isEmpty(priceMap), "订单中所有商品没有查询到价格");
    // 过滤价格不相等的商品SKU
    List<OrderDetailDto> errorList = orderDetails.stream()
        .filter((OrderDetailDto dto) -> {
          String goodsCode = dto.getGoodsCode();
          PriceModelVo priceModelVo = priceMap.get(goodsCode);
          Validate.notNull(priceModelVo, StringUtils.join("没有查询到商品价格:", dto.getGoodsCode()));
          BigDecimal price = priceModelVo.getPrice();
          Validate.notNull(price, StringUtils.join("没有查询到商品价格:", dto.getGoodsCode()));
          BigDecimal presetUnitPrice = dto.getPresetUnitPrice();
          Validate.notNull(presetUnitPrice, "没有入参价格：", dto.getGoodsCode());
          boolean equal = presetUnitPrice.compareTo(price) == 0;
          return !equal;
        })
        .collect(Collectors.toList());
    if (!CollectionUtils.isEmpty(errorList)) {
      List<String> goodsNames = errorList.stream().map(OrderDetailDto::getGoodsName).collect(Collectors.toList());
      String join = StringUtils.join(goodsNames, "、");
      String msg = StringUtils.join("订单商品中：", join, "，价格不是最新价格");
      throw new IllegalArgumentException(msg);
    }
  }
}
