package com.biz.crm.dms.business.order.verification.local.service.internal;

import com.biz.crm.dms.business.order.common.sdk.enums.OrderTypeEnum;
import com.biz.crm.dms.business.order.common.sdk.enums.OrderValidationProcessNodeEnum;
import com.biz.crm.dms.business.order.verification.sdk.exception.OrderVerificationException;
import com.biz.crm.dms.business.order.verification.sdk.model.OrderVerificationContext;
import com.biz.crm.dms.business.order.verification.sdk.service.OrderVerificationService;
import com.biz.crm.dms.business.order.verification.sdk.strategy.OrderMustVerificationStrategy;
import com.biz.crm.dms.business.order.verification.sdk.strategy.OrderVerificationStrategy;
import com.biz.crm.dms.business.order.verification.sdk.vo.OrderValidateDetailVo;
import com.google.common.collect.Lists;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.ApplicationContext;
import org.springframework.stereotype.Service;

import java.util.List;
import java.util.Map;
import java.util.Objects;

/**
 * 执行订单校验统一接口实现
 *
 * @author pengxi
 * @date 2021/12/27
 */
@Slf4j
@Service
public class OrderVerificationServiceImpl implements OrderVerificationService {

  @Autowired(required = false)
  private ApplicationContext applicationContext;

  @Override
  public void execute(OrderVerificationContext context) {
    if (context == null) {
      log.warn("订单上下文VO不能为空");
      return;
    }
    if (context.getOrderDto() == null) {
      log.warn("订单原始报文不能为空");
      return;
    }
    if (Objects.isNull(context.getOrderConfigVo()) || CollectionUtils.isEmpty(context.getOrderConfigVo().getVerificationVoList())) {
      log.warn("订单配置中没有需要执行校验逻辑的策略设置");
      return;
    }
    OrderValidationProcessNodeEnum processNodeEnum = context.getProcessNodeEnum();
    if (processNodeEnum == null) {
      log.warn("订单校验流程节点未定义");
      return;
    }

    // 系统设定订单必校验项
    Map<String, OrderMustVerificationStrategy> orderMustVerificationStrategyMap = applicationContext.getBeansOfType(OrderMustVerificationStrategy.class);
    orderMustVerificationStrategyMap.forEach((k, orderMustVerificationStrategy)->{
      try {
        if (CollectionUtils.isEmpty(orderMustVerificationStrategy.processNodeEnums())) {
          log.warn("系统设定订单必校验[{}]未注册", orderMustVerificationStrategy.title());
          return;
        }
        if (orderMustVerificationStrategy.processNodeEnums().contains(processNodeEnum)) {
          orderMustVerificationStrategy.execute(context);
        } else {
          log.warn("订单所处节点[{}]不需要执行系统设定校验[{}]", processNodeEnum.getValue(), orderMustVerificationStrategy.title());
        }
      } catch (Exception e) {
        log.error("执行系统设定订单必校验策略：{} 时，发生异常：{}", orderMustVerificationStrategy.title(), e.getMessage(), e);
        throw new OrderVerificationException(e.getMessage());
      } finally {
        log.info("已完成系统设定执订单必校验策略：{}", orderMustVerificationStrategy.title());
      }
    });

    // 订单配置页面配置的订单校验项
    log.info("需要执行订单校验策略：{}", context.getOrderConfigVo().getVerificationVoList());
    context.getOrderConfigVo().getVerificationVoList().forEach(strategy->{
      try {
        if (strategy.getIsOpen()) {
          OrderVerificationStrategy orderVerificationStrategy = (OrderVerificationStrategy) applicationContext.getBean(strategy.getName());
          Validate.notNull(orderVerificationStrategy, String.format("没有找到当前校验策略实现：%s", strategy.getTitle()));
          if (CollectionUtils.isEmpty(orderVerificationStrategy.processNodeEnums())) {
            log.warn("订单校验[{}]未注册", orderVerificationStrategy.title());
            return;
          }
          if (orderVerificationStrategy.processNodeEnums().contains(processNodeEnum)) {
            orderVerificationStrategy.execute(context);
          } else {
            log.warn("订单所处节点[{}]不需要执行校验[{}]", processNodeEnum.getValue(), orderVerificationStrategy.title());
          }
        } else {
          log.warn("跳过已禁用校验策略实现：{}", strategy.getTitle());
        }
      } catch (Exception e) {
        log.error("执行订单校验策略：{} 时，发生异常：{}", strategy.getName(), e.getMessage(), e);
        throw new OrderVerificationException(e.getMessage());
      } finally {
        log.info("已完成执订单校验策略：{}", strategy.getName());
      }
    });
  }

  @Override
  public List<OrderValidateDetailVo> findProviders (String orderType) {
    Map<String, OrderVerificationStrategy> orderVerificationStrategyMap = applicationContext.getBeansOfType(OrderVerificationStrategy.class);
    log.info("获取订单校验提供者列表信息：{}", orderVerificationStrategyMap);
    OrderTypeEnum orderTypeEnum = OrderTypeEnum.findByCode(orderType);
    Validate.notNull(orderTypeEnum, "订单类型不存在");
    List<OrderValidateDetailVo> orderValidateDetailVos = Lists.newArrayList();
    orderVerificationStrategyMap.forEach((s, strategy) -> {
      if (strategy.orderTypeEnums().contains(orderTypeEnum)) {
        OrderValidateDetailVo orderValidateDetailVo = new OrderValidateDetailVo();
        orderValidateDetailVo.setName(s);
        orderValidateDetailVo.setTitle(strategy.title());
        orderValidateDetailVo.setRequired(strategy.required());
        orderValidateDetailVos.add(orderValidateDetailVo);
      }
    });
    return orderValidateDetailVos;
  }
}
