package com.biz.crm.dms.business.order.verification.local.strategy.internal;

import com.biz.crm.dms.business.costpool.sdk.dto.CostPoolDto;
import com.biz.crm.dms.business.costpool.sdk.strategy.CostPoolRegister;
import com.biz.crm.dms.business.costpool.sdk.strategy.CostPoolStrategy;
import com.biz.crm.dms.business.costpool.sdk.vo.CostPoolVo;
import com.biz.crm.dms.business.order.common.sdk.dto.OrderPayDto;
import com.biz.crm.dms.business.order.common.sdk.enums.OrderTypeEnum;
import com.biz.crm.dms.business.order.common.sdk.enums.OrderValidationProcessNodeEnum;
import com.biz.crm.dms.business.order.common.sdk.register.TallyItemRegister;
import com.biz.crm.dms.business.order.verification.sdk.model.OrderVerificationContext;
import com.biz.crm.dms.business.order.verification.sdk.strategy.OrderMustVerificationStrategy;
import com.google.common.collect.Lists;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import java.math.BigDecimal;
import java.util.Collections;
import java.util.List;
import java.util.stream.Collectors;

/**
 * @description: 资金账户余额校验
 * @author: rentao
 * @date: 2022/3/2 16:45
 */
@Service
public class CostPoolCapitalVerificationStrategyImpl implements OrderMustVerificationStrategy {

  @Autowired(required = false)
  @Qualifier("costPoolCapitalRegisterImpl")
  private CostPoolRegister costPoolRegister;

  @Autowired(required = false)
  @Qualifier("costPoolCapitalStrategyImpl")
  private CostPoolStrategy costPoolStrategy;

  @Autowired(required = false)
  @Qualifier("CapitalTallyItemRegister")
  private TallyItemRegister tallyItemRegister;

  @Override
  public List<OrderValidationProcessNodeEnum> processNodeEnums() {
    return Lists.newArrayList(OrderValidationProcessNodeEnum.THREE);
  }

  @Override
  public List<OrderTypeEnum> orderTypeEnums() {
    return Collections.singletonList(OrderTypeEnum.STANDARD);
  }

  @Override
  public Boolean required() {
    return Boolean.TRUE;
  }

  @Override
  public String title() {
    return "资金账户余额校验";
  }

  @Override
  public void execute(OrderVerificationContext orderVerificationContext) {
    Validate.notNull(orderVerificationContext, "订单上下文信息缺失！");
    Validate.notNull(orderVerificationContext.getOrderDto(), "订单信息缺失！");
    List<OrderPayDto> orderPays = orderVerificationContext.getOrderDto().getOrderPays();
    if (CollectionUtils.isEmpty(orderPays)) {
      return;
    }
    String customerCode = orderVerificationContext.getOrderDto().getRelateCode();
    Validate.notBlank(customerCode, "订单客户编码不能为空！");
    // - 汇总本单用了多少资金
    BigDecimal amount = BigDecimal.ZERO;
    List<OrderPayDto> payDtos = orderPays.stream()
        .filter(orderPayDto ->
            tallyItemRegister.findTallyItemRegisterModel().getItemKey().equals(orderPayDto.getItemKey())
        )
        .collect(Collectors.toList());
    if (!CollectionUtils.isEmpty(payDtos)) {
      BigDecimal reduce = payDtos.stream()
          .map(OrderPayDto::getItemAmount)
          .reduce(BigDecimal.ZERO, BigDecimal::add);
      amount = amount.add(reduce);
    }
    boolean isZero = amount.compareTo(BigDecimal.ZERO) == 0;
    if (isZero) {
      return;
    }
    // - 查询客户的资金信息
    CostPoolDto costPoolDto = new CostPoolDto();
    costPoolDto.setPoolType(costPoolRegister.getKey());
    costPoolDto.setCustomerCode(customerCode);
    List<CostPoolVo> costPoolVos = costPoolStrategy.onRequestCostPoolVos(costPoolDto);
    Validate.isTrue(!CollectionUtils.isEmpty(costPoolVos), "该订单使用了资金，但客户不存在资金池信息！");
    //汇总剩余可使用金额
    BigDecimal usableAmount = costPoolVos.stream()
        .map(CostPoolVo::getUsableAmount)
        .reduce(BigDecimal.ZERO, BigDecimal::add);
    //比较
    Validate.isTrue(amount.compareTo(usableAmount) <= 0, "资金账户余额不足!");
  }
}
