package com.biz.crm.dms.business.order.verification.local.strategy.internal;

import com.biz.crm.dms.business.costpool.sdk.dto.CostPoolDto;
import com.biz.crm.dms.business.costpool.sdk.strategy.CostPoolRegister;
import com.biz.crm.dms.business.costpool.sdk.strategy.CostPoolStrategy;
import com.biz.crm.dms.business.costpool.sdk.vo.CostPoolVo;
import com.biz.crm.dms.business.order.common.sdk.dto.OrderDetailDto;
import com.biz.crm.dms.business.order.common.sdk.dto.OrderDto;
import com.biz.crm.dms.business.order.common.sdk.enums.ItemTypeEnum;
import com.biz.crm.dms.business.order.common.sdk.enums.OrderTypeEnum;
import com.biz.crm.dms.business.order.common.sdk.enums.OrderValidationProcessNodeEnum;
import com.biz.crm.dms.business.order.verification.sdk.strategy.OrderMustVerificationStrategy;
import com.google.common.collect.Lists;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.stereotype.Service;

import java.math.BigDecimal;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * @author HanJiaJun
 * @describe: 货补费用池余额校验
 * @createTime 2022年01月13日 15:10:00
 */
@Service
public class CostPoolReplenishmentVerificationStrategyImpl implements OrderMustVerificationStrategy {


  @Autowired(required = false)
  @Qualifier("costPoolReplenishmentStrategyImpl")
  private CostPoolStrategy costPoolStrategy;

  @Autowired(required = false)
  @Qualifier("CostPoolReplenishmentRegisterImpl")
  private CostPoolRegister costPoolRegister;

  @Override
  public List<OrderValidationProcessNodeEnum> processNodeEnums() {
    return Lists.newArrayList(OrderValidationProcessNodeEnum.THREE);
  }

  @Override
  public List<OrderTypeEnum> orderTypeEnums() {
    return Lists.newArrayList(OrderTypeEnum.STANDARD, OrderTypeEnum.REPLENISHMENT);
  }

  @Override
  public Boolean required() {
    return true;
  }

  @Override
  public String title() {
    return "货补费用池余额校验";
  }

  @Override
  public void execute(OrderDto orderContextDto) {
    Validate.notNull(orderContextDto, "订单上下文信息缺失");
    String customerCode = orderContextDto.getRelateCode();
    Validate.notBlank(customerCode, "订单客户编码不能为空");
    //获取明细
    List<OrderDetailDto> detailList = orderContextDto.getOrderDetails();
    List<String> poolCodes = detailList.stream()
        .filter(s -> s.getItemType().equals(ItemTypeEnum.COMPENSATED_GOODS.getDictCode()))
        .filter(s -> StringUtils.isNotBlank(s.getItemCode()))
        .map(OrderDetailDto::getItemCode).collect(Collectors.toList());
    CostPoolDto costPoolDto = new CostPoolDto();
    costPoolDto.setPoolCodeList(poolCodes);
    costPoolDto.setCustomerCode(customerCode);
    costPoolDto.setPoolType(this.costPoolRegister.getKey());
    //查询费用池信息
    List<CostPoolVo> costPoolList = this.costPoolStrategy.onRequestCostPoolVos(costPoolDto);
    //费用池信息分组
    Map<String, BigDecimal> costPoolMap = costPoolList.stream()
        .filter(s -> StringUtils.isNotBlank(s.getPoolCode()))
        .filter(s -> s.getUsableAmount() != null)
        .collect(Collectors.toMap(CostPoolVo::getPoolCode, CostPoolVo::getUsableAmount, (oldv, newv) -> newv));
    //明细分组
    Map<String, List<OrderDetailDto>> detailMap = detailList.stream()
        .filter(s -> s.getItemType().equals(ItemTypeEnum.COMPENSATED_GOODS.getDictCode()))
        .collect(Collectors.groupingBy(OrderDetailDto::getItemCode));
    //遍历明细MAP 比较金额
    detailMap.forEach((k, v) -> {
      BigDecimal amount = new BigDecimal(0);
      for (OrderDetailDto dto : v) {
        amount = amount.add(dto.getSalesAmount());
      }
      if (costPoolMap.containsKey(k)) {
        BigDecimal usableAmount = costPoolMap.get(k);
        Validate.isTrue(!(amount.compareTo(usableAmount) == 1), "货补费用池余额不足");
      }
    });
  }
}
