package com.biz.crm.dms.business.order.verification.local.strategy.internal;

import com.biz.crm.dms.business.costpool.sdk.dto.CostPoolDto;
import com.biz.crm.dms.business.costpool.sdk.strategy.CostPoolRegister;
import com.biz.crm.dms.business.costpool.sdk.strategy.CostPoolStrategy;
import com.biz.crm.dms.business.costpool.sdk.vo.CostPoolVo;
import com.biz.crm.dms.business.order.common.sdk.dto.OrderDto;
import com.biz.crm.dms.business.order.common.sdk.dto.OrderPayDto;
import com.biz.crm.dms.business.order.common.sdk.enums.OrderTypeEnum;
import com.biz.crm.dms.business.order.common.sdk.enums.OrderValidationProcessNodeEnum;
import com.biz.crm.dms.business.order.common.sdk.register.TallyItemRegister;
import com.biz.crm.dms.business.order.verification.sdk.strategy.OrderMustVerificationStrategy;
import com.google.common.collect.Lists;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import java.math.BigDecimal;
import java.util.List;
import java.util.stream.Collectors;

/**
 * 授信资金校验
 *
 * @author ning.zhang
 * @date 2022/3/2
 */
@Slf4j
@Service
public class CreditOrderVerificationStrategyImpl implements OrderMustVerificationStrategy {

  @Autowired(required = false)
  @Qualifier("CostPoolCreditStrategyImpl")
  private CostPoolStrategy costPoolStrategy;

  @Autowired(required = false)
  @Qualifier("CostPoolCreditRegisterImpl")
  private CostPoolRegister costPoolRegister;

  @Autowired(required = false)
  @Qualifier("CreditTallyItemRegister")
  private TallyItemRegister tallyItemRegister;

  @Override
  public List<OrderValidationProcessNodeEnum> processNodeEnums() {
    return Lists.newArrayList(OrderValidationProcessNodeEnum.THREE);
  }

  @Override
  public List<OrderTypeEnum> orderTypeEnums() {
    return Lists.newArrayList(OrderTypeEnum.STANDARD);
  }

  @Override
  public Boolean required() {
    return Boolean.TRUE;
  }

  @Override
  public String title() {
    return "授信账户可用余额校验";
  }

  @Override
  public void execute(OrderDto orderContextDto) {
    Validate.notNull(orderContextDto, "订单上下文信息缺失");
    String customerCode = orderContextDto.getRelateCode();
    Validate.notBlank(customerCode, "订单客户编码不能为空");
    CostPoolDto costPoolDto = new CostPoolDto();
    costPoolDto.setPoolType(this.costPoolRegister.getKey());
    costPoolDto.setCustomerCode(customerCode);
    //查询授信资金信息
    List<CostPoolVo> costPoolVoList = this.costPoolStrategy.onRequestCostPoolVos(costPoolDto);
    //汇总剩余可使用金额
    BigDecimal usableAmount = costPoolVoList.stream().map(CostPoolVo::getUsableAmount).reduce(BigDecimal.ZERO, BigDecimal::add);
    BigDecimal amount = new BigDecimal(0);
    List<OrderPayDto> payDtos = orderContextDto.getOrderPays().stream().filter(
        orderPayDto -> tallyItemRegister.findTallyItemRegisterModel().getItemKey()
            .equals(orderPayDto.getItemKey())).collect(
        Collectors.toList());
    if (!CollectionUtils.isEmpty(payDtos)) {
      BigDecimal reduce = payDtos.stream()
          .map(OrderPayDto::getItemAmount).reduce(BigDecimal.ZERO, BigDecimal::add);
      amount = amount.add(reduce);
    }
    Validate.isTrue(usableAmount.compareTo(amount) >= 0, "授信资金余额不足");
  }
}
