package com.biz.crm.dms.business.order.verification.local.strategy.internal;

import com.biz.crm.dms.business.order.common.sdk.dto.OrderDetailTabulateDto;
import com.biz.crm.dms.business.order.common.sdk.enums.ItemTypeEnum;
import com.biz.crm.dms.business.order.common.sdk.enums.OrderTypeEnum;
import com.biz.crm.dms.business.order.common.sdk.enums.OrderValidationProcessNodeEnum;
import com.biz.crm.dms.business.order.verification.sdk.model.OrderVerificationContext;
import com.biz.crm.dms.business.order.verification.sdk.strategy.OrderVerificationStrategy;
import com.biz.crm.mdm.business.price.sdk.constant.PriceConstant;
import com.biz.crm.mdm.business.price.sdk.dto.FindPriceDto;
import com.biz.crm.mdm.business.price.sdk.enums.FindPriceUserTypeEnum;
import com.biz.crm.mdm.business.price.sdk.service.PriceModelVoService;
import com.biz.crm.mdm.business.price.sdk.vo.PriceModelVo;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import java.math.BigDecimal;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

/**
 * 最低价校验实现
 *
 * @author sunx
 * @date 2022/1/12
 */
@Service
public class LowestPriceOrderVerificationStrategyImpl implements OrderVerificationStrategy {

  @Autowired(required = false) private PriceModelVoService priceModelVoService;

  @Override
  public List<OrderValidationProcessNodeEnum> processNodeEnums() {
    return Lists.newArrayList(OrderValidationProcessNodeEnum.TEN);
  }

  @Override
  public List<OrderTypeEnum> orderTypeEnums() {
    return Lists.newArrayList(OrderTypeEnum.STANDARD);
  }

  @Override
  public Boolean required() {
    return Boolean.FALSE;
  }

  @Override
  public String title() {
    return "最低价校验";
  }

  @Override
  public void execute(OrderVerificationContext context) {
    Validate.notNull(context, "订单上下文信息缺失");
    Validate.notNull(context.getOrderDto(), "订单原始报文缺失");
    Validate.notNull(context.getOrderConfigVo(), "订单配置信息缺失");
    Validate.notNull(context.getOrderTabulateDto(), "订单分摊信息缺失");
    Validate.isTrue(
        !CollectionUtils.isEmpty(context.getOrderTabulateDto().getOrderDetails()), "分摊后的订单明细不能为空");
    // FIXME 目前只是默认实现客户订单最低价校验，后续扩展其他订单类型需完善此处逻辑
    String customerCode = context.getOrderDto().getRelateCode();
    List<OrderDetailTabulateDto> orderDetails = context.getOrderTabulateDto().getOrderDetails();
    // 不揉价时取本品进行最低价校验
    if (Boolean.FALSE.equals(context.getOrderConfigVo().getIsKneadingPrice())) {
      orderDetails =
          context.getOrderTabulateDto().getOrderDetails().stream()
              .filter(a -> a.getItemType().equals(ItemTypeEnum.NORMAL_GOODS.getDictCode()))
              .collect(Collectors.toList());
      if (CollectionUtils.isEmpty(orderDetails)) {
        return;
      }
    }
    Set<String> productCodeSet =
        orderDetails.stream()
            .filter(a -> StringUtils.isNotBlank(a.getGoodsCode()))
            .map(OrderDetailTabulateDto::getGoodsCode)
            .collect(Collectors.toSet());
    Validate.isTrue(!CollectionUtils.isEmpty(productCodeSet), "订单明细商品编码不能为空");

    FindPriceDto dto = new FindPriceDto();
    dto.setPriceTypeCode(Sets.newHashSet(PriceConstant.LOWEST_PRICE));
    dto.setProductCodeSet(productCodeSet);
    dto.setUserType(FindPriceUserTypeEnum.CUSTOMER.getDictCode());
    dto.setUserCode(customerCode);
    Map<String, PriceModelVo> map = this.priceModelVoService.findPrice(dto);

    for (OrderDetailTabulateDto item : orderDetails) {
      if (!map.containsKey(item.getGoodsCode())) {
        Validate.isTrue(false, StringUtils.join("商品编码", item.getGoodsCode(), "的产品未配置最低价"));
      }
      BigDecimal averageUnitPrice =
          Optional.ofNullable(item.getAverageUnitPrice()).orElse(BigDecimal.ZERO);
      BigDecimal lowestPrice =
          Optional.ofNullable(map.get(item.getGoodsCode()).getPrice()).orElse(BigDecimal.ZERO);
      Validate.isTrue(
          averageUnitPrice.compareTo(lowestPrice) >= 0,
          StringUtils.join("商品编码", item.getGoodsCode(), "优惠后单价低于最低价！"));
    }
  }
}
