package com.biz.crm.dms.business.order.verification.local.strategy.internal;

import com.biz.crm.dms.business.order.common.sdk.dto.OrderDto;
import com.biz.crm.dms.business.order.common.sdk.dto.OrderPayDto;
import com.biz.crm.dms.business.order.common.sdk.dto.OrderTabulateDto;
import com.biz.crm.dms.business.order.common.sdk.enums.OrderCategoryEnum;
import com.biz.crm.dms.business.order.common.sdk.enums.OrderTypeEnum;
import com.biz.crm.dms.business.order.common.sdk.enums.OrderValidationProcessNodeEnum;
import com.biz.crm.dms.business.order.verification.sdk.model.OrderVerificationContext;
import com.biz.crm.dms.business.order.verification.sdk.strategy.OrderMustVerificationStrategy;
import com.google.common.collect.Lists;
import org.apache.commons.lang3.Validate;
import org.springframework.stereotype.Component;

import java.math.BigDecimal;
import java.util.List;
import java.util.Objects;

/**
 * The type Payment complete verification strategy.
 * 应付金额支付完成校验
 *
 * @author he fan
 */
@Component
public class PaymentCompleteVerificationStrategyImpl implements OrderMustVerificationStrategy {
  @Override
  public List<OrderValidationProcessNodeEnum> processNodeEnums() {
    return Lists.newArrayList(OrderValidationProcessNodeEnum.TEN);
  }

  @Override
  public List<OrderTypeEnum> orderTypeEnums() {
    return Lists.newArrayList(OrderTypeEnum.STANDARD, OrderTypeEnum.REPLENISHMENT, OrderTypeEnum.DISCOUNT);
  }

  @Override
  public Boolean required() {
    return Boolean.TRUE;
  }

  @Override
  public String title() {
    return "应付金额支付完成校验";
  }

  @Override
  public void execute(OrderVerificationContext orderVerificationContext) {
    /**
     * 校验订单上的所有应付金额都能够支付完成，若账户余额不足，订单不可提交并弹出提示。
     * 校验订单金额-政策优惠金额-货补-折扣=X（应付金额）
     * 使用授信金额+使用资金金额=Y
     * 校验Y是否等于X。等于才允许通过。
     */
    OrderTabulateDto orderTabulateDto = orderVerificationContext.getOrderTabulateDto();
    OrderDto orderDto = orderVerificationContext.getOrderDto();
    // - 不处理物料订单
    String orderCategory = orderDto.getOrderCategory();
    if (orderCategory.equals(OrderCategoryEnum.MATERIAL_ORDER.getDictCode())) {
      return;
    }
    String orderType = orderDto.getOrderType();
    boolean isFree = OrderTypeEnum.FREE.getDictCode().equals(orderType);
    if (isFree) {
      return;
    }
    List<OrderPayDto> orderPays = orderDto.getOrderPays();
    //订单总金额
    BigDecimal totalOrderAmount = orderTabulateDto.getTotalOrderAmount();
    //政策优惠金额
    BigDecimal policyDiscountAmount = orderTabulateDto.getPolicyDiscountAmount();
    //货补金额
    BigDecimal totalCompensatedAmount = orderTabulateDto.getTotalCompensatedAmount();
    //折扣金额--其实是所有该扣项目的汇总
    BigDecimal totalDiscountAmount = BigDecimal.ZERO;
    for (OrderPayDto orderPay : orderPays) {
      Boolean itemGroupType = orderPay.getItemGroupType();
      if (!itemGroupType) {
        BigDecimal itemAmount = orderPay.getItemAmount();
        itemAmount = Objects.nonNull(itemAmount) ? itemAmount : BigDecimal.ZERO;
        totalDiscountAmount = totalDiscountAmount.add(itemAmount);
      }
    }
    //授信金额 + 资金金额--其实是所有支付项目的汇总
    BigDecimal totalPaymentAmount = BigDecimal.ZERO;
    for (OrderPayDto orderPay : orderPays) {
      Boolean itemGroupType = orderPay.getItemGroupType();
      if (itemGroupType) {
        BigDecimal itemAmount = orderPay.getItemAmount();
        itemAmount = Objects.nonNull(itemAmount) ? itemAmount : BigDecimal.ZERO;
        totalPaymentAmount = totalPaymentAmount.add(itemAmount);
      }
    }
    // 校验订单金额-政策优惠金额-货补-折扣=X（应付金额）
    BigDecimal amountPayable = totalOrderAmount.subtract(policyDiscountAmount).subtract(totalCompensatedAmount).subtract(totalDiscountAmount);
    int compare = amountPayable.compareTo(totalPaymentAmount);
    Validate.isTrue(compare == 0, "应付金额与实际支付金额不相等");
  }
}
