package com.biz.crm.dms.business.order.verification.local.strategy.internal;

import com.biz.crm.dms.business.order.common.sdk.dto.OrderDetailDto;
import com.biz.crm.dms.business.order.common.sdk.dto.OrderDto;
import com.biz.crm.dms.business.order.common.sdk.enums.OrderCategoryEnum;
import com.biz.crm.dms.business.order.common.sdk.enums.OrderTypeEnum;
import com.biz.crm.dms.business.order.common.sdk.enums.OrderValidationProcessNodeEnum;
import com.biz.crm.dms.business.order.verification.sdk.strategy.OrderVerificationStrategy;
import com.biz.crm.dms.business.psi.product.sdk.service.productstock.ProductStockVoService;
import com.biz.crm.dms.business.psi.product.sdk.vo.productstock.ProductStockVo;
import com.biz.crm.mdm.business.warehouse.sdk.service.WarehouseVoService;
import com.biz.crm.mdm.business.warehouse.sdk.vo.WarehouseVo;
import com.google.common.collect.Lists;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.math.BigDecimal;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * 库存可用性校验
 * @Author: zengxingwang
 * @Date: 2021/12/29 17:58
 */
@Service
public class StockOrderVerificationStrategyImpl implements OrderVerificationStrategy {

  @Autowired(required = false)
  private WarehouseVoService warehouseVoService;
  @Autowired(required = false)
  private ProductStockVoService productStockVoService;

  @Override
  public List<OrderValidationProcessNodeEnum> processNodeEnums() {
    return Lists.newArrayList(OrderValidationProcessNodeEnum.THREE);
  }

  @Override
  public List<OrderTypeEnum> orderTypeEnums() {
    return Lists.newArrayList(OrderTypeEnum.values());
  }

  @Override
  public Boolean required() {
    return Boolean.FALSE;
  }

  @Override
  public String title() {
    return "库存可用性校验";
  }

  @Override
  public void execute(OrderDto orderDto) {
    Map<String, BigDecimal> productMap = new HashMap<>();
    BigDecimal quantity = BigDecimal.ZERO;
    for(OrderDetailDto vo : orderDto.getOrderDetails()){
      if(productMap.containsKey(vo.getGoodsCode())){
        quantity = productMap.get(vo.getGoodsCode()).add(vo.getQuantity());
      }else {
        quantity = vo.getQuantity();
      }
      productMap.put(vo.getGoodsCode(), quantity);
    }

    Validate.notBlank(orderDto.getCityCode(), "库存可用性校验时，市编码不能为空");
    WarehouseVo warehouse = null;
    if (orderDto.getOrderCategory().equals(OrderCategoryEnum.MATERIAL_ORDER.getDictCode())) {
      String warehouseCode = orderDto.getWarehouseCode();
      warehouse = warehouseVoService.findDetailsByCode(warehouseCode);
    } else {
      warehouse = warehouseVoService.findDetailsByCityCode(orderDto.getCityCode());
      if(warehouse == null){
        Boolean bool = true;
        warehouse = warehouseVoService.findDetailsByWarehouseDefault(bool);
      }
    }

    List<ProductStockVo> productStockVos = productStockVoService.findByWarehouseAndProducts(warehouse.getWarehouseCode(), productMap.keySet());

    if(productStockVos.size() != productMap.keySet().size()){
      for(String key : productMap.keySet()){
        Boolean flag = false;
        for(ProductStockVo vo : productStockVos){
          if(vo.getProductCode().equals(key)){
            flag = true;
            break;
          }
        }
        Validate.isTrue(flag, "仓库%s，无商品%s，请检查", warehouse.getWarehouseCode(), key);
        break;
      }
    }

    for(ProductStockVo vo : productStockVos){
      BigDecimal bigDecimal = productMap.get(vo.getProductCode()) == null ? BigDecimal.ZERO : productMap.get(vo.getProductCode());
      Validate.isTrue(vo.getAvailableStock().compareTo(bigDecimal) != -1 ,"商品%s，库存不足", vo.getProductCode());
    }
  }

}
