package com.biz.crm.dms.business.policy.local.characteristic;

import java.util.ArrayList;
import java.util.LinkedHashSet;
import java.util.Set;
import java.util.stream.Collectors;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;

import com.biz.crm.dms.business.policy.local.entity.NotStackingCharacteristicInfo;
import com.biz.crm.dms.business.policy.local.repository.NotStackingCharacteristicInfoRepository;
import com.biz.crm.dms.business.policy.local.vo.NotStackingCharacteristicInfoVo;
import com.biz.crm.dms.business.policy.sdk.characteristic.CharacteristicStrategy;
import com.biz.crm.dms.business.policy.sdk.characteristic.CharacteristicVoter;
import com.biz.crm.dms.business.policy.sdk.context.AbstractPolicyExecuteContext;
import com.biz.crm.dms.business.policy.sdk.vo.AbstractCharacteristicInfo;
import com.biz.crm.dms.business.policy.sdk.vo.SalePolicyVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;

/**
 * “不允许叠加”的优惠政策执行特性，由标品进行提供的操作特性；
 * @author yinwenjie
 */
@Component
public class NotStackingCharacteristicStrategy implements CharacteristicStrategy<NotStackingCharacteristicInfoVo> {

  @Autowired(required = false)
  private NotStackingCharacteristicInfoRepository notStackingCharacteristicInfoRepository;
  
  @Autowired(required = false)
  private NebulaToolkitService nebulaToolkitService;
  
  @Override
  public String characteristicCode() {
    return "notStackingCharacteristicStrategy";
  }

  @Override
  public String characteristicName() {
    return "选定后表示该优惠政策不可进行叠加";
  }

  @Override
  public String getExpression() {
    return "选定后该优惠政策不可进行叠加";
  }

  @Override
  public Class<NotStackingCharacteristicInfoVo> getCharacteristicInfoClass() {
    return NotStackingCharacteristicInfoVo.class;
  }

  @Override
  public NotStackingCharacteristicInfoVo onRequestCharacteristicInfo(String salePolicyCode) {
    // 从数据层，通过当前优惠政策编号和租户编号，查询特定信息
    String tenantCode = TenantUtils.getTenantCode();
    NotStackingCharacteristicInfo notStackingCharacteristicInfo = this.notStackingCharacteristicInfoRepository.findBySalePolicyCodeAndTenantCode(salePolicyCode, tenantCode);
    if(notStackingCharacteristicInfo == null) {
      return null;
    }
    
    // 开始进行VO转换
    NotStackingCharacteristicInfoVo vo = this.nebulaToolkitService.copyObjectByWhiteList(notStackingCharacteristicInfo, NotStackingCharacteristicInfoVo.class, LinkedHashSet.class, ArrayList.class);
    return vo;
  }

  @Override
  public void onSaveCharacteristicInfo(boolean update, SalePolicyVo currentSalePolicy, SalePolicyVo oldSalePolicy) {
    /*
     * 这里要判定新老结构中的信息情况，以便确定是对执行特性的新增还是删除
     * 1、如果update为false，且currentSalePolicy中存在characteristicCode为“notStackingCharacteristicStrategy”
     * 的执行特性，那么说明是新增；
     * 2、如果update为true，且currentSalePolicy存在，说明可能是新增；
     * 3、如果update为true。且currentSalePolicy不存在，说明可能是删除；
     * */
    
    // 1和2、======
    Set<AbstractCharacteristicInfo> characteristicInfos = currentSalePolicy.getCharacteristicInfos();
    String characteristicCode = this.characteristicCode();
    if(!CollectionUtils.isEmpty(characteristicInfos) 
        && characteristicInfos.stream().map(AbstractCharacteristicInfo::getCharacteristicCode).collect(Collectors.toSet()).contains(characteristicCode)) {
      String salePolicyCode = currentSalePolicy.getSalePolicyCode();
      String tenantCode = currentSalePolicy.getTenantCode();
      NotStackingCharacteristicInfo exsit = this.notStackingCharacteristicInfoRepository.findBySalePolicyCodeAndTenantCode(salePolicyCode, tenantCode);
      // 说明已经存在，不需要重复添加
      if(exsit != null) {
        return;
      }
      // 走到这里说明是添加操作  
      NotStackingCharacteristicInfo notStackingCharacteristicInfo = new NotStackingCharacteristicInfo();
      notStackingCharacteristicInfo.setCharacteristicCode(characteristicCode);
      notStackingCharacteristicInfo.setSalePolicyCode(salePolicyCode);
      notStackingCharacteristicInfo.setTenantCode(tenantCode);
      this.notStackingCharacteristicInfoRepository.save(notStackingCharacteristicInfo);
    } 
    // 3、======
    else if(update && (CollectionUtils.isEmpty(characteristicInfos) 
            || !characteristicInfos.stream().map(AbstractCharacteristicInfo::getCharacteristicCode).collect(Collectors.toSet()).contains(characteristicCode))) {
      String salePolicyCode = currentSalePolicy.getSalePolicyCode();
      String tenantCode = currentSalePolicy.getTenantCode();
      this.notStackingCharacteristicInfoRepository.deleteBySalePolicyCodeAndTenantCode(salePolicyCode, tenantCode);
    }
  }

  @Override
  public CharacteristicVoter make(AbstractPolicyExecuteContext executeContext, SalePolicyVo salePolicyVo, String customerCode, boolean formal, String relevanceCode) {
    /*
     * 在预执行/正式执行过程中，如果特定的优惠政策勾选了“不允许叠加”，那么本方法就会投出独占票
     * */
    String salePolicyCode = salePolicyVo.getSalePolicyCode();
    String tenantCode = salePolicyVo.getTenantCode();
    NotStackingCharacteristicInfo exsit = this.notStackingCharacteristicInfoRepository.findBySalePolicyCodeAndTenantCode(salePolicyCode, tenantCode);
    if(exsit == null) {
      return CharacteristicVoter.GRANTED;
    }
    
    // 投出独占票
    return CharacteristicVoter.MONOPOLY;
  }

  @Override
  public String errorMsg(SalePolicyVo salePolicyVo) {
    String salePolicyName = salePolicyVo.getSalePolicyName();
    return String.format("选择了不可叠加的优惠政策【%s】", salePolicyName);
  }
}
