package com.biz.crm.dms.business.policy.local.context;

import java.math.BigDecimal;
import java.util.Deque;
import java.util.Set;

import org.apache.commons.lang3.Validate;
import org.springframework.util.CollectionUtils;

import com.biz.crm.dms.business.policy.sdk.context.AbstractCycleExecuteContext;
import com.biz.crm.dms.business.policy.sdk.context.CycleStepResult;
import com.biz.crm.dms.business.policy.sdk.context.SalePolicyConProduct;
import com.google.common.collect.Queues;

/**
 * 由优惠政策标品实现的优惠政策阶梯计算上下文</br>
 * 注意，这里只记录每一次阶梯计算的优惠情况，并不会将每一次优惠后的情况分摊到每一条参与优惠的本品上。也就是说：</br>
 * 本次进行优惠的本品是A、B、C，优惠后这些本品的小计价格总共优惠了50元，所以这里的阶梯计算上下文只会记录优惠了50元，而不会分摊到 A本品小计价格优惠了10元，B本品小计价格优惠了20元、C本品的小计价格优惠了20元
 * @author yinwenjie
 */
public class DefaultCycleExecuteContext implements AbstractCycleExecuteContext {
  /**
   * 优惠政策阶梯正式开始执行前，参与本次执行的本品情况（注意，这些对象其中也包括了价格、数量信息，但都是最初的原始价格和原始数量，对本次阶梯循环计算帮助不大）
   */
  private Set<SalePolicyConProduct> initPolicyProducts;
  /**
   * 本次优惠政策阶梯执行所使用的优惠政策执行器编码
   */
  private String executeCode;
  
  // ======= 以下是小计价格和小计优惠信息
  
  /**
   * 本次执行开始前，参与本次优惠执行的所有本品的小计价格</br>
   * 例如 A本品小计101.00 ， B本品小计202.05，则小计价格为303.05</br>
   * 这个初始值，在整个结算过程中都不会改变
   */
  private final BigDecimal initSubtotalAmount;
  
  /**
   * 本次执行开始前，参与本次优惠执行的所有本品的小计数量</br> 
   * 例如 A本品数量30 ， B本品小计数量73，则小计数量为103</br>
   * 这个初始值，在整个结算过程中都不会改变
   */
  private final Integer initSubtotalNumber;
  /**
   * 当前执行阶梯正在使用的客户业务编号
   */
  private String customerCode;
  
  /**
   * 本次执行过程中，每一次阶梯优惠的结果情况，都有序的记录在这里，这里的顺序主要依据CycleStepResult对象中的ladderIndex属性和executeTimes属性</br>
   * 且这两个属性叠加后不会重复
   */
  private Deque<CycleStepResult> cycleStepResults;

  public DefaultCycleExecuteContext(String executeCode, String customerCode , Set<SalePolicyConProduct> initPolicyProducts, BigDecimal initSubtotalAmount, Integer initSubtotalNumber , 
                                    BigDecimal preSubtotalAmount, BigDecimal preSurplusSubtotalAmount, Integer preSurplusSubtotalNumber) {
    this.customerCode = customerCode;
    Validate.notBlank(executeCode, "优惠政策步进执行时，必须设定步进执行所使用的客户编号customerCode");
    this.executeCode = executeCode;
    Validate.notBlank(executeCode, "优惠政策步进执行时，必须设定步进执行所使用的执行器编号executorCode");
    this.initPolicyProducts = initPolicyProducts;
    Validate.isTrue(!CollectionUtils.isEmpty(initPolicyProducts) , "在初始化政策阶梯执行上下文时，能够参与本次优惠政策计算的本品信息必须传入。");
    this.initSubtotalAmount = initSubtotalAmount;
    Validate.notNull(this.initSubtotalAmount , "在初始化政策阶梯执行上下文时，计算初始的小计价格必须传入。");
    Validate.isTrue(this.initSubtotalAmount.floatValue() > 0.0f , "在初始化政策阶梯执行上下文时，计算初始的小计价格必须大于0。");
    this.initSubtotalNumber = initSubtotalNumber;
    Validate.notNull(this.initSubtotalNumber , "在初始化政策阶梯执行上下文时，计算初始的小计数量必须传入。");
    Validate.isTrue(this.initSubtotalNumber.intValue() > 0 , "在初始化政策阶梯执行上下文时，计算初始的小计数量必须大于0。");
    
    // 其它信息也要进行相应的初始化(阶梯执行结果中，必须要添加一条信息，作为初始信息)
    this.cycleStepResults = Queues.newArrayDeque();
    CycleStepResult initCycleStepResult = new CycleStepResult(0, 0, preSubtotalAmount, preSurplusSubtotalAmount, preSurplusSubtotalNumber);
    initCycleStepResult.setLastSubtotalAmount(preSubtotalAmount);
    initCycleStepResult.setLastSurplusSubtotalAmount(preSurplusSubtotalAmount);
    initCycleStepResult.setLastSurplusSubtotalNumber(preSurplusSubtotalNumber);
    this.cycleStepResults.push(initCycleStepResult);
  }
  
  /**
   * 为当前阶梯执行上下文添加一个新的阶梯优惠结果步进
   */
  public void addStepResult(CycleStepResult newCycleStepResult) {
    Validate.notNull(newCycleStepResult , "新的阶梯步骤优惠结果必须传入!!");
    Integer ladderIndex = newCycleStepResult.getLadderIndex();
    Integer executeTimes = newCycleStepResult.getExecuteTimes();
    Validate.isTrue(ladderIndex != null && ladderIndex >= 0 , "新的阶梯步骤优惠层级需要正确，请检查!!");
    Validate.isTrue(executeTimes != null && executeTimes >= 1 , "新的阶梯步骤优惠执行序号需要正确，请检查!!");
    for (CycleStepResult oldCycleStepResult : cycleStepResults) {
      Validate.isTrue(oldCycleStepResult.getExecuteTimes().intValue() != executeTimes.intValue() , "新的阶梯步骤优惠执行序号已存在，请检查重复性!!");
    }
    
    // 初始信息必须有值
    BigDecimal preSubtotalAmount = newCycleStepResult.getPreSubtotalAmount();
    Validate.notNull(preSubtotalAmount , "新的阶梯步骤必须传入preSubtotalAmount，阶梯优惠前的总小计价格");
    BigDecimal preSurplusSubtotalAmount = newCycleStepResult.getPreSurplusSubtotalAmount();
    Validate.notNull(preSurplusSubtotalAmount , "新的阶梯步骤必须传入preSurplusSubtotalAmount，阶梯优惠前还未进行优惠的小计价格");
    Integer preSurplusSubtotalNumber = newCycleStepResult.getPreSurplusSubtotalNumber();
    Validate.notNull(preSurplusSubtotalNumber , "新的阶梯步骤必须传入preSurplusSubtotalNumber，阶梯优惠前还未进行优惠的小计数量");
    // 结果必须有值
    BigDecimal lastSubtotalAmount = newCycleStepResult.getLastSubtotalAmount();
    Validate.notNull(lastSubtotalAmount , "新的阶梯步骤必须传入lastSubtotalAmount，阶梯优惠后的总小计价格");
    BigDecimal lastSurplusSubtotalAmount = newCycleStepResult.getLastSurplusSubtotalAmount();
    Validate.notNull(lastSurplusSubtotalAmount , "新的阶梯步骤必须传入lastSurplusSubtotalAmount，阶梯优惠后的还未进行优惠的小计价格");
    Integer lastSurplusSubtotalNumber = newCycleStepResult.getLastSurplusSubtotalNumber();
    Validate.notNull(lastSurplusSubtotalNumber , "新的阶梯步骤必须传入lastSurplusSubtotalNumber，阶梯优惠后的还未进行优惠的小计数量");
    this.cycleStepResults.push(newCycleStepResult);
  }
  
  /**
   * 该方法为调用者返回第一次阶梯执行的结果</br>
   * @return 如果还没有任何执行结果，则返回null
   */
  public CycleStepResult findFirstStepResult() {
    return this.cycleStepResults.peekLast();
  }
  
  /**
   * 该方法为调用者返回最后一次阶梯执行的结果，这个返回值可以帮助调用者构造本次优惠阶梯执行结果的初始值</br>
   * @return 如果还没有任何执行结果，则返回null
   */
  public CycleStepResult findLastStepResult() {
    return this.cycleStepResults.peek();
  }

  public Set<SalePolicyConProduct> getInitPolicyProducts() {
    return initPolicyProducts;
  }

  public BigDecimal getInitSubtotalAmount() {
    return initSubtotalAmount;
  }

  public Integer getInitSubtotalNumber() {
    return initSubtotalNumber;
  }

  public String getExecuteCode() {
    return executeCode;
  }

  public String getCustomerCode() {
    return customerCode;
  }
}