package com.biz.crm.dms.business.policy.local.context;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.math.BigDecimal;
import java.math.RoundingMode;
import java.util.Arrays;
import java.util.Deque;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Map.Entry;
import java.util.stream.Collectors;
import java.util.Set;

import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.apache.commons.lang3.tuple.Triple;
import org.springframework.util.CollectionUtils;

import com.biz.crm.dms.business.policy.sdk.context.AbstractPolicyExecuteContext;
import com.biz.crm.dms.business.policy.sdk.context.GiftResultInfo;
import com.biz.crm.dms.business.policy.sdk.context.PolicyStepResult;
import com.biz.crm.dms.business.policy.sdk.context.ProductPolicyStepResult;
import com.biz.crm.dms.business.policy.sdk.context.SalePolicyConProduct;
import com.biz.crm.dms.business.policy.sdk.context.StepType;
import com.biz.crm.dms.business.policy.sdk.vo.SalePolicyVo;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Queues;
import com.google.common.collect.Sets;

import io.swagger.annotations.ApiModel;
import io.swagger.annotations.ApiModelProperty;
import lombok.Getter;

/**
 * 关于优惠政策执行/预执行上下的默认实现
 * @author yinwenjie
 */
@Getter
@ApiModel(value = "DefaultPolicyExecuteContext", description = "关于优惠政策执行/预执行上下的默认实现") 
public class DefaultPolicyExecuteContext implements AbstractPolicyExecuteContext {
  private static final long serialVersionUID = 7243177308073572661L;

  /**
   * 构造的执行器上下文，必须指定经销商、客户信息，以及参与本次计算的商品规格信息、有序的活动政策信息
   * @param initPolicyProducts 参与本次政策计算的商品规格信息
   * @param initSalePolicys 初始化的有效的按照活动开始时间排列的政策信息
   * @param initTotalAmount 整张单据开始所有优惠计算前的初始总价
   * @param initTotalNumber 整张单据开始所有优惠计算前的初始本品数量
   * @param tenantCode 当前活动政策执行所涉及的经销商信息
   * @param customerCode 当前活动政策执行所涉及的客户信息
   * @param kneading 该属性表示在进行优惠政策执行后的分摊过程中，是否进行揉价
   */
  public DefaultPolicyExecuteContext(Set<SalePolicyConProduct> initPolicyProducts , Set<SalePolicyVo> initSalePolicys , BigDecimal initTotalAmount , Integer initTotalNumber , String tenantCode, String customerCode , boolean kneading , boolean formal , String relevanceCode) {
    this.initPolicyProducts = initPolicyProducts;
    if(CollectionUtils.isEmpty(initSalePolicys)) {
      initSalePolicys = Sets.newConcurrentHashSet();
    }
    this.initSalePolicys = initSalePolicys;
    this.tenantCode = tenantCode;
    this.customerCode = customerCode;
    this.initTotalAmount = initTotalAmount;
    this.initTotalNumber = initTotalNumber;
    this.kneading = kneading;
    this.formal = formal;
    this.relevanceCode = relevanceCode;
    Validate.notBlank(tenantCode , "tenantCode not blank !");
    Validate.notBlank(customerCode , "customerCode not blank !");
    Validate.notNull(initTotalAmount , "initTotalAmount not null !");
    Validate.notNull(initTotalNumber , "initTotalNumber not null !");
    if(this.formal) {
      Validate.notBlank(relevanceCode , "relevanceCode not blank !");
    }
    
    this.matchedProductInfoMapping = Maps.newLinkedHashMap();
    this.excludedPolicyMapping = Maps.newLinkedHashMap();
    this.matchedPolicyInfos = Sets.newLinkedHashSet();
    this.productPolicyStepResultMapping = Maps.newLinkedHashMap();
    
    // 完成productPolicyStepResultMapping的初始化（INIT）
    if(CollectionUtils.isEmpty(initPolicyProducts)) {
      return;
    }
    // 验证initPolicyProducts中信息的正确性
    // 初始化商品中的商品业务编号不能重复
    long distinctCount = initPolicyProducts.stream().map(SalePolicyConProduct::getProductCode).distinct().count();
    Validate.isTrue(initPolicyProducts.size() == distinctCount , "创建上下文时，发现传入的商品编号重复或者没有填写，请检查!!");
    if(this.formal) {
      distinctCount = initPolicyProducts.stream().map(SalePolicyConProduct::getBillItemCode).distinct().count();
      Validate.isTrue(initPolicyProducts.size() == distinctCount , "在正式执行创建上下文时，发现传入的商品票据行项目号重复或者没有填写，请检查!!");
    }
    // 一个一个的初始化商品都要验证
    Set<String> salePolicyCodes = initSalePolicys.stream().map(SalePolicyVo::getSalePolicyCode).collect(Collectors.toSet());
    for (SalePolicyConProduct defaultPolicyConProduct : initPolicyProducts) {
      String billItemCode = defaultPolicyConProduct.getBillItemCode();
      if(this.formal) {
        Validate.notBlank(billItemCode , "当前执行是正式执行，但是发现给定的初始化商品中，至少有一个商品没有传入票据行项目号（billItemCode），请检查!!");
      }
      String productCode = defaultPolicyConProduct.getProductCode();
      Validate.notBlank(productCode , "传入的初始化商品必须有商品的业务编号，请检查!!");
      BigDecimal quantity = defaultPolicyConProduct.getQuantity();
      Validate.isTrue(quantity != null && quantity.compareTo(BigDecimal.ZERO) == 1 , "传入的初始化商品必须有数量信息，且数量信息必须大于0");
      BigDecimal subtotalAmount = defaultPolicyConProduct.getSubtotalAmount();
      Validate.isTrue(subtotalAmount != null && subtotalAmount.compareTo(BigDecimal.ZERO) != -1 , "传入的初始化商品必须有小计金额信息，且小计金额信息必须不为负数");
      BigDecimal unitPrice = defaultPolicyConProduct.getUnitPrice();
      Validate.isTrue(unitPrice != null && unitPrice.compareTo(BigDecimal.ZERO) != -1 , "传入的初始化商品必须有单价信息，且单价信息必须不为负数");
      if(defaultPolicyConProduct.getProductName() == null) {
        defaultPolicyConProduct.setProductName("");
      }
      // 判定当前商品能够指定参与的优惠政策，是在initSalePolicys范围以内，否则就要做排除(也就是直接求交集)
      // 这主要是防止在优惠政策使用页面（例如订单页面）工作的过程中，某些优惠政策被突然禁用掉
      String[] selectedPolicyCodes = defaultPolicyConProduct.getSelectedPolicyCodes();
      if(selectedPolicyCodes != null && selectedPolicyCodes.length > 0) {
        String[] realSelectedPolicyCodes = Sets.intersection(Sets.newLinkedHashSet(Lists.newArrayList(selectedPolicyCodes)), salePolicyCodes).toArray(new String[] {});
        defaultPolicyConProduct.setSelectedPolicyCodes(realSelectedPolicyCodes);
      }
    }
    // 开始完成商品本品步进信息的初始化
    for (SalePolicyConProduct policyConProduct : initPolicyProducts) {
      ProductPolicyStepResult productPolicyStepResult = new ProductPolicyStepResult();
      productPolicyStepResult.setStepType(StepType.INIT);
      productPolicyStepResult.setProductCode(policyConProduct.getProductCode());
      productPolicyStepResult.setSalePolicyCode(null);
      productPolicyStepResult.setInitNumbers(policyConProduct.getQuantity().intValue());
      productPolicyStepResult.setInitPrices(policyConProduct.getUnitPrice().setScale(4, RoundingMode.HALF_UP));
      productPolicyStepResult.setInitSubtotal(policyConProduct.getSubtotalAmount().setScale(4, RoundingMode.HALF_UP));
      productPolicyStepResult.setPrePrices(productPolicyStepResult.getInitPrices());
      productPolicyStepResult.setPreSubtotal(productPolicyStepResult.getInitSubtotal());
      productPolicyStepResult.setPreSurplusTotalAmount(productPolicyStepResult.getInitSubtotal());
      productPolicyStepResult.setPreSurplusTotalNumber(productPolicyStepResult.getInitNumbers());
      productPolicyStepResult.setPreGiftEnjoyedTotalNumber(0);
      productPolicyStepResult.setPreGiftEnjoyedTotalAmount(BigDecimal.ZERO);
      productPolicyStepResult.setPreGifts(Lists.newLinkedList());
      productPolicyStepResult.setLastPrices(productPolicyStepResult.getInitPrices());
      productPolicyStepResult.setLastSubtotal(productPolicyStepResult.getInitSubtotal());
      productPolicyStepResult.setLastSurplusTotalAmount(productPolicyStepResult.getInitSubtotal());
      productPolicyStepResult.setLastSurplusTotalNumber(productPolicyStepResult.getInitNumbers());
      productPolicyStepResult.setLastGiftEnjoyedTotalNumber(0);
      productPolicyStepResult.setLastGiftEnjoyedTotalAmount(BigDecimal.ZERO);
      productPolicyStepResult.setLastGifts(Lists.newLinkedList());
      
      // 存入到productPolicyStepResultMapping映射信息中
      Deque<ProductPolicyStepResult> productPolicyStepResults = Queues.newArrayDeque();
      productPolicyStepResults.add(productPolicyStepResult);
      this.productPolicyStepResultMapping.put(policyConProduct.getProductCode(), productPolicyStepResults);
    }
    
    // 完成policyStepResultMapping的初始化（INIT）
    PolicyStepResult policyStepResult = new PolicyStepResult();
    // 初始化的优惠步进信息不存在优惠政策业务编号
    policyStepResult.setSalePolicyCode(null);
    policyStepResult.setStepType(StepType.INIT);
    policyStepResult.setInitNumbers(initTotalNumber);
    policyStepResult.setInitSubtotal(initTotalAmount);
    policyStepResult.setPreGifts(Lists.newArrayList());
    policyStepResult.setPreSubtotal(initTotalAmount);
    policyStepResult.setPreSurplusTotalAmount(initTotalAmount);
    policyStepResult.setPreSurplusTotalNumber(initTotalNumber);
    policyStepResult.setPreGiftEnjoyedTotalNumber(0);
    policyStepResult.setPreGiftEnjoyedTotalAmount(BigDecimal.ZERO);
    policyStepResult.setLastGifts(Lists.newArrayList());
    policyStepResult.setLastSubtotal(initTotalAmount);
    policyStepResult.setLastSurplusTotalNumber(initTotalNumber);
    policyStepResult.setLastSurplusTotalAmount(initTotalAmount);
    policyStepResult.setLastGiftEnjoyedTotalNumber(0);
    policyStepResult.setLastGiftEnjoyedTotalAmount(BigDecimal.ZERO);
    this.policyStepResults = Queues.newArrayDeque();
    this.policyStepResults.push(policyStepResult);
  }
  
  // 以下属性和执行/预执行的初始值有关 =======
  
  /**
   * 在优惠活动正式执行/预执行前，需要传入本次计算所圈定的商品本品范围
   */
  private Set<SalePolicyConProduct> initPolicyProducts;
  /**
   * 在优惠活动正式执行/预执行前，指定经销商可以使用的政策信息(注意是可使用的，不是经过门槛匹配的，也没有经过商品范围匹配)
   * （上下文初始化后不可变）
   */
  @ApiModelProperty("在活动政策执行的一瞬间，指定经销商可以使用的政策信息(注意是可使用的，不是经过门槛匹配的，也没有经过商品范围匹配)")
  private Set<SalePolicyVo> initSalePolicys;
  /**
   * 当前活动政策执行所涉及的经销商信息（上下文初始化后不可变）
   */
  @ApiModelProperty("当前活动政策执行所涉及的经销商信息（上下文初始化后不可变）")
  private final String tenantCode;
  /**
   * 当前活动政策执行所涉及的客户信息（上下文初始化后不可变）
   */
  @ApiModelProperty("当前活动政策执行所涉及的客户信息（上下文初始化后不可变）")
  private final String customerCode;
  /**
   * 该属性表示在进行优惠政策执行后的分摊过程中，是否进行揉价</br>
   * 所谓揉价，就是在分摊时，考虑赠品的优惠幅度，将优惠后的金额分摊到赠品上。
   */
  private final boolean kneading; 
  /**
   * 是否正式执行（false：预执行，true：正式执行）
   */
  private final boolean formal;
  /**
   * 如果是正式执行，则这里记录了优惠政策涉及的业务单据信息
   */
  private String relevanceCode;
  
  // 以下属性和优惠汇总结果相关 =======
  
  /**
   * 该值表示在正确完成计算后，总体优惠金额差值
   */
  @ApiModelProperty("该值表示在正确完成计算后，总体优惠金额差值")
  private BigDecimal totalDiscountAmount = BigDecimal.ZERO;
  /**
   * 该值表示在正确完成计算后，总体优惠赠品的汇总
   */
  @ApiModelProperty("该值表示在正确完成计算后，总体优惠赠品的汇总")
  private List<GiftResultInfo> totalGiftResultInfos = Lists.newArrayList();
  /**
   * 整张单据的初始价格（即所有计算开始前的价格）
   */
  @ApiModelProperty("整张单据的初始价格（即所有计算开始前的价格）")
  private BigDecimal initTotalAmount;
  /**
   * 整张单据的初始数量（即所有计算开始前的数量）
   */
  @ApiModelProperty("整张单据的初始数量（即所有计算开始前的数量）")
  private Integer initTotalNumber;
  
  // 以下属性和商品优惠结果中，以商品维度（视角）出发进行描述的优惠结果内容相关 =======
  /**
   * 商品理论上已匹配的优惠政策基本信息（Key,商品业务编号，Value：优惠政策编号，注意，这里只是优惠政策的信息(业务编号)，并不是每一个商品针对优惠政策的执行结果（也并不是说一定会执行这个优惠政策），后者请参见productPolicyStepResultMapping属性
   */
  @ApiModelProperty("商品已匹配的优惠政策基本信息（Key,商品业务编号，Value：优惠政策编号，注意，这里只是优惠政策的信息(业务编号)，并不是每一个商品针对优惠政策的执行结果，后者请参见productPolicyStepResultMapping属性")
  private Map<String , Set<String>> matchedProductInfoMapping;
  /**
   * 商品每一次执行阶梯计算后的结果信息（无论商品是否满足优惠政策门槛，并进行了实际的优惠），每一个initPolicyProducts集合中的商品，在这里都至少有一条信息（既这个商品的初始优惠步进信息）
   */
  @ApiModelProperty("商品每一次执行阶梯计算后的结果信息（无论商品是否满足优惠政策门槛，并进行了实际的优惠），每一个initPolicyProducts集合中的商品，在这里都至少有一条信息（既这个商品的初始优惠步进信息）")
  private Map<String , Deque<ProductPolicyStepResult>> productPolicyStepResultMapping;
  
  // 以下属性和优惠结果中，以优惠信息维度（视角）出发进行描述的优惠结果内容相关 ======
  
  /**
   * 已被排除的优惠策略，包括商品优惠和整单优惠（Key：优惠策略编号；Value：排除原因的中文描述）
   */
  @ApiModelProperty("已被排除的优惠策略（Key：优惠政策业务编号；Value：排除原因的中文描述）")
  private Map<String , String> excludedPolicyMapping;
  /**
   * 已匹配的优惠政策（包括商品优惠和整单优惠）业务编号信息（注意，这里只有政策信息的业务编号，并不是每一个优惠政策的每一次执行结果，后者请参见matchedPolicyResultMapping）
   */
  @ApiModelProperty("已匹配的优惠政策（包括商品优惠和整单优惠）业务编号信息（注意，这里只有政策信息的业务编号，并不是每一个优惠政策的每一次执行结果，后者请参见matchedPolicyResultMapping）")
  private Set<String> matchedPolicyInfos;
  /**
   * 已匹配的并执行完成的优惠策略（包括商品优惠和整单优惠）结果步进，
   * 它与matchedPolicyInfos的关系是，只有matchedPolicyInfos存在数据，这里才会有步进信息
   */
  private Deque<PolicyStepResult> policyStepResults;
  
  // 以下属性和优惠结果中，以限量信息维度（视角）出发进行描述的优惠结果内容相关 ======
  
  // =========== 以下是对各种计算结果数据的添加操作
  
  /**
   * 将存在于initSalePolicys中的副本添加到不能销售的优惠政策集合中，并附上原因
   * @param excludedPolicy
   * @param message
   */
  public void addExcludedPolicy(SalePolicyVo excludedPolicy , String message) {
    Validate.notNull(excludedPolicy , "添加排除政策时，政策信息必须填写");
    Validate.notBlank(message , "添加排除政策时，排除原因必须填写");
    Validate.isTrue(this.initSalePolicys.contains(excludedPolicy) , "添加排除政策时，政策信息必须已存在于initSalePolicys清单中");
    this.excludedPolicyMapping.put(excludedPolicy.getSalePolicyCode() , message);
  }
  
  /**
   * 该方法用于排除指定本品和其已匹配优惠政策的的关联关系。</br>
   * 排除后，如果一个优惠政策不再和任何本品关联，则该优惠政策将计入excludedPolicyMapping集合中，
   * 并且排除原因会记为“优惠政策被排除”</p>
   * 
   * 注意，该方法只能在执行上下文创建后，但还未执行任何优惠政策步进前，进行操作
   * 
   * @param productCode
   * @param policyCode
   */
  public void excludeMatchedPolicyMapping(String productCode , String policyCode) {
    // 注意，该方法只能在执行上下文创建后，但还未执行任何优惠政策步进前，进行操作
    Validate.isTrue(this.policyStepResults.size() == 1 , "因为优惠政策已经开始执行步进，所以不能在修正本品和优惠政策的匹配关系，如果要进行修改请调用resetExcludedPolicy方法");
    
    // 1、====== 边界校验并移除绑定关系
    // 指定的本品编号和优惠政策编号必须都存在
    Validate.notBlank(productCode , "进行本品和已匹配优惠政策的解绑时，本品业务编码必须传入!!");
    Validate.notBlank(policyCode , "进行本品和已匹配优惠政策的解绑时，优惠政策编码必须传入!!");
    Validate.isTrue(!CollectionUtils.isEmpty(productPolicyStepResultMapping.get(productCode)) , "进行本品和已匹配优惠政策的解绑时，本品信息应该是参与本次优惠政策执行的本品!!");
    Validate.isTrue(matchedPolicyInfos.contains(policyCode) , "进行本品和已匹配优惠政策的解绑时，指定的优惠政策应该是以正确匹配的优惠政策!!");
    Set<String> matchedPolicies = this.matchedProductInfoMapping.get(productCode);
    Validate.isTrue(!CollectionUtils.isEmpty(matchedPolicies) , "进行本品和已匹配优惠政策的解绑时，指定的本品应该存在已匹配的优惠政策");
    Validate.isTrue(matchedPolicies.contains(policyCode) , "进行本品和已匹配优惠政策的解绑时，指定的本品和指定的优惠政策，应该存在匹配关系，请检查!!");
    matchedPolicies.remove(policyCode);
    
    // 2、===== 如果指定的优惠政策一不被所有的本品匹配，则设置该优惠政策为不匹配的优惠政策
    for (Map.Entry<String,Set<String>> matchedProductItem : this.matchedProductInfoMapping.entrySet()) {
      Set<String> matchedPolicyItems = matchedProductItem.getValue();
      if(matchedPolicyItems.contains(policyCode)) {
        return;
      }
    }
    // 如果执行到这里，说明从匹配的优惠政策清单，移除该优惠政策
    this.excludedPolicyMapping.put(policyCode, "优惠政策被操作者指定排除");
  }
  
  /**
   * 设定一个已经匹配了的优惠政策为“不匹配”，并设定不匹配的原因。</br>
   * 调用这个方法的最常见情况，就是优惠政策执行完成后，比满足优惠政策设定的某个或者某几个限量政策。</p>
   * 
   * 这个过程相当于回滚动作，要求这个优惠政策是最近才执行的。包括回退已经执行的本品状态、赠品状态、整体优惠执行状态等等。
   * 
   * @param policyCode
   * @param requestRollbackProductCodes 由外部指定的，本次要回滚的特定产品编号（特别是在重复执行同一类型优惠政策模板时，并不是所有物资都需要回滚）
   * @param erroeMsg
   */
  public void resetExcludedPolicy(String policyCode , Set<String> requestRollbackProductCodes , String erroeMsg) {
    // 一定是先完成验证
    Validate.notBlank(policyCode , "回退优惠政策的执行结果时，未发现指定的优惠政策（%s），请检查!!" , policyCode);
    Validate.isTrue(initSalePolicys.stream().filter(item -> StringUtils.equals(policyCode, item.getSalePolicyCode())).count() != 0 , "回退优惠政策的执行结果时，未发现指定的优惠政策（%s），请检查!!" , policyCode);
    Validate.notBlank(erroeMsg , "回退优惠政策执行结果时，必须说明优惠政策回退的原因");
    // 主要：要找出这个优惠政策被哪些本品所匹配，后续逻辑要用
    Set<Entry<String, Set<String>>> matchedProductInfoEntrys = this.matchedProductInfoMapping.entrySet();
    Set<String> mustRollbackProductCodes = Sets.newLinkedHashSet();
    for (Entry<String, Set<String>> matchedProductInfoMappingItem : matchedProductInfoEntrys) {
      Set<String> matchedPolicyCodes = matchedProductInfoMappingItem.getValue();
      if(matchedPolicyCodes.contains(policyCode)) {
        mustRollbackProductCodes.add(matchedProductInfoMappingItem.getKey());
      }
    }
    // 如果requestRollbackProductCodes存在值，那么只有和mustRollbackProductCodes的“交集”需要回滚
    if(!CollectionUtils.isEmpty(requestRollbackProductCodes)) {
      mustRollbackProductCodes = Sets.intersection(requestRollbackProductCodes, mustRollbackProductCodes);
    }
    // 验证需要回退的本品详情步进，是否是最新步进
    for (String mustRollbackProductCode : mustRollbackProductCodes) {
      Deque<ProductPolicyStepResult> productPolicyStepResults = productPolicyStepResultMapping.get(mustRollbackProductCode);
      Validate.isTrue(StringUtils.equals(productPolicyStepResults.peek().getSalePolicyCode(), policyCode) , "无法回退本品(%s)针对优惠政策的执行情况，请检查" , mustRollbackProductCode);
    }
    // 验证需要回退的整体步进
    PolicyStepResult lastPolicyStepResult = policyStepResults.peek();
    Validate.isTrue(StringUtils.equals(lastPolicyStepResult.getSalePolicyCode(), policyCode) , "无法回退整体优惠步进的执行情况，因为它不是最后一个执行的优惠政策，请检查");
    
    /* 
     * 操作过程为：
     * 1、增加excludedPolicyMapping中的信息
     * 2、回退productPolicyStepResultMapping中，关于每个本品针对这个优惠政策的执行步进（要求这个优惠政策必须是最近执行的，否则要报错）
     * 3、回退policyStepResults中，关于本次优惠政策执行的步进（要求这个优惠政策必须是最近执行的，否则要报错）
     * 4、重新设定totalDiscountAmount、totalGiftResultInfos 这两个值
     * 
     * 注意：一定是先完成正确定验证，再进行删除。因为如果回滚到一半才发现数据有问题，就会造成数据失真影响后续的优惠政策计算
     * */
    // 1、=======
    this.excludedPolicyMapping.put(policyCode, erroeMsg);
    // 2、=====
    Map<String , Deque<ProductPolicyStepResult>> productPolicyStepResultMappingEntrys = this.productPolicyStepResultMapping;
    for (String mustRollbackProductCode : mustRollbackProductCodes) {
      Deque<ProductPolicyStepResult> productPolicyStepResults = productPolicyStepResultMappingEntrys.get(mustRollbackProductCode);
      productPolicyStepResults.poll();
    }
    // 3、=====
    policyStepResults.poll();
    BigDecimal preSubtotal = lastPolicyStepResult.getPreSubtotal();
    BigDecimal lastSubtotal = lastPolicyStepResult.getLastSubtotal();
    // 这个是产生的差价（后续也要回退）
    BigDecimal discountAmount = lastSubtotal.subtract(preSubtotal);
    // 4、=====
    this.totalDiscountAmount = this.totalDiscountAmount.subtract(discountAmount);
    this.totalGiftResultInfos = lastPolicyStepResult.getPreGifts();
  }
  
  /**
   * 查询得到当前已被排除的活动政策数量
   * @return
   */
  public Integer sizeOfExcludedPolices() {
    return this.excludedPolicyMapping.size();
  }
  
  /**
   * 向优惠政策上下文添加一个满足匹配要求的整单优惠信息，只有添加了这个信息，才能在后续计算中逐步添加这个优惠政策的详细优惠结果</p>
   * 注意，调用该方法将会使上下文认为其initPolicyProducts集合中所有的商品信息都匹配当前优惠政策
   * @param matchedPolicy 只能是整单优惠信息，且必须存在于initSalePolicys中
   */
  public void addNewMathedPolicy(SalePolicyVo matchedPolicy) {
    Validate.notNull(matchedPolicy , "添加匹配的整单优惠时，必须传入优惠信息对象，请检查!!");
    Validate.isTrue(matchedPolicy.getWholePolicy() , "添加匹配的整单优惠时，传入的优惠信息必须是整单优惠，请检查!!");
    Validate.isTrue(initSalePolicys.contains(matchedPolicy) , "添加匹配的整单优惠时，传入的优惠信息必须存在于上下文初始化时的集合中(%s)，请检查!!" , matchedPolicy.getSalePolicyCode());
    // 不能存在于excludedPolicyMapping集合中
    if(!CollectionUtils.isEmpty(excludedPolicyMapping)) {
      Validate.isTrue(excludedPolicyMapping.get(matchedPolicy.getSalePolicyCode()) == null , "添加匹配的整单优惠时，优惠政策不能存在于已经被配出的记录信息中");
    }
    this.matchedPolicyInfos.add(matchedPolicy.getSalePolicyCode());
    
    // 接着设定所有的参与优惠的商品，均可匹配该整单优惠政策
    if(!CollectionUtils.isEmpty(this.initPolicyProducts)) {
      for (SalePolicyConProduct policyConProduct : this.initPolicyProducts) {
        this.setMatchedProductInfoMapping(policyConProduct.getProductCode(), matchedPolicy.getSalePolicyCode());
      }
    }
  }
  
  /**
   * 向优惠政策上下文添加一个满足匹配要求的商品优惠信息，只有添加了这个信息，才能在后续计算中逐步添加这个优惠政策的详细优惠结果
   * @param matchedPolicy 只能是商品优惠信息，且必须存在于initSalePolicys中
   * @param matchedProductCodes 这个优惠政策最终匹配的商品业务编号信息
   */
  public void addNewMathedPolicy(SalePolicyVo matchedPolicy , String[] matchedProductCodes) {
    Validate.notNull(matchedPolicy , "添加匹配的整单优惠时，必须传入优惠信息对象，请检查!!");
    Validate.isTrue(initSalePolicys.contains(matchedPolicy) , "添加匹配的商品优惠时，传入的优惠信息必须存在于上下文初始化时的集合中(%s)，请检查!!" , matchedPolicy.getSalePolicyCode());
    // 不能存在于excludedPolicyMapping集合中
    if(!CollectionUtils.isEmpty(excludedPolicyMapping)) {
      Validate.isTrue(excludedPolicyMapping.get(matchedPolicy.getSalePolicyCode()) == null , "添加匹配的整单优惠时，优惠政策不能存在于已经被排除的记录信息中");
    }
    Validate.isTrue(matchedProductCodes != null && matchedProductCodes.length > 0 , "添加匹配的商品优惠时，至少需要传入一条商品业务编码信息");
    int disCount = (int)Arrays.stream(matchedProductCodes).distinct().count();
    Validate.isTrue(disCount == matchedProductCodes.length , "添加匹配的商品优惠时，商品业务编码信息不能重复，请检查!!");
    
    // 进行新的匹配索引的添加（包括匹配优惠政策维度的索引和匹配商品本品维度的索引）
    this.matchedPolicyInfos.add(matchedPolicy.getSalePolicyCode());
    for (String matchedProductCode : matchedProductCodes) {
      this.setMatchedProductInfoMapping(matchedProductCode, matchedPolicy.getSalePolicyCode());
    }
  }
  
  /**
   * 该私有方法，为指定的商品业务编号设定匹配的优惠政策信息
   */
  private void setMatchedProductInfoMapping(String productCode , String policyCode) {
    Set<String> matchedPolicyCodes = this.matchedProductInfoMapping.get(productCode);
    if(!CollectionUtils.isEmpty(matchedPolicyCodes)) {
      matchedPolicyCodes.add(policyCode);
    } else {
      synchronized (AbstractPolicyExecuteContext.class) {
        while(this.matchedProductInfoMapping.get(productCode) == null) {
          matchedPolicyCodes = Sets.newLinkedHashSet();
          this.matchedProductInfoMapping.put(productCode, matchedPolicyCodes);
          matchedPolicyCodes.add(policyCode);
        }
      }
    }
  }
  
  /**
   * 向优惠政策计算上下文中，增加一条优惠政策计算结果，这条计算结果包括了每个参与优惠的商品本品的变化情况，和单据整体状态的变化情况（注意，该方法不能用于本品优惠步进的初始化）
   * @param productStepResults 
   */
  public void addPolicyStepResult(Set<ProductPolicyStepResult> productStepResults) {
    if(CollectionUtils.isEmpty(productStepResults)) {
      return;
    }
    // 这些产品步进明细，只能设定一致的优惠政策信息salePolicyCode
    long salePolicyCodeCount = productStepResults.stream().map(ProductPolicyStepResult::getSalePolicyCode).filter(item-> Objects.nonNull(item)).distinct().count();
    Validate.isTrue(salePolicyCodeCount == 1l , "这些产品步进明细，只能设定一致的优惠政策信息salePolicyCode，请检查值的传递!!");
    // 这些产品步进明细，只能设定一致的优惠政策执行器executorCode
    long executorCodeCount = productStepResults.stream().map(ProductPolicyStepResult::getExecuteCode).filter(item-> Objects.nonNull(item)).distinct().count();
    Validate.isTrue(executorCodeCount == 1l , "这些产品步进明细，只能设定一致的优惠政策执行器executorCode，请检查值的传递!!");
    
    /*
     * 处理过程为：
     * 1、首先根据每一个productStepResults，来进行优惠政策中matchedProductInfoMapping集合里每一个本品优惠步进的处理
     * 注意，要验证每一个发生优惠步进的本品是否在matchedProductInfoMapping集合中
     * 2、然后在根据本次优惠后的整体情况，进行整体优惠情况的处理
     * */
    
    // 1、======
    String currentSalePolicyCode = null;
    String currentExecutorCode = null;
    for (ProductPolicyStepResult newProductPolicyStepResult : productStepResults) {
      String salePolicyCode = newProductPolicyStepResult.getSalePolicyCode();
      Validate.notBlank(salePolicyCode , "这批优惠步进设计的本品信息，对应的优惠政策业务编号salePolicyCode不能为空，请检查!!");
      currentSalePolicyCode = salePolicyCode;
      String executorCode = newProductPolicyStepResult.getExecuteCode();
      Validate.notBlank(executorCode , "这批优惠步进设计的本品信息，对应的优惠政策执行器编号executorCode不能为空，请检查!!");
      currentExecutorCode = executorCode;
      String productCode = newProductPolicyStepResult.getProductCode();
      Validate.notBlank(productCode , "必须传入本品业务编号信息");
      Set<String> matchedSalePolicyCodes = this.matchedProductInfoMapping.get(productCode);
      Validate.isTrue(!CollectionUtils.isEmpty(matchedSalePolicyCodes) , "传入的优惠政策步进本品信息并不存在于最初的优惠匹配集合（matchedProductInfoMapping）中，请检查!!");
      Validate.isTrue(matchedSalePolicyCodes.contains(salePolicyCode) , "传入的优惠政策步进本品信息并没有在最初预判定中匹配当前的优惠政策[%s]，请检查!!" , salePolicyCode);
      StepType stepType = newProductPolicyStepResult.getStepType();
      Validate.isTrue(stepType != null && stepType != StepType.INIT , "必须传入优惠本品步进的性质(StepType)");
      
      // 继续验证步进情况
      Deque<ProductPolicyStepResult> productPolicyStepResultQueue = productPolicyStepResultMapping.get(productCode);
      ProductPolicyStepResult lastProductPolicyStepResult = productPolicyStepResultQueue.peek();
      List<GiftResultInfo> giftResultInfos = lastProductPolicyStepResult.getLastGifts(); 
      BigDecimal lastPrices = lastProductPolicyStepResult.getLastPrices();
      BigDecimal lastSubtotal = lastProductPolicyStepResult.getLastSubtotal();
      BigDecimal lastSurplusTotalAmount = lastProductPolicyStepResult.getLastSurplusTotalAmount();
      Integer lastSurplusTotalNumber = lastProductPolicyStepResult.getLastSurplusTotalNumber();
      // 新增加的步进信息，其初始状态要和这个本品已有步进的最后值一致
      List<GiftResultInfo> newPreGiftResultInfos = newProductPolicyStepResult.getPreGifts();
      BigDecimal newPrePrices = newProductPolicyStepResult.getPrePrices();
      BigDecimal newPreSubtotal = newProductPolicyStepResult.getPreSubtotal();
      BigDecimal newPreSurplusTotalAmount = newProductPolicyStepResult.getPreSurplusTotalAmount();
      Integer newPreSurplusTotalNumber = newProductPolicyStepResult.getPreSurplusTotalNumber();
      Validate.isTrue(this.validateGiftMatch(giftResultInfos, newPreGiftResultInfos, newProductPolicyStepResult.getLastGifts()) , "在进行本品优惠情况步进添加时，发现两次步进的赠品初始情况不匹配，请检查!!");
      Validate.isTrue(lastPrices.compareTo(newPrePrices) == 0 , "在进行本品优惠情况步进添加时，发现两次步进的单价初始情况不匹配，请检查!!");
      Validate.isTrue(lastSubtotal.compareTo(newPreSubtotal) == 0 , "在进行本品优惠情况步进添加时，发现两次步进的小计价格初始情况不匹配，请检查!!");
      Validate.isTrue(lastSurplusTotalAmount.compareTo(newPreSurplusTotalAmount) == 0 , "在进行本品优惠情况步进添加时，发现两次步进的未优惠金额初始情况不匹配，请检查!!");
      Validate.isTrue(lastSurplusTotalNumber.compareTo(newPreSurplusTotalNumber) == 0 , "在进行本品优惠情况步进添加时，发现两次步进的未优惠数量初始情况不匹配，请检查!!");
      
      // 开始添加步进
      newProductPolicyStepResult.setInitNumbers(lastProductPolicyStepResult.getInitNumbers());
      newProductPolicyStepResult.setInitPrices(lastProductPolicyStepResult.getInitPrices());
      newProductPolicyStepResult.setInitSubtotal(lastProductPolicyStepResult.getInitSubtotal());
      productPolicyStepResultQueue.push(newProductPolicyStepResult);
    }
    
    // 2、=====
    Validate.isTrue(matchedPolicyInfos.contains(currentSalePolicyCode) , "在进行优惠政策步进添加时，传入的优惠政策信息并没有在最初预判定中匹配当前的优惠政策（%），请检查!!" , currentSalePolicyCode);
    Map<String, SalePolicyVo> initSalePolicyMapping = this.initSalePolicys.stream().collect(Collectors.toMap(SalePolicyVo::getSalePolicyCode, item -> item));
    SalePolicyVo currentSalePolicy = initSalePolicyMapping.get(currentSalePolicyCode);
    Validate.notNull(currentSalePolicy , "在进行优惠政策步进添加时，未找到指定的优惠政策信息，请检查!!");
    StepType stepType = currentSalePolicy.getWholePolicy()?StepType.POLICY:StepType.PRODUCT;
    PolicyStepResult lastPolicyStepResult = this.policyStepResults.peek();
    List<GiftResultInfo> lastGiftResultInfos = lastPolicyStepResult.getLastGifts();
    BigDecimal lastSubtotal = lastPolicyStepResult.getLastSubtotal();
    BigDecimal lastSurplusTotalAmount = lastPolicyStepResult.getLastSurplusTotalAmount();
    Integer lastSurplusTotalNumber = lastPolicyStepResult.getLastSurplusTotalNumber();
    // 新增加的步进信息，其初始状态要和整个优惠政策已有步进的最后值一致（所以以下信息不会改变）
    PolicyStepResult newPolicyStepResult = new PolicyStepResult();
    newPolicyStepResult.setInitNumbers(lastPolicyStepResult.getInitNumbers());
    newPolicyStepResult.setInitSubtotal(lastPolicyStepResult.getInitSubtotal());
    newPolicyStepResult.setSalePolicyCode(currentSalePolicyCode);
    newPolicyStepResult.setExecutorCode(currentExecutorCode);
    newPolicyStepResult.setStepType(stepType);
    newPolicyStepResult.setPreGifts(lastPolicyStepResult.getLastGifts());
    newPolicyStepResult.setPreSubtotal(lastPolicyStepResult.getLastSubtotal());
    newPolicyStepResult.setPreSurplusTotalAmount(lastPolicyStepResult.getLastSurplusTotalAmount());
    newPolicyStepResult.setPreSurplusTotalNumber(lastPolicyStepResult.getLastSurplusTotalNumber());
    newPolicyStepResult.setPreGiftEnjoyedTotalNumber(lastPolicyStepResult.getLastGiftEnjoyedTotalNumber());
    newPolicyStepResult.setPreGiftEnjoyedTotalAmount(lastPolicyStepResult.getLastGiftEnjoyedTotalAmount());
    /*
     * 以下值可能会发生变化
     * 2.1、重新根据每个本品最新的状态，汇总赠品信息，并重新根据赠品累计的数量和小计价格计算赠品
     * 2.2、重新根据每个本品最新的“还未享受优惠的数量和金额”，计算整个优惠过程“还未享受优惠的数量和金额”
     * 重新根据每个本品的最新小计价格，计算整个优惠过程的小计优惠价格
     * 2.3、最后对形成的最新步进的状态情况进行验证
     * */
    // summaryGiftStatuMapping 代表汇总的最新的各个赠品的数量和小计金额
    // 其中Value是一个Triple对象，这个对象L值表示赠品的中文信息，M值表示汇总的数量；R值表示汇总的小计金额
    Map<String , Triple<String , Integer , BigDecimal>> summaryGiftStatuMapping = Maps.newLinkedHashMap();
    // summarySurplusTotalAmount 代表汇总的各个本品还未享受的小计金额
    // summarySurplusTotalNumber 代表汇总的各个本品还未享受的小计数量
    // summarySubTotalAmount 代表汇总的各个本品的小计价格
    // summaryGiftEnjoyedTotalNumber 代表汇总的所有已享受的赠品数量
    // summaryGiftsEnjoyedTotalAmount 代表汇总的所有已享受的赠品金额
    BigDecimal summarySurplusTotalAmount = BigDecimal.ZERO;
    Integer summarySurplusTotalNumber = 0;
    BigDecimal summarySubTotalAmount = BigDecimal.ZERO;
    Integer summaryGiftEnjoyedTotalNumber = 0;
    BigDecimal summaryGiftsEnjoyedTotalAmount = BigDecimal.ZERO;
    for (Map.Entry<String,Deque<ProductPolicyStepResult>> productPolicyStepResult : productPolicyStepResultMapping.entrySet()) {
      ProductPolicyStepResult productPolicyStepResultItem = productPolicyStepResult.getValue().peek();
      List<GiftResultInfo> giftResultInfos = productPolicyStepResultItem.getLastGifts();
      // 2.1、======
      if(!CollectionUtils.isEmpty(giftResultInfos)) {
        for (GiftResultInfo lastGiftResultInfo : giftResultInfos) {
          // 累加数量和小计金额
          String giftProductCode = lastGiftResultInfo.getProductCode();
          Triple<String , Integer , BigDecimal> exsitTriple = null;
          Triple<String , Integer , BigDecimal> newTriple = null;
          if((exsitTriple = summaryGiftStatuMapping.get(giftProductCode)) == null) {
            newTriple = Triple.of(lastGiftResultInfo.getProductName() , lastGiftResultInfo.getQuantity(), lastGiftResultInfo.getSubtotalAmount());
          } else {
            Integer newQuantity = exsitTriple.getMiddle() + lastGiftResultInfo.getQuantity();
            BigDecimal newSubtotalAmount = exsitTriple.getRight().add(lastGiftResultInfo.getSubtotalAmount());
            newTriple = Triple.of(lastGiftResultInfo.getProductName() , newQuantity , newSubtotalAmount);
          }
          summaryGiftStatuMapping.put(giftProductCode, newTriple);
        }
      }
      // 2.2 ========
      summarySurplusTotalAmount = summarySurplusTotalAmount.add(productPolicyStepResultItem.getLastSurplusTotalAmount());
      summarySurplusTotalNumber = summarySurplusTotalNumber + productPolicyStepResultItem.getLastSurplusTotalNumber();
      summarySubTotalAmount = summarySubTotalAmount.add(productPolicyStepResultItem.getLastSubtotal());
      summaryGiftEnjoyedTotalNumber = summaryGiftEnjoyedTotalNumber + (productPolicyStepResultItem.getLastGiftEnjoyedTotalNumber() == null?0:productPolicyStepResultItem.getLastGiftEnjoyedTotalNumber());
      summaryGiftsEnjoyedTotalAmount = summaryGiftsEnjoyedTotalAmount.add(productPolicyStepResultItem.getLastGiftEnjoyedTotalAmount() == null?BigDecimal.ZERO:productPolicyStepResultItem.getLastGiftEnjoyedTotalAmount());
    }
    // 根据2.1得到的汇总信息，重新得到最新的赠品汇总信息
    if(!CollectionUtils.isEmpty(summaryGiftStatuMapping)) {
      List<GiftResultInfo> newGiftResultInfos = Lists.newArrayList();
      for (Map.Entry<String,Triple<String,Integer,BigDecimal>> summaryGiftResultInfo : summaryGiftStatuMapping.entrySet()) {
        Triple<String, Integer, BigDecimal> giftItem = summaryGiftResultInfo.getValue();
        GiftResultInfo newGiftResultInfo = new GiftResultInfo();
        newGiftResultInfo.setProductCode(summaryGiftResultInfo.getKey());
        newGiftResultInfo.setProductName(giftItem.getLeft());
        newGiftResultInfo.setQuantity(giftItem.getMiddle());
        newGiftResultInfo.setSubtotalAmount(giftItem.getRight());
        newGiftResultInfos.add(newGiftResultInfo);
      }
      newPolicyStepResult.setLastGifts(newGiftResultInfos);
    } else {
      newPolicyStepResult.setLastGifts(Lists.newArrayList());
    }
    newPolicyStepResult.setLastSubtotal(summarySubTotalAmount);
    newPolicyStepResult.setLastSurplusTotalAmount(summarySurplusTotalAmount);
    newPolicyStepResult.setLastSurplusTotalNumber(summarySurplusTotalNumber);
    newPolicyStepResult.setLastGiftEnjoyedTotalNumber(summaryGiftEnjoyedTotalNumber);
    newPolicyStepResult.setLastGiftEnjoyedTotalAmount(summaryGiftsEnjoyedTotalAmount);
    // 2.3、========
    List<GiftResultInfo> newPreGiftResultInfos = newPolicyStepResult.getPreGifts();
    BigDecimal newPreSubtotal = newPolicyStepResult.getPreSubtotal();
    BigDecimal newPreSurplusTotalAmount = newPolicyStepResult.getPreSurplusTotalAmount();
    Integer newPreSurplusTotalNumber = newPolicyStepResult.getPreSurplusTotalNumber();
    Validate.isTrue(this.validateGiftMatch(lastGiftResultInfos, newPreGiftResultInfos, newPolicyStepResult.getLastGifts()) , "在进行优惠政策步进添加时，发现两次步进的赠品初始情况不匹配，请检查!!");
    Validate.isTrue(lastSubtotal.compareTo(newPreSubtotal) == 0 , "在进行优惠政策步进添加时，发现两次步进的小计价格初始情况不匹配，请检查!!");
    Validate.isTrue(lastSurplusTotalAmount.compareTo(newPreSurplusTotalAmount) == 0 , "在进行优惠政策步进添加时，发现两次步进的未优惠金额初始情况不匹配，请检查!!");
    Validate.isTrue(lastSurplusTotalNumber.compareTo(newPreSurplusTotalNumber) == 0 , "在进行优惠政策步进添加时，发现两次步进的未优惠数量初始情况不匹配，请检查!!");
    // 最终进行新的全局步进的添加
    newPolicyStepResult.setInitNumbers(lastPolicyStepResult.getInitNumbers());
    newPolicyStepResult.setInitSubtotal(lastPolicyStepResult.getInitSubtotal());
    this.policyStepResults.push(newPolicyStepResult);
  }
  
  /**
   * 该逻辑用于完成两次步进中赠品信息变化的基本验证
   * @param lastGiftResultInfos 最后一次步进优惠后的赠品信息
   * @param newPreGiftResultInfos 新的步进优惠前的赠品信息
   * @param newLastGiftResultInfos 新的步进优惠后的赠品信息
   * @return
   */
  private boolean validateGiftMatch(List<GiftResultInfo> lastGiftResultInfos , List<GiftResultInfo> newPreGiftResultInfos , List<GiftResultInfo> newLastGiftResultInfos) {
    /*
     * 赠品累加情况验证最基本的验证情况是：
     * 1、原有最后一次步进在结束时的赠品情况应该和新步进开始时的赠品情况一致
     * 2、新步进的开始时的赠品情况只能是新步进结束时赠品情况的子级
     * 且后者每个赠品的数量只能大于或者等于前者中的每个赠品数量
     * */
    // 1、=======
    if(CollectionUtils.isEmpty(lastGiftResultInfos) && CollectionUtils.isEmpty(newPreGiftResultInfos)) {
      return true;
    } 
    if(!CollectionUtils.isEmpty(lastGiftResultInfos) && !CollectionUtils.isEmpty(newPreGiftResultInfos)
         && lastGiftResultInfos.size() == newPreGiftResultInfos.size()) {
      return true;
    }
    // 2、======
    if(CollectionUtils.isEmpty(newLastGiftResultInfos)) {
      newLastGiftResultInfos = Lists.newArrayList();
    }
    if(CollectionUtils.isEmpty(newPreGiftResultInfos)) {
      newPreGiftResultInfos = Lists.newArrayList();
    }
    Map<String , GiftResultInfo> newLastGiftResultInfoMappings = newLastGiftResultInfos.stream().collect(Collectors.toMap(GiftResultInfo::getProductCode, item -> item));
    Map<String , GiftResultInfo> newPreGiftResultInfoMappings = newPreGiftResultInfos.stream().collect(Collectors.toMap(GiftResultInfo::getProductCode, item -> item));
    Set<String> newLastGiftResultInfoCodes = newLastGiftResultInfoMappings.keySet();
    Set<String> newPreGiftResultInfoCodes = newPreGiftResultInfoMappings.keySet();
    // 要么两个集合一致，要么newPreGiftResultInfoMappings是newLastGiftResultInfoMappings的子级
    Set<String> intersectionGiftProductCodes = Sets.intersection(newPreGiftResultInfoCodes , newLastGiftResultInfoCodes);
    // 如果条件成立，说明newPreGiftResultInfoMappings是newLastGiftResultInfoMappings的子级
    if (!CollectionUtils.isEmpty(intersectionGiftProductCodes) && intersectionGiftProductCodes.size() == newPreGiftResultInfoCodes.size()) {
      return true;
    } else if(newLastGiftResultInfoCodes.size() == newPreGiftResultInfoCodes.size()) {
      return true;
    }
    return false;
  }
  
  public ProductPolicyStepResult findLastProductStepResultByProductCode(String productCode) {
    if(StringUtils.isBlank(productCode)) {
      return null;
    }
    Deque<ProductPolicyStepResult> productPolicyStepResults = this.productPolicyStepResultMapping.get(productCode);
    if(CollectionUtils.isEmpty(productPolicyStepResults)) {
      return null;
    }
    return productPolicyStepResults.peek();
  }
  
  public PolicyStepResult findLastStepResult() {
    return this.policyStepResults.peek();
  }
  
  /**
   * 根据一个商品信息，查询得到这个商品信息在本次优惠执行过程中的每一个优惠步骤</br>
   * 需要注意的是，调用者不应该改变返回值集合中的数据，否则执行上下文的结果正确性会受到影响
   * @param productCode
   * @return
   */
  public Deque<ProductPolicyStepResult> findPolicyStepResultsByProductCode(String productCode) {
    if(StringUtils.isBlank(productCode)) {
      return null;
    }
    Deque<ProductPolicyStepResult> productPolicyStepResults = this.productPolicyStepResultMapping.get(productCode);
    return productPolicyStepResults;
  }
  
  /**
   * 该方法按照执行的优惠政策业务编号，查询当前优惠执行过程中，有多少本品（业务编号）参与了这个优惠政策
   * @param policyCode 指定的优惠政策
   * @return
   */
  public Set<String> findProductCodesByPolicyCode(String policyCode) {
    if(StringUtils.isBlank(policyCode)) {
      return null;
    }
    // 如果这个优惠政策业务编码已经被归纳到了不匹配的优惠政策中，则也返回null
    if(StringUtils.isNotBlank(this.excludedPolicyMapping.get(policyCode))) {
      return null;
    }
    
    Set<String> productCodes = Sets.newLinkedHashSet();
    for (Map.Entry<String,Set<String>> matchedProductInfoMappingItem : this.matchedProductInfoMapping.entrySet()) {
      String productCode = matchedProductInfoMappingItem.getKey();
      Set<String> salepolicyCodes = matchedProductInfoMappingItem.getValue();
      // 如果条件成立，说明这个本品是匹配了需要查询的优惠政策的
      if(salepolicyCodes.contains(policyCode)) {
        productCodes.add(productCode);
      }
    }
    return productCodes;
  }
  
  public Set<String> findPolicyCodesByProductCode(String productCode) {
    if(StringUtils.isBlank(productCode)) {
      return null;
    }
    return this.matchedProductInfoMapping.get(productCode);
  }
  
  /**
   * 按照指定的商品编码查询，这个商品应该参与但是由于门槛未到等错误原因，最终未能参与的优惠政策（人为排除的优惠政策不在此列）
   * @param productCode 指定的商品编码信息
   * @return
   */
  public Map<String , String> findExcludedPolicyByProductCode(String productCode) {
    /*
     * 实际上这就是基于matchedProductInfoMapping集合和productPolicyStepResultMapping集合中，
     * 关于指定商品编码信息的差集，进行excludedPolicyMapping集合信息提取的过程
     * */
    Set<String> matchedPolicyCodes = this.matchedProductInfoMapping.get(productCode);
    // 说明可能传入的productCode就是错误的
    if(CollectionUtils.isEmpty(matchedPolicyCodes)) {
      return Maps.newLinkedHashMap();
    }
    Deque<ProductPolicyStepResult> productPolicyStepResults = productPolicyStepResultMapping.get(productCode);
    Set<String> productStepPolicyCodes = Sets.newLinkedHashSet();
    if(!CollectionUtils.isEmpty(productPolicyStepResults)) {
      productStepPolicyCodes = productPolicyStepResults.stream().filter(item -> StringUtils.isNoneBlank(item.getSalePolicyCode())).map(ProductPolicyStepResult::getSalePolicyCode).collect(Collectors.toSet());
    }
    
    // 取得差集，并通过差集从excludedPolicyMapping中提取数据
    Set<String> differences = Sets.difference(matchedPolicyCodes, productStepPolicyCodes);
    Map<String , String> excludedPolicyByProductCodeMapping = Maps.newLinkedHashMap();
    if(CollectionUtils.isEmpty(differences)) {
      return excludedPolicyByProductCodeMapping;
    }
    for (String difference : differences) {
      String errorMsg = this.excludedPolicyMapping.get(difference);
      if(StringUtils.isNotBlank(errorMsg)) {
        excludedPolicyByProductCodeMapping.put(difference, errorMsg);
      }
    }
    return excludedPolicyByProductCodeMapping;
  }
  
  /**
   * 该功能负责在完成计算后（无论计算结果如何），在正式返回给调用者之前；
   * 需要对上下文中的记录信息进行精简。</p>
   * 请注意，该方法应该在整个营销过程完成后，最后再进行调用
   */
  public DefaultPolicyExecuteContext streamlining() {
    /*
     * 主要精简的信息包括：
     * 1、为了不影响缓存中的原始数据，需要对数据进行克隆
     * 2、initSalePolicys中的salePolicy信息（只保留基本信息即可）
     * */
    
    // 1、=======
    DefaultPolicyExecuteContext clonePolicyExecuteContext = ObjectUtils.clone(this);
    
    // 2、=======
    Set<SalePolicyVo> initSalePolicys = clonePolicyExecuteContext.getInitSalePolicys();
    if(!CollectionUtils.isEmpty(initSalePolicys)) {
      for (SalePolicyVo cloneSalePolicys : initSalePolicys) {
        cloneSalePolicys.setCustomerScopeMapping(null);
        cloneSalePolicys.setSalePolicyExecutorInfos(null);
        cloneSalePolicys.setSalePolicyLimitInfos(null);
        cloneSalePolicys.setSalePolicyProductInfos(null);
        cloneSalePolicys.setSalePolicyProductThreshold(null);
      }
    }
    return clonePolicyExecuteContext;
  }
  
  @Override
  public Object clone() throws CloneNotSupportedException {
    // 全深度clone
    byte[] contextBytes = null;
    ByteArrayOutputStream byteStream = new ByteArrayOutputStream();
    try(ObjectOutputStream out = new ObjectOutputStream(byteStream);) {
      out.writeObject(this);
      contextBytes = byteStream.toByteArray();
    } catch(IOException e) {
      throw new CloneNotSupportedException(e.getMessage());
    }
    
    // 生成一个新的对象
    DefaultPolicyExecuteContext clonePolicyExecuteContext = null;
    try(ObjectInputStream in = new ObjectInputStream(new ByteArrayInputStream(contextBytes));) {
      clonePolicyExecuteContext = (DefaultPolicyExecuteContext)in.readObject();
    } catch(IOException | ClassNotFoundException e) {
      throw new CloneNotSupportedException(e.getMessage());
    }
    return clonePolicyExecuteContext;
  }

  @Override
  public BigDecimal findEnjoyedTotalAmount(String salePolicyCode) {
    /*
     * 需要注意，由于一个优惠政策存在多个执行器，所以一个优惠政策可能执行多次
     * */
    BigDecimal enjoyedTotalAmount = BigDecimal.ZERO;
    Iterator<PolicyStepResult> policyStepResultIterator = this.policyStepResults.descendingIterator();
    while(policyStepResultIterator.hasNext()) {
      PolicyStepResult policyStepResult = policyStepResultIterator.next();
      String currentSalePolicyCode = policyStepResult.getSalePolicyCode();
      if(!StringUtils.equals(currentSalePolicyCode, salePolicyCode)) {
        continue;
      }
      
      // 再次注意，是实际优惠了多少钱
      BigDecimal preSubtotal = policyStepResult.getPreSubtotal();
      BigDecimal lastSubtotal = policyStepResult.getLastSubtotal();
      enjoyedTotalAmount = enjoyedTotalAmount.add(preSubtotal.subtract(lastSubtotal));
    }
    return enjoyedTotalAmount;
  }

  @Override
  public Integer findEnjoyedTotalNumber(String salePolicyCode) {
    /*
     * 需要注意，由于一个优惠政策存在多个执行器，所以一个优惠政策可能执行多次
     * */
    Integer enjoyedTotalNumber = 0;
    Iterator<PolicyStepResult> policyStepResultIterator = this.policyStepResults.descendingIterator();
    while(policyStepResultIterator.hasNext()) {
      PolicyStepResult policyStepResult = policyStepResultIterator.next();
      String currentSalePolicyCode = policyStepResult.getSalePolicyCode();
      if(!StringUtils.equals(currentSalePolicyCode, salePolicyCode)) {
        continue;
      }
      
      // 已经优惠了多少数量
      Integer preSurplusTotalNumber = policyStepResult.getPreSurplusTotalNumber();
      preSurplusTotalNumber = preSurplusTotalNumber == null?0:preSurplusTotalNumber;
      Integer lastSurplusTotalNumber = policyStepResult.getLastSurplusTotalNumber();
      lastSurplusTotalNumber = lastSurplusTotalNumber == null?0:lastSurplusTotalNumber;
      enjoyedTotalNumber += preSurplusTotalNumber - lastSurplusTotalNumber;
    }
    return enjoyedTotalNumber;
  }


  @Override
  public BigDecimal findGiftEnjoyedTotalAmount(String salePolicyCode) {
    /*
     * 需要注意，由于一个优惠政策存在多个执行器，所以一个优惠政策可能执行多次
     * */
    BigDecimal enjoyedTotalAmount = BigDecimal.ZERO;
    Iterator<PolicyStepResult> policyStepResultIterator = this.policyStepResults.descendingIterator();
    while(policyStepResultIterator.hasNext()) {
      PolicyStepResult policyStepResult = policyStepResultIterator.next();
      String currentSalePolicyCode = policyStepResult.getSalePolicyCode();
      if(!StringUtils.equals(currentSalePolicyCode, salePolicyCode)) {
        continue;
      }

      // 再次注意，是实际优惠了多少钱
      BigDecimal preEnjoyedTotalAmount = policyStepResult.getPreGiftEnjoyedTotalAmount();
      preEnjoyedTotalAmount = preEnjoyedTotalAmount == null?BigDecimal.ZERO:preEnjoyedTotalAmount;
      BigDecimal lastEnjoyedTotalAmount = policyStepResult.getLastGiftEnjoyedTotalAmount();
      lastEnjoyedTotalAmount = lastEnjoyedTotalAmount == null?BigDecimal.ZERO:lastEnjoyedTotalAmount;
      enjoyedTotalAmount = enjoyedTotalAmount.add(lastEnjoyedTotalAmount.subtract(preEnjoyedTotalAmount));
    }
    return enjoyedTotalAmount;
  }

  @Override
  public Integer findGiftEnjoyedTotalNumber(String salePolicyCode) {
    /*
     * 需要注意，由于一个优惠政策存在多个执行器，所以一个优惠政策可能执行多次
     * */
    Integer enjoyedTotalNumber = 0;
    Iterator<PolicyStepResult> policyStepResultIterator = this.policyStepResults.descendingIterator();
    while(policyStepResultIterator.hasNext()) {
      PolicyStepResult policyStepResult = policyStepResultIterator.next();
      String currentSalePolicyCode = policyStepResult.getSalePolicyCode();
      if(!StringUtils.equals(currentSalePolicyCode, salePolicyCode)) {
        continue;
      }

      // 已经优惠了多少数量
      Integer preEnjoyedTotalNumber = policyStepResult.getPreGiftEnjoyedTotalNumber();
      preEnjoyedTotalNumber = preEnjoyedTotalNumber == null?0:preEnjoyedTotalNumber;
      Integer lastEnjoyedTotalNumber = policyStepResult.getLastGiftEnjoyedTotalNumber();
      lastEnjoyedTotalNumber = lastEnjoyedTotalNumber == null?0:lastEnjoyedTotalNumber;
      enjoyedTotalNumber += lastEnjoyedTotalNumber - preEnjoyedTotalNumber;
    }
    return enjoyedTotalNumber;
  }
}
