package com.biz.crm.dms.business.policy.local.controller;

import java.util.Collection;
import java.util.List;

import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.ApplicationContext;
import org.springframework.util.CollectionUtils;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.biz.crm.business.common.sdk.model.Result;
import com.biz.crm.dms.business.policy.sdk.characteristic.CharacteristicStrategy;
import com.biz.crm.dms.business.policy.sdk.enums.ProductSelectionMethod;
import com.biz.crm.dms.business.policy.sdk.register.SalePolicyTempleteRegister;
import com.biz.crm.dms.business.policy.sdk.strategy.SalePolicyCustomerScopeStrategy;
import com.biz.crm.dms.business.policy.sdk.strategy.SalePolicyExecuteStrategy;
import com.biz.crm.dms.business.policy.sdk.strategy.SalePolicyLimitStrategy;
import com.biz.crm.dms.business.policy.sdk.vo.AbstractCharacteristicInfo;
import com.biz.crm.dms.business.policy.sdk.vo.AbstractSalePolicyCustomerScopeInfo;
import com.biz.crm.dms.business.policy.sdk.vo.AbstractSalePolicyExecutorInfo;
import com.biz.crm.dms.business.policy.sdk.vo.AbstractSalePolicyLimitInfo;

import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;

/**
 * 该路径下的http restful地址用来查询当前系统已经支持的优惠政策模板注册器管理
 * @author yinwenjie
 */
@Api(tags = "该路径下的http restful地址用来查询当前系统已经支持的优惠政策模板注册器管理")
@RestController
@RequestMapping("/v1/salepolicies/templeteregisters")
public class SalePolicyTempleteRegisterController {
  /**
   * 日志
   */
  private static final Logger LOGGER = LoggerFactory.getLogger(SalePolicyTempleteRegisterController.class);
  /**
   * 现在系统中支持的所有优惠政策注册信息
   */
  @Autowired(required = false)
  private List<SalePolicyTempleteRegister> salePolicyTempleteRegisters;
  @Autowired(required = false)
  private ApplicationContext applicationContext;
  
  /**
   * 查询目前系统中所有已注册的优惠政策模板信息
   * */
  @ApiOperation(value = "查询目前系统中所有已注册的优惠政策模板信息")
  @GetMapping(value="/findAll")
  public Result<?> findAll() {
    try {
      JSONArray resutls = new JSONArray();
      for (SalePolicyTempleteRegister salePolicyTempleteRegister : salePolicyTempleteRegisters) {
        String type = salePolicyTempleteRegister.getType();
        String typeDesc = salePolicyTempleteRegister.getTypeDesc();
        Boolean supportProduct = salePolicyTempleteRegister.supportProduct();
        
        JSONObject item = new JSONObject();
        item.put("type", type);
        item.put("typeDesc", typeDesc);
        item.put("supportProduct", supportProduct);
        // 寻找已经注册支持的“商品选择方式”信息：productSelectionMethods
        JSONArray productSelectionMethodJson = this.findProductSelectionMethods(salePolicyTempleteRegister);
        item.put("productSelectionMethods", productSelectionMethodJson);
        // 寻找已注册支持的“优惠政策特性执行策略”信息：characteristicStraties 可能为null
        JSONArray characteristicStratiesJson = this.findCharacteristicStraties(salePolicyTempleteRegister);
        item.put("characteristicStraties", characteristicStratiesJson);
        // 寻找已注册支持的“客户范围选择方式”信息，customerScopeStraties
        JSONArray customerScopeStratiesJson = this.findCustomerScopeStraties(salePolicyTempleteRegister);
        item.put("customerScopeStraties", customerScopeStratiesJson);
        // 寻找已注册支持的“政策限量信息”信息，limitStraties;
        JSONArray limitStratiesJson = this.findLimitStraties(salePolicyTempleteRegister);
        item.put("limitStraties", limitStratiesJson);
        // 寻找已注册支持的“优惠政策执行方式”信息，executeStraties;
        JSONArray executeStratiesJson = this.findExecuteStraties(salePolicyTempleteRegister);
        item.put("executeStraties", executeStratiesJson);
        resutls.add(item);
      }
      return Result.ok(resutls);
    } catch(RuntimeException e) {
      LOGGER.error(e.getMessage() , e);
      return Result.error(e.getMessage());
    }
  }
  
  /**
   * 查询目前系统中某个指定的优惠政策模板信息
   * */
  @ApiOperation(value = "查询目前系统中某个指定的优惠政策模板信息")
  @GetMapping(value="/findByType")
  public Result<?> findByType(@RequestParam("type") @ApiParam(name = "type" , required = true , value = "优惠政策类型") String type) {
    try {
      SalePolicyTempleteRegister matchedSalePolicyTempleteRegister = null;
      for (SalePolicyTempleteRegister salePolicyTempleteRegister : salePolicyTempleteRegisters) {
        if(StringUtils.equals(salePolicyTempleteRegister.getType(), type)) {
          matchedSalePolicyTempleteRegister = salePolicyTempleteRegister;
          break;
        }
      }
      Validate.notNull(matchedSalePolicyTempleteRegister , "注册信息存在设定问题(%s)，请联系技术人员。" , type);
      // 开始组装返回信息
      String typeDesc = matchedSalePolicyTempleteRegister.getTypeDesc();
      Boolean supportProduct = matchedSalePolicyTempleteRegister.supportProduct();
      JSONObject item = new JSONObject();
      item.put("type", type);
      item.put("typeDesc", typeDesc);
      item.put("supportProduct", supportProduct);
      // 寻找已经注册支持的“商品选择方式”信息：productSelectionMethods
      JSONArray productSelectionMethodJson = this.findProductSelectionMethods(matchedSalePolicyTempleteRegister);
      item.put("productSelectionMethods", productSelectionMethodJson);
      // 寻找已注册支持的“优惠政策特性执行策略”信息：characteristicStraties 可能为null
      JSONArray characteristicStratiesJson = this.findCharacteristicStraties(matchedSalePolicyTempleteRegister);
      item.put("characteristicStraties", characteristicStratiesJson);
      // 寻找已注册支持的“客户范围选择方式”信息，customerScopeStraties
      JSONArray customerScopeStratiesJson = this.findCustomerScopeStraties(matchedSalePolicyTempleteRegister);
      item.put("customerScopeStraties", customerScopeStratiesJson);
      // 寻找已注册支持的“政策限量信息”信息，limitStraties;
      JSONArray limitStratiesJson = this.findLimitStraties(matchedSalePolicyTempleteRegister);
      item.put("limitStraties", limitStratiesJson);
      // 寻找已注册支持的“优惠政策执行方式”信息，executeStraties;
      JSONArray executeStratiesJson = this.findExecuteStraties(matchedSalePolicyTempleteRegister);
      item.put("executeStraties", executeStratiesJson);
      return Result.ok(item);
    } catch(RuntimeException e) {
      LOGGER.error(e.getMessage() , e);
      return Result.error(e.getMessage());
    }
  }
  
  private JSONArray findCharacteristicStraties(SalePolicyTempleteRegister salePolicyTempleteRegister) {
    // 开始查询这个优惠政策类型可以使用的限量控制信息
    Collection<Class<? extends CharacteristicStrategy<? extends AbstractCharacteristicInfo>>> characteristicStrategyClasses = salePolicyTempleteRegister.getCharacteristicStrategyClasses();
    if(CollectionUtils.isEmpty(characteristicStrategyClasses)) {
      return new JSONArray();
    }
    JSONArray resutls = new JSONArray();
    for (Class<? extends CharacteristicStrategy<? extends AbstractCharacteristicInfo>> characteristicStrategyClass : characteristicStrategyClasses) {
      CharacteristicStrategy<? extends AbstractCharacteristicInfo> characteristicStrategy = applicationContext.getBean(characteristicStrategyClass);
      JSONObject item = new JSONObject();
      String characteristicCode = characteristicStrategy.characteristicCode();
      String characteristicName = characteristicStrategy.characteristicName();
      String expression = characteristicStrategy.getExpression();
      
      item.put("characteristicCode", characteristicCode);
      item.put("characteristicName", characteristicName);
      item.put("expression", expression);
      resutls.add(item);
    }
    return resutls;
  }
  
  /**
   * 该私有方法用于查询特定的优惠政策模板注册器已注册支持的“商品选择方式”信息
   * @param salePolicyTempleteRegister
   * @return
   */
  private JSONArray findProductSelectionMethods(SalePolicyTempleteRegister salePolicyTempleteRegister) {
    ProductSelectionMethod[] productSelectionMethods = salePolicyTempleteRegister.getProductSelectionMethods();
    if(productSelectionMethods == null || productSelectionMethods.length == 0) {
      return new JSONArray();
    }
    
    JSONArray resutls = new JSONArray();
    for (ProductSelectionMethod productSelectionMethod : productSelectionMethods) {
      JSONObject itemObject = new JSONObject();
      String code = productSelectionMethod.getCode();
      String desc = productSelectionMethod.getDesc();
      itemObject.put("code", code);
      itemObject.put("desc", desc);
      resutls.add(itemObject);
    }
    return resutls;
  }
  
  /**
   * 该私有方法用于查询特定的优惠政策模板注册器已注册支持的“客户范围选择方式”信息
   * @param salePolicyTempleteRegister
   * @return
   */
  private JSONArray findCustomerScopeStraties(SalePolicyTempleteRegister salePolicyTempleteRegister) {
    // 开始查询这个优惠政策类型可以使用的限量控制信息
    Collection<Class<? extends SalePolicyCustomerScopeStrategy<? extends AbstractSalePolicyCustomerScopeInfo>>> customerScopeStrategyClasses = salePolicyTempleteRegister.getCustomerScopeStrategyClasses();
    if(CollectionUtils.isEmpty(customerScopeStrategyClasses)) {
      return new JSONArray();
    }
    JSONArray resutls = new JSONArray();
    for (Class<? extends SalePolicyCustomerScopeStrategy<? extends AbstractSalePolicyCustomerScopeInfo>> customerScopeStrategyClass : customerScopeStrategyClasses) {
      SalePolicyCustomerScopeStrategy<? extends AbstractSalePolicyCustomerScopeInfo> customerScopeStrategy = applicationContext.getBean(customerScopeStrategyClass);
      JSONObject item = new JSONObject();
      String scopeType = customerScopeStrategy.getScopeType();
      String scopeTypeDesc = customerScopeStrategy.getScopeTypeDesc();
      item.put("scopeType", scopeType);
      item.put("scopeTypeDesc", scopeTypeDesc);
      resutls.add(item);
    }
    return resutls;
  }
  
  /**
   * 该私有方法用于查询特定的优惠政策模板注册器已注册支持的“政策限量”信息
   * @param salePolicyTempleteRegister
   * @return
   */
  private JSONArray findLimitStraties(SalePolicyTempleteRegister salePolicyTempleteRegister) {
    // 开始查询这个优惠政策类型可以使用的限量控制信息
    Collection<Class<? extends SalePolicyLimitStrategy<? extends AbstractSalePolicyLimitInfo>>> bindableLimitStrategyClasses =  salePolicyTempleteRegister.getBindableLimitStrategyClasses();
    if(CollectionUtils.isEmpty(bindableLimitStrategyClasses)) {
      return new JSONArray();
    }
    JSONArray resutls = new JSONArray();
    for (Class<? extends SalePolicyLimitStrategy<? extends AbstractSalePolicyLimitInfo>> bindableLimitStrategyClass : bindableLimitStrategyClasses) {
      SalePolicyLimitStrategy<? extends AbstractSalePolicyLimitInfo> salePolicyLimitStrategy = applicationContext.getBean(bindableLimitStrategyClass);
      String limitStrategyCode = salePolicyLimitStrategy.getLimitStrategyCode();
      String limitStrategyDesc = salePolicyLimitStrategy.getLimitStrategyDesc();
      String expression = salePolicyLimitStrategy.getExpression();
      JSONObject item = new JSONObject();
      item.put("limitStrategyCode", limitStrategyCode);
      item.put("limitStrategyDesc", limitStrategyDesc);
      item.put("expression", expression);
      resutls.add(item);
    }
    return resutls;
  }
  
  /**
   * 该私有方法用于查询特定的优惠政策模板注册器已注册支持的“优惠政策执行方式”信息
   * @param salePolicyTempleteRegister
   * @return
   */
  private JSONArray findExecuteStraties(SalePolicyTempleteRegister salePolicyTempleteRegister) {
    // 开始查询这里匹配了type的优惠政策执行阶梯信息
    Collection<Class<? extends SalePolicyExecuteStrategy<? extends AbstractSalePolicyExecutorInfo>>> salePolicyExecuteStrategyClasses =  salePolicyTempleteRegister.getExecuteStrategyClasses();
    Validate.isTrue(!CollectionUtils.isEmpty(salePolicyExecuteStrategyClasses) , "注册信息存在设定问题(%s)，请联系技术人员。" , salePolicyTempleteRegister.getType());
    
    JSONArray resutls = new JSONArray();
    for (Class<? extends SalePolicyExecuteStrategy<? extends AbstractSalePolicyExecutorInfo>> salePolicyExecuteStrategyClass : salePolicyExecuteStrategyClasses) {
      SalePolicyExecuteStrategy<? extends AbstractSalePolicyExecutorInfo> salePolicyExecuteStrategy = applicationContext.getBean(salePolicyExecuteStrategyClass);
      // 组装那些满足要求的优惠政策执行策略
      String executeStrategyCode = salePolicyExecuteStrategy.getExecuteStrategyCode();
      String executeStrategyDesc = salePolicyExecuteStrategy.getExecuteStrategyDesc();
      String expression = salePolicyExecuteStrategy.getExpression();
      JSONObject item = new JSONObject();
      item.put("executeStrategyCode", executeStrategyCode);
      item.put("executeStrategyDesc", executeStrategyDesc);
      item.put("expression", expression);
      resutls.add(item);
    }
    return resutls;
  }
}
