package com.biz.crm.dms.business.policy.local.controller;

import java.util.Set;

import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Pageable;
import org.springframework.data.web.PageableDefault;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PatchMapping;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.business.common.sdk.model.Result;
import com.biz.crm.dms.business.policy.local.service.SalePolicyTempleteVoService;
import com.biz.crm.dms.business.policy.sdk.dto.SalePolicyTempletePageDto;
import com.biz.crm.dms.business.policy.sdk.vo.SalePolicyTempleteVo;
import com.bizunited.nebula.common.util.tenant.TenantUtils;

import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;

/**
 * 该路径下的http restful地址用来查询当前系统已经支持的优惠政策模板管理
 * @author yinwenjie
 */
@Api(tags = "该路径下的http restful地址用来查询当前系统已经支持的优惠政策模板管理")
@RestController
@RequestMapping("/v1/salepolicies/templetes")
public class SalePolicyTempleteVoController {
  /**
   * 日志
   */
  private static final Logger LOGGER = LoggerFactory.getLogger(SalePolicyTempleteVoController.class);
  @Autowired(required = false)
  private SalePolicyTempleteVoService salePolicyTempleteVoService;
  
  @ApiOperation(value = "创建一个新的优惠政策模板")
  @PostMapping(value="")
  public Result<SalePolicyTempleteVo> create(@RequestBody @ApiParam(name="salePolicyTemplete" , value="创建一个新的优惠政策模板") SalePolicyTempleteVo salePolicyTemplete) {
    try {
      SalePolicyTempleteVo current = this.salePolicyTempleteVoService.create(salePolicyTemplete);
      return Result.ok(current);
    } catch(RuntimeException e) {
      LOGGER.error(e.getMessage() , e);
      return Result.error(e.getMessage());
    }
  }
  
  @ApiOperation(value = "对特定的优惠政策模板进行修改" , notes = "注意，就该操作并不会影响根据这个优惠政策模板已经创建的优惠政策信息")
  @PatchMapping(value="")
  public Result<SalePolicyTempleteVo> update(@RequestBody @ApiParam(name="salePolicyTemplete" , value="对特定的优惠政策模板进行修改（注意，就该操作并不会影响根据这个优惠政策模板已经创建的优惠政策信息）") SalePolicyTempleteVo salePolicyTemplete) {
    try {
      SalePolicyTempleteVo current = this.salePolicyTempleteVoService.update(salePolicyTemplete);
      return Result.ok(current);
    } catch(RuntimeException e) {
      LOGGER.error(e.getMessage() , e);
      return Result.error(e.getMessage());
    }
  }
  
  /**
   * 将指定的营销活动下架（也就是将enableStatus标记设置为EnableStatusEnum.DISABLE）
   * */
  @ApiOperation(value = "将指定的营销活动下架（也就是将enableStatus标记设置为EnableStatusEnum.DISABLE）")
  @PostMapping(value="invalid")
  public Result<?> invalid(@RequestParam("salePolicyTempleteCode") @ApiParam(name = "salePolicyTempleteCode", value = "促销政策模板指定的业务编号")  String salePolicyTempleteCode) {
    try {
      this.salePolicyTempleteVoService.invalid(salePolicyTempleteCode);
      return Result.ok();
    } catch(RuntimeException e) {
      LOGGER.error(e.getMessage() , e);
      return Result.error(e.getMessage());
    }
  }
  
  /**
   * 将指定的营销活动启用（也就是将enableStatus标记设置为EnableStatusEnum.ENABLE）
   * */
  @ApiOperation(value = "将指定的营销活动启用（也就是将enableStatus标记设置为EnableStatusEnum.ENABLE）")
  @PostMapping(value="effective")
  public Result<?> effective(@RequestParam("salePolicyTempleteCode") @ApiParam(name = "salePolicyTempleteCode", value = "促销政策模板指定的业务编号")  String salePolicyTempleteCode) {
    try {
      this.salePolicyTempleteVoService.effective(salePolicyTempleteCode);
      return Result.ok();
    } catch(RuntimeException e) {
      LOGGER.error(e.getMessage() , e);
      return Result.error(e.getMessage());
    }
  }
  
  /**
   * 该方法可用于物理删除指定的优惠政策模板，但是只有那些没有关联任何优惠政策的模板，才能被删除
   * */
  @ApiOperation(value = "该方法可用于物理删除指定的优惠政策模板，但是只有那些没有关联任何优惠政策的模板，才能被删除")
  @PostMapping(value="deleteByTempleteCode")
  public Result<?> deleteByTempleteCode(@RequestParam("templeteCode") @ApiParam(name = "templeteCode", value = "促销政策模板指定的业务编号")  String templeteCode) {
    try {
      this.salePolicyTempleteVoService.deleteByTempleteCode(templeteCode);
      return Result.ok();
    } catch(RuntimeException e) {
      LOGGER.error(e.getMessage() , e);
      return Result.error(e.getMessage());
    }
  }
  
  @ApiOperation(value = "按照当前二级租户查询已经常见的所有优惠政策模板（按照创建时间排序）")
  @GetMapping(value="/findByCurrentTenantCode")
  public Result<Set<SalePolicyTempleteVo>> findByCurrentTenantCode(@RequestParam(required = false , value = "enableStatus") @ApiParam(name = "enableStatus", value = "启用或者禁用标记(非必要参数)，并参见EnableStatusEnum枚举信息：003：禁用；009：启用")  String enableStatus) {
    try {
      String tenantCode = TenantUtils.getTenantCode();
      EnableStatusEnum enableStatusEnum = null;
      if(StringUtils.isNotBlank(enableStatus)) {
        enableStatusEnum = EnableStatusEnum.codeToEnum(enableStatus);
      }
      Set<SalePolicyTempleteVo> results = this.salePolicyTempleteVoService.findByTenantCode(tenantCode , enableStatusEnum);
      return Result.ok(results);
    } catch(RuntimeException e) {
      LOGGER.error(e.getMessage() , e);
      return Result.error(e.getMessage());
    }
  }
  
  @ApiOperation(value = "按照当前二级租户查询已经常见的所有优惠政策模板（按照创建时间排序）")
  @GetMapping(value="/findByTempleteCode")
  public Result<SalePolicyTempleteVo> findByTempleteCode(@RequestParam("templeteCode") @ApiParam(name="tempeleteCode" , value="指定的优惠政策模板编号") String tempeleteCode) {
    try {
      String tenantCode = TenantUtils.getTenantCode();
      SalePolicyTempleteVo result = this.salePolicyTempleteVoService.findByTempleteCode(tempeleteCode, tenantCode);
      return Result.ok(result);
    } catch(RuntimeException e) {
      LOGGER.error(e.getMessage() , e);
      return Result.error(e.getMessage());
    }
  }
  
  @ApiOperation(value = "针对特定二级租户的已存在的优惠政策模板分页查询信息")
  @GetMapping(value="/findByConditions")
  public Result<Page<SalePolicyTempleteVo>> findByConditions(@PageableDefault(50) Pageable pageable, @ApiParam(name = "dto", value = "分页Dto") SalePolicyTempletePageDto dto) {
    try { 
      Page<SalePolicyTempleteVo> results = this.salePolicyTempleteVoService.findByConditions(pageable, dto);
      return Result.ok(results);
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return Result.error(e.getMessage());
    }
  }
}
