package com.biz.crm.dms.business.policy.local.controller;

import com.alibaba.fastjson.JSONObject;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.sdk.model.Result;
import com.biz.crm.dms.business.policy.local.service.task.SalePolicyCacheLoadingService;
import com.biz.crm.dms.business.policy.sdk.dto.SalePolicyPageDto;
import com.biz.crm.dms.business.policy.sdk.service.SalePolicyVoService;
import com.biz.crm.dms.business.policy.sdk.vo.SalePolicyVo;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.google.common.collect.Lists;

import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;

import java.lang.String;
import java.util.List;
import java.util.Set;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Pageable;
import org.springframework.data.web.PageableDefault;
import org.springframework.util.CollectionUtils;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PatchMapping;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

/**
 * SalePolicyVo（优惠政策）业务模型的MVC Controller层实现，基于HTTP Restful风格
 * @author yinwenjie
 */
@Api(tags = " SalePolicyVo（优惠政策）业务模型的MVC Controller层实现，基于HTTP Restful风格")
@RestController
@RequestMapping("/v1/salepolicies/policies")
public class SalePolicyVoController { 
  /**
   * 日志
   */
  private static final Logger LOGGER = LoggerFactory.getLogger(SalePolicyVoController.class);
  
  @Autowired(required = false)
  private SalePolicyVoService salePolicyVoService;
  @Autowired(required = false)
  private SalePolicyCacheLoadingService salePolicyCacheLoadingService;

  /**
   * 由于营销活动使用创建授权标签来避免重复提交的问题。所以在创建营销活动前，需要使用该方法获得预授权</br>
   * 预授权成功后，才能通过预授权信息进行添加，
   * @return 
   */
  @ApiOperation(value = "由于促销政策单使用创建授权标签来避免重复提交的问题。所以在创建/修改营销政策单前，需要使用该方法获得预授权。预授权成功后，才能通过预授权信息进行添加，")
  @PostMapping(value="/preSave")
  public Result<?> preSave() {
    try {
      String prefix = this.salePolicyVoService.preSave();
      return Result.ok(prefix);
    } catch(RuntimeException e) {
      LOGGER.error(e.getMessage() , e);
      return Result.error(e.getMessage());
    }
  }
  
  @ApiOperation(value = "创建一个新的SalePolicy模型对象（包括了可能的第三方系统调用、复杂逻辑处理等）")
  @PostMapping(value="")
  public Result<?> create(@RequestBody @ApiParam(name="salePolicy" , value="创建一个新的SalePolicy模型对象（JSON结构是动态的包括了可能的第三方系统调用、复杂逻辑处理等）") JSONObject salePolicyJson) {
    try {
      SalePolicyVo current = this.salePolicyVoService.create(salePolicyJson);
      return Result.ok(current);
    } catch(RuntimeException e) {
      LOGGER.error(e.getMessage() , e);
      return Result.error(e.getMessage());
    }
  }

  @ApiOperation(value = "更新一个已有的SalePolicy模型对象，其主键属性必须有值。")
  @PatchMapping(value="")
  public Result<?> update(@RequestBody @ApiParam(name="salePolicyJson" , value="更新一个新的SalePolicy模型对象（JSON结构是动态的包括了可能的第三方系统调用、复杂逻辑处理等）") JSONObject salePolicyJson) {
    try {
      SalePolicyVo current = this.salePolicyVoService.update(salePolicyJson);
      return Result.ok(current);
    } catch(RuntimeException e) {
      LOGGER.error(e.getMessage() , e);
      return Result.error(e.getMessage());
    }
  }
  
  /**
   * 将指定的营销活动下架（也就是将effective标记设置为false）
   * */
  @ApiOperation(value = "将指定的营销活动下架（也就是将effective标记设置为false）")
  @PostMapping(value="invalid")
  public Result<?> invalid(@RequestParam("salePolicyCodes") @ApiParam(name = "salePolicyCodes", value = "营销活动的业务编号")  String[] salePolicyCodes) {
    try {
      this.salePolicyVoService.invalid(salePolicyCodes);
      return Result.ok();
    } catch(RuntimeException e) {
      LOGGER.error(e.getMessage() , e);
      return Result.error(e.getMessage());
    }
  }
  
  /**
   * 将指定的营销活动下架（也就是将effective标记设置为true）
   * */
  @ApiOperation(value = "将指定的营销活动下架（也就是将effective标记设置为true）")
  @PostMapping(value="effective")
  public Result<?> effective(@RequestParam("salePolicyCodes") @ApiParam(name = "salePolicyCodes", value = "营销活动的业务编号")  String[] salePolicyCodes) {
    try {
      this.salePolicyVoService.effective(salePolicyCodes);
      return Result.ok();
    } catch(RuntimeException e) {
      LOGGER.error(e.getMessage() , e);
      return Result.error(e.getMessage());
    }
  }
  
  /**
   * 该方法可用于物理删除指定的优惠政策，但是只有那些没有开始执行、没有关联优惠政策执行流水的优惠政策，可以被删除
   */
  @ApiOperation(value = "该方法可用于物理删除指定的优惠政策，但是只有那些没有开始执行、没有关联优惠政策执行流水的优惠政策，可以被删除")
  @PostMapping(value="deleteBySalePolicyCode")
  public Result<?> deleteBySalePolicyCode(@RequestParam("salePolicyCode") String salePolicyCode) {
    try {
      this.salePolicyVoService.deleteBySalePolicyCode(salePolicyCode);
      return Result.ok();
    } catch(RuntimeException e) {
      LOGGER.error(e.getMessage() , e);
      return Result.error(e.getMessage());
    }
  }
  
  @ApiOperation(value = "进行当前租户所有优惠政策信息缓存的刷新，以保证数据库中最新的优惠政策数据进行加载同步")
  @PostMapping(value="refresh")
  public Result<?> refresh() {
    try {
      String tenantCode = TenantUtils.getTenantCode();
      this.salePolicyCacheLoadingService.notifyCacheRefresh(tenantCode);
      return Result.ok();
    } catch(RuntimeException e) {
      LOGGER.error(e.getMessage() , e);
      return Result.error(e.getMessage());
    }
  } 
  
  /**
   * 查询当前系统已经支持的在优惠政策管理中，确定客户范围的策略信息
   * */
  @ApiOperation(value = "查询当前系统已经支持的在优惠政策管理中，确定客户范围的策略信息")
  @GetMapping(value="findDetailsByCode")
  public Result<SalePolicyVo> findDetailsByCode(@RequestParam("salePolicyCode") String salePolicyCode) {
    try {
      SalePolicyVo salePolicyVo = this.salePolicyVoService.findDetailsByCode(salePolicyCode);
      return Result.ok(salePolicyVo);
    } catch(RuntimeException e) {
      LOGGER.error(e.getMessage() , e);
      return Result.error(e.getMessage());
    }
  }
  
  /**
   * 查询当前系统已经支持的在优惠政策管理中，确定客户范围的策略信息(多个)
   * */
  @ApiOperation(value = "查询当前系统已经支持的在优惠政策管理中，确定客户范围的策略信息（多个）")
  @GetMapping(value="findDetailsByCodes")
  public Result<List<SalePolicyVo>> findDetailsByCodes(@RequestParam("salePolicyCodes") List<String> salePolicyCodes) {
    try {
      List<SalePolicyVo> results = Lists.newArrayList();
      for (String salePolicyCode : salePolicyCodes) {
        SalePolicyVo salePolicyVo = this.salePolicyVoService.findDetailsByCode(salePolicyCode);
        if(salePolicyVo != null) {
          results.add(salePolicyVo);
        }
      }
      return Result.ok(results);
    } catch(RuntimeException e) {
      LOGGER.error(e.getMessage() , e);
      return Result.error(e.getMessage());
    }
  }
  
  /**
   * 查询和指定商品匹配的正在进行的营销活动信息（多个）。 
   * */
  @ApiOperation(value = "查询和指定商品匹配的正在进行的营销活动信息（多个）。 ")
  @GetMapping(value="findDetailsByProcessingAndCustomerCodesAndProductCode")
  public Result<List<SalePolicyVo>> findDetailsByProcessingAndCustomerCodesAndProductCode(@RequestParam("productCode") String productCode ,@RequestParam("customerCode") String customerCode) {
    try {
      List<SalePolicyVo> results = Lists.newArrayList();
      String tenantCode = TenantUtils.getTenantCode();
      Set<SalePolicyVo> salePolicySet = this.salePolicyVoService.findDetailsByProcessingAndCustomerCodesAndProductCodes(tenantCode, true , customerCode , null , productCode);
      if(!CollectionUtils.isEmpty(salePolicySet)) {
        results.addAll(salePolicySet);
      }
      return Result.ok(results);
    } catch(RuntimeException e) {
      LOGGER.error(e.getMessage() , e);
      return Result.error(e.getMessage());
    }
  }
  
  // ===== 以下查询直接来源于数据层
  
  @ApiOperation(value = "针对特定二级租户的已存在的优惠政策分页查询信息。")
  @GetMapping(value="/findByConditions")
  public Result<Page<SalePolicyVo>> findByConditions(@PageableDefault(50) Pageable pageable, @ApiParam(name = "dto", value = "分页Dto") SalePolicyPageDto dto) {
    try { 
      Page<SalePolicyVo> results = this.salePolicyVoService.findByConditions(pageable, dto);
      return Result.ok(results);
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return Result.error(e.getMessage());
    }
  }
} 
