package com.biz.crm.dms.business.policy.local.executestrategy;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.util.CollectionUtils;

import com.biz.crm.dms.business.policy.local.entity.SalePolicyExecutor;
import com.biz.crm.dms.business.policy.local.entity.SalePolicyExecutorLadder;
import com.biz.crm.dms.business.policy.local.repository.SalePolicyExecutorLadderGiftRepository;
import com.biz.crm.dms.business.policy.local.repository.SalePolicyExecutorLadderRepository;
import com.biz.crm.dms.business.policy.local.repository.SalePolicyExecutorLadderVarRepository;
import com.biz.crm.dms.business.policy.local.repository.SalePolicyExecutorRepository;
import com.biz.crm.dms.business.policy.local.service.SalePolicyExecutorService;
import com.biz.crm.dms.business.policy.local.utils.AnalyzeExpressionUtils;

import com.biz.crm.dms.business.policy.local.vo.SalePolicyExecutorLadderVarVo;
import com.biz.crm.dms.business.policy.local.vo.SalePolicyExecutorLadderVo;
import com.biz.crm.dms.business.policy.local.vo.SalePolicyExecutorVo;
import com.biz.crm.dms.business.policy.local.context.DefaultCycleExecuteContext;
import com.biz.crm.dms.business.policy.local.context.DefaultPolicyExecuteContext;
import com.biz.crm.dms.business.policy.sdk.context.AbstractCycleExecuteContext;
import com.biz.crm.dms.business.policy.sdk.context.AbstractPolicyExecuteContext;
import com.biz.crm.dms.business.policy.sdk.context.ProductPolicyStepResult;
import com.biz.crm.dms.business.policy.sdk.context.SalePolicyConProduct;
import com.biz.crm.dms.business.policy.sdk.vo.AbstractSalePolicyExecutorInfo;
import com.biz.crm.dms.business.policy.sdk.vo.SalePolicyVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;

/**
 * 这个抽象父类，帮助DMS标品中的“折扣”优惠政策管理冗余代码，并没有任何业务意义
 * @author yinwenjie
 */
public abstract class AbstractStandardExecuteStrategy extends AbstractExecuteStrategy {
  @Autowired(required = false)
  private SalePolicyExecutorRepository salePolicyExecutorRepository;
  @Autowired(required = false)
  private SalePolicyExecutorLadderRepository salePolicyExecutorLadderRepository;
  @Autowired(required = false)
  private SalePolicyExecutorLadderGiftRepository salePolicyExecutorLadderGiftRepository;
  @Autowired(required = false)
  private SalePolicyExecutorLadderVarRepository salePolicyExecutorLadderVarRepository; 
  @Autowired(required = false)
  protected NebulaToolkitService nebulaToolkitService;
  @Autowired(required = false)
  protected SalePolicyExecutorService salePolicyExecutorService;
  
  /**
   * 这个方法将返回这个具体执行策略的全系统唯一业务编号
   * @return
   */
  protected abstract String getExecuteStrategyCode();
  
  /**
   * 该方法将返回这个具体执行策略的页面显示内容表达式
   * @return
   */
  protected abstract String getExpression();
  
  /**
   * 该方法创建一个默认的阶梯循环上下文，这也是标品中提供的多种优惠政策在阶梯计算时所使用的上下文
   * @return
   */
  protected DefaultCycleExecuteContext buildDefaultCycleExecuteContext(String executorCode , AbstractPolicyExecuteContext abstractExecuteContext , Set<SalePolicyConProduct> initPolicyProducts , String customerCode) {
    DefaultPolicyExecuteContext executeContext = (DefaultPolicyExecuteContext)abstractExecuteContext;
    /*
     * 初始化优惠政策阶梯循环执行上下文，并开始执行。注意：
     * a、在每一次阶梯循环控制器正式返回执行结果后，都代表整个门槛中的商品完成了优惠执行，接着就要进行分摊
     * b、另外需要注意每次实例化DefaultCycleExecuteContext对象时，initPolicyProducts、initSubtotalAmount、initSubtotalNumber，三个参数的数据来源
     * */
    BigDecimal initSubtotalAmountForCycleExecutes = BigDecimal.ZERO;
    BigDecimal initSubtotalNumberForCycleExecutes = BigDecimal.ZERO;
    BigDecimal preSubtotalAmountForCycleExecutes = BigDecimal.ZERO;
    BigDecimal preSurplusSubtotalAmountForCycleExecutes = BigDecimal.ZERO;
    BigDecimal preSurplusSubtotalNumberForCycleExecutes = BigDecimal.ZERO;
    Set<SalePolicyConProduct> policyProductsForCycleExecutes = Sets.newLinkedHashSet();
    for (SalePolicyConProduct abstractPolicyConProduct : initPolicyProducts) {
      SalePolicyConProduct policyProductForCycleExecutes = (SalePolicyConProduct)abstractPolicyConProduct;
      policyProductsForCycleExecutes.add(policyProductForCycleExecutes);
      ProductPolicyStepResult productPolicyStepResult = executeContext.findLastProductStepResultByProductCode(policyProductForCycleExecutes.getProductCode());
      initSubtotalAmountForCycleExecutes = initSubtotalAmountForCycleExecutes.add(productPolicyStepResult.getInitSubtotal());
      initSubtotalNumberForCycleExecutes = initSubtotalNumberForCycleExecutes.add(new BigDecimal(productPolicyStepResult.getInitNumbers()));
      preSubtotalAmountForCycleExecutes = preSubtotalAmountForCycleExecutes.add(productPolicyStepResult.getLastSubtotal());
      preSurplusSubtotalAmountForCycleExecutes = preSurplusSubtotalAmountForCycleExecutes.add(productPolicyStepResult.getLastSurplusTotalAmount());
      preSurplusSubtotalNumberForCycleExecutes = preSurplusSubtotalNumberForCycleExecutes.add(new BigDecimal(productPolicyStepResult.getLastSurplusTotalNumber()));
    }
    DefaultCycleExecuteContext cycleExecuteContext = new DefaultCycleExecuteContext(executorCode , customerCode , policyProductsForCycleExecutes, initSubtotalAmountForCycleExecutes, initSubtotalNumberForCycleExecutes.intValue(),
      preSubtotalAmountForCycleExecutes , preSurplusSubtotalAmountForCycleExecutes , preSurplusSubtotalNumberForCycleExecutes.intValue());
    return cycleExecuteContext;
  }
  
  /**
   * 为了在标品默认实现中，达到降低冗余代码、降低逻辑理解难度的目的，标品中对于执行策略的执行过程，一般都实现该execute(...)方法。</br>
   * 这个方法相比SDK层面（SalePolicyExecuteStrategy）要求实现的execute(...)方法来说更具有便宜性
   * @param cycleExecuteContext 本次阶梯循环执行的上下文
   * @param index 阶梯索引，从0开始
   * @param times 第几次进行阶梯执行，从1开始
   * @param varParams 从调用者处获得的阶梯变量传值信息
   * @param salePolicyExecutorLadder 当前的阶梯信息
   * @return 
   */
  protected abstract boolean execute(AbstractCycleExecuteContext abstractCycleExecuteContext , int index, int times, Map<String, Object> varParams , SalePolicyExecutorLadderVo salePolicyExecutorLadder);
  
  /**
   * 对SDK层面提供的SalePolicyExecuteStrategy接口方法进行部分实现，以便降低标品实现中的代码冗余度
   */
  public boolean execute(AbstractCycleExecuteContext abstractCycleExecuteContext, int index, int times, AbstractSalePolicyExecutorInfo correctAbstractExecutorInfo) {
    SalePolicyExecutorVo correctExecutorInfo = (SalePolicyExecutorVo) correctAbstractExecutorInfo;
    Set<SalePolicyExecutorLadderVo> salePolicyExecutorLadders = correctExecutorInfo.getSalePolicyExecutorLadders();
    
    // 找到目前正确的阶梯
    SalePolicyExecutorLadderVo salePolicyExecutorLadder = Lists.newArrayList(salePolicyExecutorLadders).get(index);
    // 获得当前阶梯的设定值
    Set<SalePolicyExecutorLadderVarVo> vars = salePolicyExecutorLadder.getExecutorLadderVars();
    Map<String, Object> varParams = super.mappingVariableValues(vars); 
    
    // ======== 以下正式开始进行计算
    return this.execute(abstractCycleExecuteContext, index, times, varParams , salePolicyExecutorLadder);
  }
  
  /**
   * 真正处理DMS标品中“折扣”类优惠政策保存的业务逻辑
   */
  protected void handleSaveSalePolicyExecutorInfo(boolean update, SalePolicyVo currentSalePolicy, SalePolicyVo oldSalePolicy, Set<SalePolicyExecutorVo> salePolicyExecutorInfoVos) {
    /*
     * 这里的运维步骤包括两个：
     * 1、如果当前运维场景是优惠政策修改，在首先根据oldSalePolicy中的优惠政策业务编号，进行历史执行信息的删除（再进行第2步）
     * 2、如果当前维护场景属于优惠信息创建，则进行优惠执行信息的添加即可
     * */
    
    // 1、=====
    if(update) {
      Set<AbstractSalePolicyExecutorInfo> exsitSalePolicyExecutors = oldSalePolicy.getSalePolicyExecutorInfos();
      // 如果条件成立，说明就是要先进行删除
      if(!CollectionUtils.isEmpty(exsitSalePolicyExecutors)) {
        String[] salePolicyExecutorIds = exsitSalePolicyExecutors.stream().map(AbstractSalePolicyExecutorInfo::getId).toArray(String[]::new);
        List<SalePolicyExecutorLadder> exsitSalePolicyExecutorLadders = this.salePolicyExecutorLadderRepository.findBySalePolicyExecutorIds(salePolicyExecutorIds);
        // 删除每一个相关执行阶梯中的变量信息
        if(!CollectionUtils.isEmpty(exsitSalePolicyExecutorLadders)) {
          String[] existSalePolicyExecutorLadderIds = exsitSalePolicyExecutorLadders.stream().map(SalePolicyExecutorLadder::getId).toArray(String[]::new);
          if(existSalePolicyExecutorLadderIds != null && existSalePolicyExecutorLadderIds.length > 0) {
            this.salePolicyExecutorLadderGiftRepository.deleteByExecutorLadderIds(existSalePolicyExecutorLadderIds);
            this.salePolicyExecutorLadderVarRepository.deleteByExecutorLadderIds(existSalePolicyExecutorLadderIds);
          }
        }
        this.salePolicyExecutorLadderRepository.deleteBySalePolicyExecutorIds(salePolicyExecutorIds);
        this.salePolicyExecutorRepository.deleteByIds(Sets.newHashSet(salePolicyExecutorIds));
      }
    }
    
    // 2、======= 
    // 开始转换实体信息
    Collection<SalePolicyExecutor> salePolicyExecutors = this.nebulaToolkitService.copyCollectionByWhiteList(salePolicyExecutorInfoVos , SalePolicyExecutorVo.class, SalePolicyExecutor.class, LinkedHashSet.class, ArrayList.class, "salePolicyExecutorLadders", "salePolicyExecutorLadders.executorLadderVars" , "salePolicyExecutorLadders.executorLadderGifts");
    // 检测这些salePolicyExecutors，executorCode是否都填写了，并有没有重复
    long count = salePolicyExecutors.stream().filter(item -> StringUtils.isBlank(item.getExecutorCode())).count();
    Validate.isTrue(count == 0 , "当保存优惠政策中的执行策略信息时，其中的执行信息业务编号executorCode必须填写，请检查!!");
    count = salePolicyExecutors.stream().map(SalePolicyExecutor::getExecutorCode).distinct().count();
    Validate.isTrue(count == salePolicyExecutors.size() , "当保存优惠政策中的执行策略信息时，发现至少有两组执行信息的业务编号executorCode重复，请检查!!");
    // 一个一个进行创建(只创建executeStrategyCode正确的执行器)
    for (SalePolicyExecutor salePolicyExecutor : salePolicyExecutors) {
      if(StringUtils.equals(salePolicyExecutor.getExecuteStrategyCode(), this.getExecuteStrategyCode())) {
        this.salePolicyExecutorService.create(salePolicyExecutor);
      }
    }
  }
  
  /**
   * 该方法用于验证将要创建或者修改的优惠政策信息中，是否携带了正确的阶梯变量信息
   * @param currentSalePolicy
   */
  protected void validatetExpressionParams(SalePolicyVo currentSalePolicy) {
    Set<AbstractSalePolicyExecutorInfo> salePolicyExecutorInfos = currentSalePolicy.getSalePolicyExecutorInfos();
    Map<Integer, String> expressionParamMapping = AnalyzeExpressionUtils.analyzeLadderExpressionMapping(getExpression());
    // 如果条件成立，说明表达式不需要填写任何变量信息，也就没有必要再进行验证了
    if(CollectionUtils.isEmpty(expressionParamMapping)) {
      return;
    }
    Set<String> expressionParamNames = expressionParamMapping.values().stream().distinct().collect(Collectors.toSet());
    for (AbstractSalePolicyExecutorInfo abstractSalePolicyExecutorInfo : salePolicyExecutorInfos) {
      SalePolicyExecutorVo salePolicyExecutorInfo = (SalePolicyExecutorVo) abstractSalePolicyExecutorInfo;
      // 如果这个阶梯并不是当前执行策略关注的执行阶梯（依据getExecuteStrategyCode()进行判定，则不进行验证）
      if(!StringUtils.equals(salePolicyExecutorInfo.getExecuteStrategyCode(), this.getExecuteStrategyCode())) {
        continue;
      }
      Set<SalePolicyExecutorLadderVo> salePolicyExecutorLadders = salePolicyExecutorInfo.getSalePolicyExecutorLadders();
      Iterator<SalePolicyExecutorLadderVo> iterator = salePolicyExecutorLadders.iterator();
      // 一个一个阶梯都要验证变量填写的情况
      for(int ladderIndex = 0 ; ladderIndex < salePolicyExecutorLadders.size() ; ladderIndex++) {
        SalePolicyExecutorLadderVo salePolicyExecutorLadder = iterator.next();
        Set<SalePolicyExecutorLadderVarVo> ladderVars = salePolicyExecutorLadder.getExecutorLadderVars();
        // 如果条件不成立，那肯定有问题
        Validate.isTrue(!CollectionUtils.isEmpty(ladderVars) , "未发现正确的解体变量信息，请检查!!");
        Set<String> ladderVarNames = ladderVars.stream().map(SalePolicyExecutorLadderVarVo::getVariableName).collect(Collectors.toSet());
        Validate.isTrue(Sets.difference(expressionParamNames, ladderVarNames).isEmpty() , "阶梯变量名匹配存在问题，请检查!!");
      }
    }
  }
}
