package com.biz.crm.dms.business.policy.local.init;

import java.util.List;
import java.util.Objects;

import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;

import com.biz.crm.dms.business.policy.sdk.characteristic.CharacteristicStrategy;
import com.biz.crm.dms.business.policy.sdk.register.SalePolicyTempleteRegister;
import com.biz.crm.dms.business.policy.sdk.strategy.SalePolicyCycleRuleStrategy;
import com.biz.crm.dms.business.policy.sdk.strategy.SalePolicyExecuteStrategy;
import com.biz.crm.dms.business.policy.sdk.strategy.SalePolicyLimitStrategy;
import com.biz.crm.dms.business.policy.sdk.strategy.SalePolicyStickupListener;
import com.biz.crm.dms.business.policy.sdk.vo.AbstractCharacteristicInfo;
import com.biz.crm.dms.business.policy.sdk.vo.AbstractSalePolicyExecutorInfo;
import com.biz.crm.dms.business.policy.sdk.vo.AbstractSalePolicyLimitInfo;
import com.biz.crm.dms.business.policy.sdk.vo.AbstractSalePolicyProductInfo;
import com.biz.crm.dms.business.policy.sdk.vo.AbstractSalePolicyThreshold;
import com.bizunited.nebula.init.service.InitProcessEnvironmentService;

/**
 * 这个初始化逻辑主要是为了验证当前应用程序中所有的优惠政策模板SalePolicyTempleteRegister，相互之间的注册信息是不冲突的
 * @author yinwenjie
 */
@Component
public class ValidateTempleteInitProcessEnvironmentService implements InitProcessEnvironmentService {
  /**
   * 表示当前应用程序的所有优惠政策模板是不是完成了可用性验证
   */
  private volatile boolean verified = false;
  @Autowired(required = false)
  private List<SalePolicyExecuteStrategy<? extends AbstractSalePolicyExecutorInfo>> salePolicyExecuteStrategies;
  @Autowired(required = false)
  private List<SalePolicyCycleRuleStrategy> salePolicyCycleRuleStrategies;
  @Autowired(required = false)
  private List<SalePolicyLimitStrategy<? extends AbstractSalePolicyLimitInfo>> salePolicyLimitStrategies;
  @Autowired(required = false)
  private List<SalePolicyStickupListener<? extends AbstractSalePolicyThreshold, ? extends AbstractSalePolicyProductInfo>> salePolicyStickupListeners;
  @Autowired(required = false)
  private List<SalePolicyTempleteRegister> salePolicyTempleteRegisters;
  @Autowired(required = false)
  private List<CharacteristicStrategy<? extends AbstractCharacteristicInfo>> characteristicStraties;
  
  @Override
  public void initForAppCode(String appCode) {
    /*
     * 整个验证过程为：
     * 1、验证所有“优惠政策规则执行策略”SalePolicyExecuteStrategy的正确性，包括：
     * executeStrategyCode都有值且不重复、表达式都有值、都有salePolicyExecutorInfoClass类型信息
     * 都有执行分摊策略salePolicyExecuteShareStrategy、都有指定阶梯循环器cycleRuleStrategyClasses
     * 
     * 2、验证所有“阶梯循环策略”SalePolicyCycleRuleStrategy都是正确的，包括：
     * 各个阶梯循环策略的cycleRuleCode都有值，且不重复，实例说明cycleRuleDesc都有值
     * 
     * 3、验证所有限量政策“SalePolicyLimitStrategy”都是正确的，包括：
     * 各个限量政策limitStrategyCode都有值，且不重复，说明信息limitStrategyDesc都有值
     * 表达式信息expression都有值，限量政策模型salePolicyLimitInfoClass都有值
     * 
     * 4、验证所有基础信息维护监听SalePolicyStickupListener都是正确的，包括：
     * 都有salePolicyProductThresholdClass(门槛信息模型)，都有salePolicyProductInfoClass（商品信息模型）
     * 
     * 5、验证所有模板注册器SalePolicyTempleteRegister都是正确的，包括：
     * 都有type信息，且不重复；都有模板描述信息typeDesc，都设定了supportProduct属性；都设定了商品支持方式productSelectionMethods
     * 都设定了customerScopeStrategyClasses客户选择返回，都设定了executeStrategyClasses执行策略类型
     * 
     * 6、验证所有的特性执行策略CharacteristicStrategy都是正确的，包括：
     * 都有characteristicCode信息，且不重复；
     * 都设定了characteristicInfoClass，且不重复
     * */
    
    // 1、=======
    long executeStrategyCodeCount = salePolicyExecuteStrategies.stream().map(SalePolicyExecuteStrategy::getExecuteStrategyCode).filter(code -> StringUtils.isNotBlank(code)).distinct().count();
    long executeStrategyDescCount = salePolicyExecuteStrategies.stream().map(SalePolicyExecuteStrategy::getExecuteStrategyDesc).filter(desc -> StringUtils.isNotBlank(desc)).count();
    long expressionCount = salePolicyExecuteStrategies.stream().map(SalePolicyExecuteStrategy::getExpression).filter(expression -> StringUtils.isNotBlank(expression)).count();
    long salePolicyExecutorInfoClassCount = salePolicyExecuteStrategies.stream().map(SalePolicyExecuteStrategy::getSalePolicyExecutorInfoClass).filter(item -> Objects.nonNull(item)).count();
    long salePolicyExecuteShareStrategyCount = salePolicyExecuteStrategies.stream().map(SalePolicyExecuteStrategy::getSalePolicyExecuteShareStrategy).filter(item -> Objects.nonNull(item)).count();
    long cycleRuleStrategyClassesGroupCount = salePolicyExecuteStrategies.stream().map(SalePolicyExecuteStrategy::getCycleRuleStrategyClasses).filter(item -> !CollectionUtils.isEmpty(item)).count();
    Validate.isTrue(executeStrategyCodeCount == salePolicyExecuteStrategies.size() , "至少有一个优惠政策规则执行策略（SalePolicyExecuteStrategy）没有正确设定executeStrategyCode信息（不能为空，且不能重复）。");
    Validate.isTrue(executeStrategyDescCount == salePolicyExecuteStrategies.size() , "至少有一个优惠政策规则执行策略（SalePolicyExecuteStrategy）没有正确设定executeStrategyDesc信息（不能为空）。");
    Validate.isTrue(expressionCount == salePolicyExecuteStrategies.size() , "至少有一个优惠政策规则执行策略（SalePolicyExecuteStrategy）没有正确设定expression信息（不能为空）。");
    Validate.isTrue(salePolicyExecutorInfoClassCount == salePolicyExecuteStrategies.size() , "至少有一个优惠政策规则执行策略（SalePolicyExecuteStrategy）没有正确设定salePolicyExecutorInfoClass信息（不能为空）。");
    Validate.isTrue(salePolicyExecuteShareStrategyCount == salePolicyExecuteStrategies.size() , "至少有一个优惠政策规则执行策略（SalePolicyExecuteStrategy）没有正确设定salePolicyExecuteShareStrategy分享策略信息（不能为空）。");
    Validate.isTrue(cycleRuleStrategyClassesGroupCount == salePolicyExecuteStrategies.size() , "至少有一个优惠政策规则执行策略（SalePolicyExecuteStrategy）没有正确设定cycleRuleStrategyClasse信息（不能为空）。");
  
    // 2、=======
    long cycleRuleCodeCount = salePolicyCycleRuleStrategies.stream().map(SalePolicyCycleRuleStrategy::getCycleRuleCode).filter(code -> StringUtils.isNotBlank(code)).distinct().count();
    long cycleRuleDescCount = salePolicyCycleRuleStrategies.stream().map(SalePolicyCycleRuleStrategy::getCycleRuleDesc).filter(desc -> StringUtils.isNotBlank(desc)).count();
    Validate.isTrue(cycleRuleCodeCount == salePolicyCycleRuleStrategies.size() , "至少有一个优惠政策阶梯循环策略（SalePolicyCycleRuleStrategy）没有正确设定cycleRuleCode信息（不能为空，且不能重复）。");
    Validate.isTrue(cycleRuleDescCount == salePolicyCycleRuleStrategies.size() , "至少有一个优惠政策阶梯循环策略（SalePolicyCycleRuleStrategy）没有正确设定cycleRuleDesc信息（不能为空）。");
    
    // 3、======
    long limitStrategyCodeCount = salePolicyLimitStrategies.stream().map(SalePolicyLimitStrategy::getLimitStrategyCode).filter(code -> StringUtils.isNotBlank(code)).distinct().count();
    long limitStrategyDescCount = salePolicyLimitStrategies.stream().map(SalePolicyLimitStrategy::getLimitStrategyDesc).filter(desc -> StringUtils.isNotBlank(desc)).count();
    expressionCount = salePolicyLimitStrategies.stream().map(SalePolicyLimitStrategy::getExpression).filter(expression -> StringUtils.isNotBlank(expression)).count();
    long salePolicyLimitInfoClassCount = salePolicyLimitStrategies.stream().map(SalePolicyLimitStrategy::getSalePolicyLimitInfoClass).filter(salePolicyLimitInfoClass -> Objects.nonNull(salePolicyLimitInfoClass)).count();
    Validate.isTrue(limitStrategyCodeCount == salePolicyLimitStrategies.size() , "至少有一个优惠限量政策（SalePolicyLimitStrategy）没有正确设定limitStrategyCode信息（不能为空，且不能重复）。");
    Validate.isTrue(limitStrategyDescCount == salePolicyLimitStrategies.size() , "至少有一个优惠限量政策（SalePolicyLimitStrategy）没有正确设定limitStrategyDesc信息（不能为空）。");
    Validate.isTrue(expressionCount == salePolicyLimitStrategies.size() , "至少有一个优惠限量政策（SalePolicyLimitStrategy）没有正确设定expressionCount信息（不能为空）。");
    Validate.isTrue(salePolicyLimitInfoClassCount == salePolicyLimitStrategies.size() , "至少有一个优惠限量政策（SalePolicyLimitStrategy）没有正确设定salePolicyLimitInfoClass（限量政策模型）信息（不能为空）。");
    
    // 4、======
    long salePolicyProductThresholdClassCount = salePolicyStickupListeners.stream().map(SalePolicyStickupListener::getSalePolicyProductThresholdClass).filter(salePolicyProductThresholdClass -> Objects.nonNull(salePolicyProductThresholdClass)).count();
    long salePolicyProductInfoClassCount = salePolicyStickupListeners.stream().map(SalePolicyStickupListener::getSalePolicyProductInfoClass).filter(salePolicyProductInfoClass -> Objects.nonNull(salePolicyProductInfoClass)).count();
    Validate.isTrue(salePolicyProductThresholdClassCount == salePolicyStickupListeners.size() , "至少有一个优惠政策数据维护策略（SalePolicyStickupListener）没有正确设定salePolicyProductThresholdClass信息（不能为空）。");
    Validate.isTrue(salePolicyProductInfoClassCount == salePolicyStickupListeners.size() , "至少有一个优惠政策数据维护策略（SalePolicyStickupListener）没有正确设定salePolicyProductInfoClass信息（不能为空）。");
    
    // 5、======
    long typeCount = salePolicyTempleteRegisters.stream().map(SalePolicyTempleteRegister::getType).filter(type -> StringUtils.isNotBlank(type)).distinct().count();
    long descCount = salePolicyTempleteRegisters.stream().map(SalePolicyTempleteRegister::getTypeDesc).filter(desc -> StringUtils.isNotBlank(desc)).count();
    long supportProductCount = salePolicyTempleteRegisters.stream().map(SalePolicyTempleteRegister::supportProduct).filter(supportProduct -> Objects.nonNull(supportProduct)).count();
    long productSelectionMethodsCount = salePolicyTempleteRegisters.stream().map(SalePolicyTempleteRegister::getProductSelectionMethods).filter(productSelectionMethods -> Objects.nonNull(productSelectionMethods) && productSelectionMethods.length > 0).count();
    long stickupListenersCount = salePolicyTempleteRegisters.stream().map(SalePolicyTempleteRegister::getStickupListener).filter(stickupListeners -> Objects.nonNull(stickupListeners)).count();
    long customerScopeStrategyClassesCount = salePolicyTempleteRegisters.stream().map(SalePolicyTempleteRegister::getCustomerScopeStrategyClasses).filter(customerScopeStrategyClasses -> !CollectionUtils.isEmpty(customerScopeStrategyClasses)).count();
    long executeStrategyClassesCount = salePolicyTempleteRegisters.stream().map(SalePolicyTempleteRegister::getExecuteStrategyClasses).filter(executeStrategyClasses -> !CollectionUtils.isEmpty(executeStrategyClasses)).count();
    Validate.isTrue(typeCount == salePolicyTempleteRegisters.size() , "至少有一个优惠政策模板（SalePolicyTempleteRegister）没有正确设定type信息（不能为空，且不能重复）。");
    Validate.isTrue(descCount == salePolicyTempleteRegisters.size() , "至少有一个优惠政策模板（SalePolicyTempleteRegister）没有正确设定typeDesc信息（不能为空）。");
    Validate.isTrue(supportProductCount == salePolicyTempleteRegisters.size() , "至少有一个优惠政策模板（SalePolicyTempleteRegister）没有正确设定supportProduct信息（不能为空）。");
    Validate.isTrue(productSelectionMethodsCount == salePolicyTempleteRegisters.size() , "至少有一个优惠政策模板（SalePolicyTempleteRegister）没有正确设定productSelectionMethods（商品选择方式）信息（不能为空）。");
    Validate.isTrue(stickupListenersCount == salePolicyTempleteRegisters.size() , "至少有一个优惠政策模板（SalePolicyTempleteRegister）没有正确设定stickupListener（数据运维监听）信息（不能为空）。");
    Validate.isTrue(customerScopeStrategyClassesCount == salePolicyTempleteRegisters.size() , "至少有一个优惠政策模板（SalePolicyTempleteRegister）没有正确设定customerScopeStrategyClasses（客户范围选择）信息（不能为空）。");
    Validate.isTrue(executeStrategyClassesCount == salePolicyTempleteRegisters.size() , "至少有一个优惠政策模板（SalePolicyTempleteRegister）没有正确设定executeStrategyClasses（执行策略）信息（不能为空）。");
    
    // 6、=====
    if(!CollectionUtils.isEmpty(characteristicStraties)) {
      long characteristicCodeConut = characteristicStraties.stream().map(CharacteristicStrategy::characteristicCode).filter(item -> StringUtils.isNotBlank(item)).distinct().count();
      long characteristicClassCount = characteristicStraties.stream().map(CharacteristicStrategy::getCharacteristicInfoClass).filter(item -> item != null).distinct().count();
      Validate.isTrue(characteristicCodeConut == characteristicStraties.size() , "至少有一个优惠政策特性执行策略（CharacteristicStrategy）没有正确设定characteristicCode信息（不能为空，且不能重复）。");
      Validate.isTrue(characteristicClassCount == characteristicStraties.size() , "至少有一个优惠政策特性执行策略（CharacteristicStrategy）没有正确设定characteristicInfoClass信息（不能为空，且不能重复）。");
    }
  }

  @Override
  public boolean doInitForAppCode(String appCode) {
    // 这个验证过程并不是每一次顶级租户启动的时候，都需要做，每一次应用程序启动都只需要验证一次
    if(!verified) {
      synchronized (this) {
        if(verified) {
          return false;
        }
        return verified = true;
      }
    }
    return false;
  }

}
