package com.biz.crm.dms.business.policy.local.repository;

import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.biz.crm.dms.business.policy.local.entity.SalePolicy;
import com.biz.crm.dms.business.policy.local.mapper.SalePolicyMapper;
import com.biz.crm.dms.business.policy.sdk.dto.SalePolicyPageDto;

import java.util.List;
import java.util.Set;

import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Component;

/**
 * sale_policy 优惠政策基本数据(repository)
 * @author yinwenjie
 */
@Component
public class SalePolicyRepository extends ServiceImpl<SalePolicyMapper, SalePolicy> {
  /**
   * 按照优惠政策的业务编号，查询优惠政策的基本信息（只包括基本信息）。
   * @param salePolicyCode 优惠政策的业务编号
   * @param tenantCode 当前二级租户的业务编号
   */
  public SalePolicy findBySalePolicyCodeAndTenantCode(String salePolicyCode , String tenantCode) {
    SalePolicy salePolicy = this.lambdaQuery()
        .eq(SalePolicy::getTenantCode, tenantCode)
        .eq(SalePolicy::getSalePolicyCode , salePolicyCode).one();
    return salePolicy;
  }
  
  /**
   * 按照优惠政策业务编号、以及租户信息查询已排除了某个id匹配的优惠政策信息
   * @param salePolicyCode
   * @param tenantCode
   * @param id 被排除的id
   * @return
   */
  public SalePolicy findBySalePolicyCodeAndTenantCodeWithoutId(String salePolicyCode , String tenantCode , String id) {
    SalePolicy salePolicy = this.lambdaQuery().eq(SalePolicy::getTenantCode, tenantCode)
        .eq(SalePolicy::getSalePolicyCode , salePolicyCode)
        .notIn(SalePolicy::getId, id)
        .one();
    return salePolicy;
  }
  
  /**
   * 根据id进行优惠政策基本信息的查询（注意，只查询基本信息）
   * @param id
   * @return
   */
  public SalePolicy findBySalePolicyId(String id) {
    SalePolicy salePolicy = this.lambdaQuery()
        .eq(SalePolicy::getId, id)
        .one();
    return salePolicy;
  }
  
  /**
   * 根据优惠政策使用的模板进行查询
   * @param templeteCode
   * @return
   */
  public List<SalePolicy> findByTempleteCodeAndTenantCode(String templeteCode , String tenantCode) {
    return this.lambdaQuery()
        .eq(SalePolicy::getTempleteCode, templeteCode)
        .eq(SalePolicy::getTenantCode, tenantCode)
        .list();
  }
  
  /**
   * 加载某个租户下的所有优惠政策信息，无论它们是否有效（该方法主要是配合缓存加载功能进行使用）
   * @param tenantCode
   * @return
   */
  public List<SalePolicy> findByTenantCode(String tenantCode) {
    return this.lambdaQuery()
        .eq(SalePolicy::getTenantCode, tenantCode)
        .orderByDesc(SalePolicy::getValidStartTime).list();
  }
  
  /**
   * 针对特定二级租户的已存在的优惠政策分页查询信息
   */
  public Page<SalePolicy> findByConditions(Pageable pageable, SalePolicyPageDto dto) {
    Page<SalePolicy> page = new Page<>(pageable.getPageNumber(), pageable.getPageSize());
    return this.baseMapper.findByConditions(page, dto);
  }
  
  /**
   * 按照id集合进行已有信息的刪除操作
   * @param ids id集合
   */
  public void deleteByIds(Set<String> ids) {
    LambdaQueryWrapper<SalePolicy> lambdaQuery = Wrappers.lambdaQuery();
    lambdaQuery.in(SalePolicy::getId, ids);
    this.baseMapper.delete(lambdaQuery);
  }
}
