package com.biz.crm.dms.business.policy.local.repository;

import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.conditions.update.LambdaUpdateWrapper;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.biz.crm.dms.business.policy.local.entity.SalePolicyScopeInfo;
import com.biz.crm.dms.business.policy.local.mapper.SalePolicyScopeInfoMapper;
import com.google.common.collect.Lists;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;

import java.util.List;

/**
 * sale_policy 优惠政策中，标品模块使用的优惠政策客户范围信息数据(repository)
 * @author yinwenjie
 */
@Component
public class SalePolicyScopeInfoRepository extends ServiceImpl<SalePolicyScopeInfoMapper, SalePolicyScopeInfo> {  
  /**
   * 按照范围类型、租户和业务编号，启用特定的优惠政策范围关联信息
   * @param customerScopeType 指定的优惠政策关联方式
   * @param tenantCode 二级租户信息
   * @param codes 业务编号（如组织机构编号、经销商编号等）
   */
  public void effectiveByCustomerScopeTypeAndTenantCodeAndCodes(String customerScopeType ,  String tenantCode, String[] codes) {
    LambdaUpdateWrapper<SalePolicyScopeInfo> lambdaUpdate = Wrappers.<SalePolicyScopeInfo>lambdaUpdate();
    lambdaUpdate.set(SalePolicyScopeInfo::getState, 1)
      .eq(SalePolicyScopeInfo::getTenantCode, tenantCode)
      .eq(SalePolicyScopeInfo::getCustomerScopeType, customerScopeType)
      .in(SalePolicyScopeInfo::getCode, Lists.newArrayList(codes));
    this.update(lambdaUpdate);
  }
  
  /**
   * 按照范围类型、租户和业务编号，禁用特定的优惠政策范围关联信息
   * @param customerScopeType 指定的优惠政策关联方式
   * @param tenantCode 二级租户信息
   * @param codes 业务编号（如组织机构编号、经销商编号等）
   */
  public void invalidByCustomerScopeTypeAndTenantCodeAndCodes(String customerScopeType ,  String tenantCode, String[] codes) {
    LambdaUpdateWrapper<SalePolicyScopeInfo> lambdaUpdate = Wrappers.<SalePolicyScopeInfo>lambdaUpdate();
    lambdaUpdate.set(SalePolicyScopeInfo::getState, 0)
      .eq(SalePolicyScopeInfo::getTenantCode, tenantCode)
      .eq(SalePolicyScopeInfo::getCustomerScopeType, customerScopeType)
      .in(SalePolicyScopeInfo::getCode, Lists.newArrayList(codes));
    this.update(lambdaUpdate);
  }
  
  /**
   * 按照数据的技术编号，删除优惠政策选择范围信息
   * @param ids
   */
  public void deleteByIds(List<String> ids) {
    if(CollectionUtils.isEmpty(ids)) {
      return;
    }
    LambdaQueryWrapper<SalePolicyScopeInfo> lambdaQuery = Wrappers.<SalePolicyScopeInfo>lambdaQuery();
    lambdaQuery.in(SalePolicyScopeInfo::getId, Lists.newArrayList(ids));
    this.baseMapper.delete(lambdaQuery);
  }

  /**
   * 按照范围类型、租户和业务编号，删除特定的优惠政策范围关联信息
   * @param customerScopeType 指定的优惠政策关联方式
   * @param tenantCode 二级租户信息
   * @param codes 业务编号（如组织机构编号、经销商编号等）
   */
  public void deleteByCustomerScopeTypeAndTenantCodeAndCodes(String customerScopeType ,  String tenantCode, String[] codes) {
    LambdaQueryWrapper<SalePolicyScopeInfo> lambdaQuery = Wrappers.<SalePolicyScopeInfo>lambdaQuery();
    lambdaQuery
      .eq(SalePolicyScopeInfo::getTenantCode, tenantCode)
      .eq(SalePolicyScopeInfo::getCustomerScopeType, customerScopeType)
      .in(SalePolicyScopeInfo::getCode, Lists.newArrayList(codes));
    this.baseMapper.delete(lambdaQuery);
  }
  
  /**
   * 根据租户编码、优惠政策业务编号、客户范围控制策略的识别类型号查询数据集合
   * @param tenantCode 租户编码
   * @param salePolicyCode 优惠政策业务编号
   * @param customerScopeType 客户范围控制策略的识别类型号
   * @return
   */
  public List<SalePolicyScopeInfo> findByTenantCodeAndSalePolicyCodeAndCustomerScopeType(String tenantCode, String salePolicyCode, String customerScopeType) {
   return this.lambdaQuery()
       .eq(SalePolicyScopeInfo::getTenantCode, tenantCode)
       .eq(SalePolicyScopeInfo::getSalePolicyCode, salePolicyCode)
       .eq(SalePolicyScopeInfo::getCustomerScopeType, customerScopeType)
       .list();
  }
}
