package com.biz.crm.dms.business.policy.local.repository;

import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.dms.business.policy.local.entity.SalePolicyTemplete;
import com.biz.crm.dms.business.policy.local.mapper.SalePolicyTempleteMapper;
import com.biz.crm.dms.business.policy.sdk.dto.SalePolicyTempletePageDto;
import com.google.common.collect.Lists;

import java.util.List;
import java.util.Set;

import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;

/**
 * SalePolicyTemplete 优惠政策模板基本数据(repository)
 * @author yinwenjie
 */
@Component
public class SalePolicyTempleteRepository extends ServiceImpl<SalePolicyTempleteMapper, SalePolicyTemplete> {
  @Autowired(required = false)
  private SalePolicyTempleteJpaRepository salePolicyTempleteJpaRepository;
  
  /**
   * 按照指定的优惠政策模板进行模板详情查询
   * @param templteCode 指定的优惠政策模板编号
   * @param tenantCode 二级租户信息
   * @return
   */
  public SalePolicyTemplete findByTempleteCode(String templeteCode , String tenantCode) {
    return this.salePolicyTempleteJpaRepository.findByTempleteCode(templeteCode, tenantCode);
  }
  
  /**
   * 按照一组优惠政策模板业务编号，查询相关的优惠政策模板数据
   * @param templeteCodes 一个或者多个优惠政策模板编号
   * @param tenantCode 二级租户信息
   * @return
   */
  public List<SalePolicyTemplete> findByTempleteCodes(List<String> templeteCodes , String tenantCode) {
    Set<SalePolicyTemplete> set = this.salePolicyTempleteJpaRepository.findByTempleteCodes(templeteCodes, tenantCode);
    if(CollectionUtils.isEmpty(set)) {
      return null;
    }
    return Lists.newArrayList(set);
  }
  
  /**
   * 按照优惠政策模板的id技术编号进行查询
   * @param id
   * @return
   */
  public SalePolicyTemplete findById(String id) {
    return this.lambdaQuery().eq(SalePolicyTemplete::getId, id).one();
  }
  
  /**
   * 按照二级租户查询已经常见的所有优惠政策模板（按照创建时间排序）
   * @param tenantCode
   * @return
   */
  public Set<SalePolicyTemplete> findByTenantCode(String tenantCode , EnableStatusEnum enableStatus) {
    if(StringUtils.isBlank(tenantCode)) {
      return null;
    }
    if(enableStatus == null) {
      return salePolicyTempleteJpaRepository.findByTenantCode(tenantCode);
    } else {
      return salePolicyTempleteJpaRepository.findByTenantCodeAndEnableStatus(tenantCode , enableStatus.getCode());
    }
  }
  
  /**
   * 针对特定二级租户的已存在的优惠政策模板分页查询信息
   */
  public Page<SalePolicyTemplete> findByConditions(Pageable pageable, SalePolicyTempletePageDto dto) {
    Page<SalePolicyTemplete> page = new Page<>(pageable.getPageNumber(), pageable.getPageSize());
    if(StringUtils.equals(dto.getTempleteName() , "")) { 
      dto.setTempleteName(null);
    } 
    if(StringUtils.equals(dto.getTempleteCode() , "")) { 
      dto.setTempleteCode(null);
    } 
    if(StringUtils.equals(dto.getEnableStatus() , "")) { 
      dto.setEnableStatus(null);
    } 
    return this.baseMapper.findByConditions(page, dto);
  }
  
  public void deleteByTempleteCode(String templeteCode , String tenantCode) {
    this.salePolicyTempleteJpaRepository.deleteByTempleteCode(templeteCode , tenantCode);
  }
}
