package com.biz.crm.dms.business.policy.local.scopestrategy;

import com.biz.crm.dms.business.policy.local.repository.SalePolicyScopeInfoRepository;
import com.biz.crm.dms.business.policy.local.service.task.SalePolicyCacheLoadingService;
import com.biz.crm.dms.business.policy.sdk.vo.AbstractSalePolicyCustomerScopeInfo;
import com.google.common.collect.Lists;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.util.CollectionUtils;

import java.util.Set;

/**
 * 这个抽象类的作用只是为了帮助标品中的几个具体的客户范围选择策略，管理重复度比较高的代码</br>
 * 如果是项目上自行实现的某种具体的客户范围选择策略，则这个抽象类没有参考价值
 * @author yinwenjie
 */
public abstract class AbstractSalePolicyCustomerScopeStrategy {
  
  @Autowired(required = false)
  private SalePolicyScopeInfoRepository salePolicyScopeInfoRepository;
  @Autowired(required = false)
  private SalePolicyCacheLoadingService salePolicyCacheLoadingService;
  
  public abstract String getScopeType();
  
  /**
   * 按照设定的客户适用范围信息，进行删除
   * @param currentScopeType
   * @param salePolicyCustomerInfos
   */
  protected void deleteSalePolicyCustomerInfos(String currentScopeType , Set<? extends AbstractSalePolicyCustomerScopeInfo> salePolicyCustomerInfos) {
    if(!CollectionUtils.isEmpty(salePolicyCustomerInfos)) {
      String[] salePolicyCustomerInfoIds = salePolicyCustomerInfos.stream().filter(item -> StringUtils.equals(currentScopeType, item.getCustomerScopeType())).map(AbstractSalePolicyCustomerScopeInfo::getId).toArray(String[]::new);
      // 如果条件成立，说明要进行删除了
      if(salePolicyCustomerInfoIds != null && salePolicyCustomerInfoIds.length > 0) {
        this.salePolicyScopeInfoRepository.deleteByIds(Lists.newArrayList(salePolicyCustomerInfoIds));
      }
    }
  }
  
  // 请参见SalePolicyCustomerScopeStrategy结构中的说明
  public void onEffective(String tenantCode, String[] codes) {
    /*
     * 当业务信息被启用时，这里的选择范围关联也要被启用。
     * 例如，组织机构被启用时，组织机构关联的优惠政策就应该被启用，并且缓存需要重新刷新
     * */
    String customerScopeType = this.getScopeType();
    if(StringUtils.isBlank(tenantCode) || codes == null || codes.length == 0) {
      return;
    }
    this.salePolicyScopeInfoRepository.effectiveByCustomerScopeTypeAndTenantCodeAndCodes(customerScopeType, tenantCode, codes);
    this.salePolicyCacheLoadingService.notifyCacheRefresh(tenantCode);
  }

  // 请参见SalePolicyCustomerScopeStrategy结构中的说明
  public void onInvalid(String tenantCode, String[] codes) {
    /*
     * 当业务信息被启用时，这里的选择范围关联也要被启用。
     * 例如，组织机构被启用时，组织机构关联的优惠政策就应该被启用，并且缓存需要重新刷新
     * */
    String customerScopeType = this.getScopeType();
    if(StringUtils.isBlank(tenantCode) || codes == null || codes.length == 0) {
      return;
    }
    this.salePolicyScopeInfoRepository.invalidByCustomerScopeTypeAndTenantCodeAndCodes(customerScopeType, tenantCode, codes);
    this.salePolicyCacheLoadingService.notifyCacheRefresh(tenantCode);
  }

  // 请参见SalePolicyCustomerScopeStrategy结构中的说明
  public void onDelete(String tenantCode, String[] codes) {
    String customerScopeType = this.getScopeType();
    if(StringUtils.isBlank(tenantCode) || codes == null || codes.length == 0) {
      return;
    }
    this.salePolicyScopeInfoRepository.deleteByCustomerScopeTypeAndTenantCodeAndCodes(customerScopeType, tenantCode, codes);
    this.salePolicyCacheLoadingService.notifyCacheRefresh(tenantCode);
  }
}
