package com.biz.crm.dms.business.policy.local.scopestrategy;

import com.biz.crm.dms.business.policy.local.entity.SalePolicyScopeInfo;
import com.biz.crm.dms.business.policy.local.service.SalePolicyScopeInfoService;
import com.biz.crm.dms.business.policy.local.vo.SalePolicyScopeDistributorInfoVo;
import com.biz.crm.dms.business.policy.sdk.enums.ScopeSelectionMethod;
import com.biz.crm.dms.business.policy.sdk.strategy.SalePolicyCustomerScopeStrategy;
import com.biz.crm.dms.business.policy.sdk.vo.AbstractSalePolicyCustomerScopeInfo;
import com.biz.crm.dms.business.policy.sdk.vo.SalePolicyVo;
import com.biz.crm.mdm.business.customer.sdk.service.CustomerVoService;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.google.common.collect.Sets;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * 该策略支持按照经销商信息来进行优惠政策范围圈定的处理方式
 * TODO 标品提供的默认用户选择方式，还要进行优化
 * @author yinwenjie
 */
@Component
public class DistributorForSalePolicyCustomerScopeStrategy extends AbstractSalePolicyCustomerScopeStrategy implements SalePolicyCustomerScopeStrategy<SalePolicyScopeDistributorInfoVo> {

  @Autowired(required = false)
  private SalePolicyScopeInfoService salePolicyScopeInfoService;

  @Autowired(required = false)
  private NebulaToolkitService nebulaToolkitService;

  @Autowired(required = false)
  private CustomerVoService customerVoService;

  public static final String SCOPE_TYPE = "distributorForSalePolicy";
  
  @Override
  public String getScopeType() {
    return SCOPE_TYPE;
  }

  @Override
  public String getScopeTypeDesc() {
    return "客 户";
  }

  @Override
  public  Class<SalePolicyScopeDistributorInfoVo> getSalePolicyCustomerInfoClass() {
    return SalePolicyScopeDistributorInfoVo.class;
  }
  
  @Override
  public void onSaveSalePolicyCustomerInfo(boolean update , SalePolicyVo currentSalePolicy , SalePolicyVo oldSalePolicy) {
    /*
     * 操作步骤可参考 #OrgForSalePolicyCustomerScopeStrategy.onSaveSalePolicyCustomerInfo 方法中的详细描述
     * */
    // 1、=====
    if(update) {
      Map<String,Set<? extends AbstractSalePolicyCustomerScopeInfo>> oldSalePolicyCustomerInfoMaps = oldSalePolicy.getCustomerScopeMapping();
      if(!CollectionUtils.isEmpty(oldSalePolicyCustomerInfoMaps) && !CollectionUtils.isEmpty(oldSalePolicyCustomerInfoMaps.get(getScopeType()))) {
        this.deleteSalePolicyCustomerInfos(getScopeType(), oldSalePolicyCustomerInfoMaps.get(getScopeType()));
      }
    }
    
    // 2、=====
    // 注意，只添加和自己相关的优惠政策信息
    Map<String,Set<? extends AbstractSalePolicyCustomerScopeInfo>> salePolicyCustomerInfoMaps = currentSalePolicy.getCustomerScopeMapping();
    if(CollectionUtils.isEmpty(salePolicyCustomerInfoMaps) || CollectionUtils.isEmpty(salePolicyCustomerInfoMaps.get(getScopeType()))) {
      return;
    }
    Set<? extends AbstractSalePolicyCustomerScopeInfo> currentSalePolicyCustomerInfos = salePolicyCustomerInfoMaps.get(getScopeType());
    // 修正每一个条信息的冗余值后，开始进行添加
    Set<SalePolicyScopeInfo> salePolicyScopeInfos = Sets.newLinkedHashSet();
    for (AbstractSalePolicyCustomerScopeInfo abstractSalePolicyCustomerInfo : currentSalePolicyCustomerInfos) {
      SalePolicyScopeDistributorInfoVo salePolicyScopeInfoVo = (SalePolicyScopeDistributorInfoVo)abstractSalePolicyCustomerInfo;
      salePolicyScopeInfoVo.setSalePolicyCode(currentSalePolicy.getSalePolicyCode());
      salePolicyScopeInfoVo.setTenantCode(currentSalePolicy.getTenantCode());
      SalePolicyScopeInfo salePolicyScopeInfo = new SalePolicyScopeInfo();
      salePolicyScopeInfo.setSalePolicyCode(currentSalePolicy.getSalePolicyCode());
      salePolicyScopeInfo.setTenantCode(currentSalePolicy.getTenantCode());
      salePolicyScopeInfo.setCode(salePolicyScopeInfoVo.getDistributorCode());
      salePolicyScopeInfo.setName(salePolicyScopeInfoVo.getDistributorName());
      salePolicyScopeInfo.setCustomerScopeType(SCOPE_TYPE);
      salePolicyScopeInfo.setState(1);
      salePolicyScopeInfo.setSelectionMethod(salePolicyScopeInfoVo.getSelectionMethod());
      salePolicyScopeInfos.add(salePolicyScopeInfo);
    }
    this.salePolicyScopeInfoService.createBatch(salePolicyScopeInfos);
  }

  /**
   * 按照经销商信息来进行优惠政策范围圈定的客户信息集合
   * @param tenantCode 指定的二级租户信息
   * @param salePolicyCode 当前构造的具体优惠政策业务编号
   * @return
   */
  @Override
  public Set<SalePolicyScopeDistributorInfoVo> requestSalePolicyCustomerInfo(String tenantCode , String salePolicyCode) {
    if (StringUtils.isAnyBlank(tenantCode, salePolicyCode)) {
      return null;
    }
    //查询经销商范围范围的客户结果
    List<SalePolicyScopeInfo> salePolicyScopeInfos = this.salePolicyScopeInfoService.findByTenantCodeAndSalePolicyCodeAndCustomerScopeType(tenantCode, salePolicyCode, SCOPE_TYPE);
    if (CollectionUtils.isEmpty(salePolicyScopeInfos)) {
      return null;
    }
    //转成经销商范围范围客户结果vo集合
    Set<SalePolicyScopeDistributorInfoVo> salePolicyScopeDistributorInfoVos = new HashSet<>();
    for (SalePolicyScopeInfo salePolicyScopeInfo: salePolicyScopeInfos) {
      SalePolicyScopeDistributorInfoVo salePolicyScopeDistributorInfoVo = this.nebulaToolkitService.copyObjectByWhiteList(salePolicyScopeInfo, SalePolicyScopeDistributorInfoVo.class, HashSet.class, ArrayList.class);
      salePolicyScopeDistributorInfoVo.setDistributorCode(salePolicyScopeInfo.getCode());
      salePolicyScopeDistributorInfoVo.setDistributorName(salePolicyScopeInfo.getName());
      salePolicyScopeDistributorInfoVos.add(salePolicyScopeDistributorInfoVo);
    }
    return salePolicyScopeDistributorInfoVos;
  }

  /**
   * 按照经销商信息来进行优惠政策范围圈定的客户编码集合
   * @param salePolicyCode 需要查询的优惠政策业务编号
   * @param tenantCode 当前二级租户信息
   * @return
   */
  @Override
  public boolean matched(String customerCode, String tenantCode , String salePolicyCode,  Set<SalePolicyScopeDistributorInfoVo> salePolicyCustomerInfos) {
    /**
     * 操作步骤：
     * 1、选定方式分组转成经销信息范围的客户编码map
     * 2、从包含中的客户编码过滤掉非包含客户编码
     * 3、获取经销商信息范围的客户编码
     */
    if (StringUtils.isAnyBlank(tenantCode, salePolicyCode) || CollectionUtils.isEmpty(salePolicyCustomerInfos)) {
      return false;
    }
    //1、======
    Map<String, List<String>> selectionMethodMap = salePolicyCustomerInfos.stream()
        .filter(a -> StringUtils.isNoneBlank(a.getSelectionMethod(), a.getDistributorCode()))
        .collect(Collectors.groupingBy(SalePolicyScopeDistributorInfoVo::getSelectionMethod, Collectors.mapping(SalePolicyScopeDistributorInfoVo::getDistributorCode, Collectors.toList())));
    //2、======
    List<String> includeCodes = selectionMethodMap.get(ScopeSelectionMethod.INCLUDE.getCode());
    List<String> excludeCodes = selectionMethodMap.get(ScopeSelectionMethod.EXCLUDE.getCode());
    if (CollectionUtils.isEmpty(includeCodes)) {
      return false;
    }
    if (!CollectionUtils.isEmpty(excludeCodes)) {
      includeCodes.removeAll(excludeCodes);
    }
    //3、======
    Boolean isExist = this.customerVoService.existByCustomerCode(customerCode);
    return isExist;
  }

}
