package com.biz.crm.dms.business.policy.local.scopestrategy;

import com.biz.crm.dms.business.policy.local.entity.SalePolicyScopeInfo;
import com.biz.crm.dms.business.policy.local.service.SalePolicyScopeInfoService;
import com.biz.crm.dms.business.policy.local.vo.SalePolicyScopeOrgInfoVo;
import com.biz.crm.dms.business.policy.sdk.enums.ScopeSelectionMethod;
import com.biz.crm.dms.business.policy.sdk.strategy.SalePolicyCustomerScopeStrategy;
import com.biz.crm.dms.business.policy.sdk.vo.AbstractSalePolicyCustomerScopeInfo;
import com.biz.crm.dms.business.policy.sdk.vo.SalePolicyVo;
import com.biz.crm.mdm.business.customer.sdk.service.CustomerVoService;
import com.biz.crm.mdm.business.customer.sdk.vo.CustomerVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.google.common.collect.Sets;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;

import javax.transaction.Transactional;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * 该策略支持从企业组织和客户的关联信息中，来选定客户范围
 * TODO 标品提供的默认用户选择方式，还要进行优化
 * @author yinwenjie
 */
@Component
public class OrgForSalePolicyCustomerScopeStrategy extends AbstractSalePolicyCustomerScopeStrategy implements SalePolicyCustomerScopeStrategy<SalePolicyScopeOrgInfoVo> {

  @Autowired(required = false)
  private SalePolicyScopeInfoService salePolicyScopeInfoService;

  @Autowired(required = false)
  private NebulaToolkitService nebulaToolkitService;

  @Autowired(required = false)
  private CustomerVoService customerVoService;
  
  public static final String SCOPE_TYPE = "orgForSalePolicyCustomer";
  
  @Override
  public String getScopeType() {
    return SCOPE_TYPE;
  }

  @Override
  public String getScopeTypeDesc() {
    return "组织机构";
  }

  @Override
  public Class<SalePolicyScopeOrgInfoVo> getSalePolicyCustomerInfoClass() {
    return SalePolicyScopeOrgInfoVo.class;
  }
  
  @Override
  @Transactional
  public void onSaveSalePolicyCustomerInfo(boolean update , SalePolicyVo currentSalePolicy , SalePolicyVo oldSalePolicy) {
    /*
     * 操作步骤为：
     * 1、如果当前操作步骤是“修改”，则需要首先删除oldSalePolicy中和本策略相关的适用范围选择信息
     * 2、再进行添加（注意，只负责添加和本适用范围相关的范围信息）
     * */
    
    // 1、=====
    if(update) {
      Map<String,Set<? extends AbstractSalePolicyCustomerScopeInfo>> oldSalePolicyCustomerInfoMaps = oldSalePolicy.getCustomerScopeMapping();
      if(!CollectionUtils.isEmpty(oldSalePolicyCustomerInfoMaps) && !CollectionUtils.isEmpty(oldSalePolicyCustomerInfoMaps.get(getScopeType()))) {
        this.deleteSalePolicyCustomerInfos(getScopeType(), oldSalePolicyCustomerInfoMaps.get(getScopeType()));
      }
    }
    
    // 2、=====
    // 注意，只添加和自己相关的优惠政策信息
    Map<String,Set<? extends AbstractSalePolicyCustomerScopeInfo>> salePolicyCustomerInfoMaps = currentSalePolicy.getCustomerScopeMapping();
    if(CollectionUtils.isEmpty(salePolicyCustomerInfoMaps) || CollectionUtils.isEmpty(salePolicyCustomerInfoMaps.get(getScopeType()))) {
      return;
    }
    Set<? extends AbstractSalePolicyCustomerScopeInfo> currentSalePolicyCustomerInfos = salePolicyCustomerInfoMaps.get(getScopeType());
    // 修正每一个条信息的冗余值后，开始进行添加
    Set<SalePolicyScopeInfo> salePolicyScopeInfos = Sets.newLinkedHashSet();
    for (AbstractSalePolicyCustomerScopeInfo abstractSalePolicyCustomerInfo : currentSalePolicyCustomerInfos) {
      SalePolicyScopeOrgInfoVo salePolicyScopeInfoVo = (SalePolicyScopeOrgInfoVo)abstractSalePolicyCustomerInfo;
      salePolicyScopeInfoVo.setSalePolicyCode(currentSalePolicy.getSalePolicyCode());
      salePolicyScopeInfoVo.setTenantCode(currentSalePolicy.getTenantCode());
      
      SalePolicyScopeInfo salePolicyScopeInfo = new SalePolicyScopeInfo();
      salePolicyScopeInfo.setSalePolicyCode(currentSalePolicy.getSalePolicyCode());
      salePolicyScopeInfo.setTenantCode(currentSalePolicy.getTenantCode());
      salePolicyScopeInfo.setCode(salePolicyScopeInfoVo.getOrgCode());
      salePolicyScopeInfo.setName(salePolicyScopeInfoVo.getOrgName());
      salePolicyScopeInfo.setCustomerScopeType(SCOPE_TYPE);
      salePolicyScopeInfo.setState(1);
      salePolicyScopeInfo.setSelectionMethod(salePolicyScopeInfoVo.getSelectionMethod());
      salePolicyScopeInfos.add(salePolicyScopeInfo);
    }
    this.salePolicyScopeInfoService.createBatch(salePolicyScopeInfos);
  }

  /**
   * 按照组织机构来进行优惠政策范围圈定的客户信息集合
   * @param tenantCode 指定的二级租户信息
   * @param salePolicyCode 当前构造的具体优惠政策业务编号
   * @return
   */
  @Override
  public Set<SalePolicyScopeOrgInfoVo> requestSalePolicyCustomerInfo(String tenantCode , String salePolicyCode) {
    if (StringUtils.isAnyBlank(tenantCode, salePolicyCode)) {
      return null;
    }
    //查询组织机构范围的客户结果
    List<SalePolicyScopeInfo> salePolicyScopeInfos = this.salePolicyScopeInfoService.findByTenantCodeAndSalePolicyCodeAndCustomerScopeType(tenantCode, salePolicyCode, SCOPE_TYPE);
    if (CollectionUtils.isEmpty(salePolicyScopeInfos)) {
      return null;
    }
    //转成组织机构范围客户结果vo集合
    Set<SalePolicyScopeOrgInfoVo> salePolicyScopeOrgInfoVos = new HashSet<>();
    for (SalePolicyScopeInfo salePolicyScopeInfo: salePolicyScopeInfos) {
      SalePolicyScopeOrgInfoVo salePolicyScopeOrgInfoVo = this.nebulaToolkitService.copyObjectByWhiteList(salePolicyScopeInfo, SalePolicyScopeOrgInfoVo.class, HashSet.class, ArrayList.class);
      salePolicyScopeOrgInfoVo.setOrgCode(salePolicyScopeInfo.getCode());
      salePolicyScopeOrgInfoVo.setOrgName(salePolicyScopeInfo.getName());
      salePolicyScopeOrgInfoVos.add(salePolicyScopeOrgInfoVo);
    }
    return salePolicyScopeOrgInfoVos;
  }

  /**
   * 筛选可以查看的组织机构
   * @param salePolicyCustomerInfos
   * @return
   */
  private Set<String> findCustomerCodes(Set<SalePolicyScopeOrgInfoVo> salePolicyCustomerInfos) {
    //选定方式分组转成组织机构范围的组织机构编码map
    Map<String, List<String>> selectionMethodMap = salePolicyCustomerInfos.stream()
        .filter(a -> StringUtils.isNoneBlank(a.getSelectionMethod(), a.getOrgCode()))
        .collect(Collectors.groupingBy(SalePolicyScopeOrgInfoVo::getSelectionMethod, Collectors.mapping(SalePolicyScopeOrgInfoVo::getOrgCode, Collectors.toList())));

    //获取包含组织机构范围的客户编码
    List<String> includeList = selectionMethodMap.get(ScopeSelectionMethod.INCLUDE.getCode());
    if (CollectionUtils.isEmpty(includeList)) {
      return null;
    }
    List<CustomerVo> includeCustomerCodeList = this.customerVoService.findByOrgCodes(includeList);
    if (CollectionUtils.isEmpty(includeCustomerCodeList)) {
      return null;
    }
    Set<String> includeCustomerCodes = includeCustomerCodeList.stream().map(CustomerVo::getCustomerCode).collect(Collectors.toSet());
    //筛选掉非包含的数据
    List<String> excludeList = selectionMethodMap.get(ScopeSelectionMethod.EXCLUDE.getCode());
    if (!CollectionUtils.isEmpty(excludeList)) { 
      List<CustomerVo> excludeCustomerCodeList = this.customerVoService.findByOrgCodes(excludeList);
      if (!CollectionUtils.isEmpty(excludeCustomerCodeList)) { 
        Set<String> excludeCustomerOrgCodes = excludeCustomerCodeList.stream().map(CustomerVo::getCustomerCode).collect(Collectors.toSet());
        includeCustomerCodes.removeAll(excludeCustomerOrgCodes);
      } 
    } 
    return includeCustomerCodes;
  }
  
  /**
   * 按照组织机构来进行优惠政策范围圈定的客户编码集合
   * @param salePolicyCode 需要查询的优惠政策业务编号
   * @param tenantCode 当前二级租户信息
   * @return
   */
  @Override
  public boolean matched(String customerCode, String tenantCode, String salePolicyCode, Set<SalePolicyScopeOrgInfoVo> salePolicyCustomerInfos) {
    if (StringUtils.isAnyBlank(tenantCode, salePolicyCode) || CollectionUtils.isEmpty(salePolicyCustomerInfos)) {
      return false;
    }
    //选定方式分组转成组织机构范围的组织机构编码map
    Map<String, List<String>> selectionMethodMap = salePolicyCustomerInfos.stream()
        .filter(a -> StringUtils.isNoneBlank(a.getSelectionMethod(), a.getOrgCode()))
        .collect(Collectors.groupingBy(SalePolicyScopeOrgInfoVo::getSelectionMethod, Collectors.mapping(SalePolicyScopeOrgInfoVo::getOrgCode, Collectors.toList())));
    //获取包含组织机构范围的客户编码
    List<String> includeList = selectionMethodMap.get(ScopeSelectionMethod.INCLUDE.getCode());
    if (CollectionUtils.isEmpty(includeList)) {
      return false;
    }
    //筛选掉非包含的数据
    List<String> excludeList = selectionMethodMap.get(ScopeSelectionMethod.EXCLUDE.getCode());
    Boolean isExist = customerVoService.existByCustomer7OrgIn7OrgNotIn(customerCode, includeList, excludeList);
    return isExist;
  }
}
