package com.biz.crm.dms.business.policy.local.service.internal;

import com.biz.crm.dms.business.policy.local.entity.SalePolicyExecutorLadderVar;
import com.biz.crm.dms.business.policy.local.repository.SalePolicyExecutorLadderVarRepository;
import com.biz.crm.dms.business.policy.local.service.SalePolicyExecutorLadderVarService;
import com.biz.crm.dms.business.policy.local.utils.AnalyzeExpressionUtils;
import com.google.common.collect.Sets;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * 标品中默认的优惠政策执行信息阶梯变量部分的服务层实现（默认的）
 * @author songjingen
 */
@Service
public class SalePolicyExecutorLadderVarServiceImpl implements SalePolicyExecutorLadderVarService {

  @Autowired(required = false)
  private SalePolicyExecutorLadderVarRepository salePolicyExecutorLadderVarRepository;

  /**
   * 批量创建阶梯变量信息
   * @param salePolicyExecutorLadderVars 阶梯变量信息集合
   * @param salePolicyExecutorLadderId 阶梯id
   * @param expression 优惠执行策略表达式
   */
  @Override
  @Transactional
  public void createBatch(Set<SalePolicyExecutorLadderVar> salePolicyExecutorLadderVars, String salePolicyExecutorLadderId, String expression) {
    Validate.isTrue(!CollectionUtils.isEmpty(salePolicyExecutorLadderVars), "创建优惠政策执行信息变量时，创建集合不能为空！");
    Validate.notBlank(salePolicyExecutorLadderId, "创建优惠政策执行信息变量时, 阶梯id不能为空");
    Validate.notBlank(expression, "创建优惠政策执行信息变量时, 未获取到系统配置的优惠执行策略表达式！");
    validateVars(salePolicyExecutorLadderVars, salePolicyExecutorLadderId, expression);
    this.salePolicyExecutorLadderVarRepository.saveBatch(salePolicyExecutorLadderVars);
  }

  /**
   *  此私有方法是为了验证变量名的准确性，以及对变量对象中的executorLadderId参数赋值
   * @param salePolicyExecutorLadderVars
   * @param salePolicyExecutorLadderId
   * @param expression
   */
  private static void validateVars(Set<SalePolicyExecutorLadderVar> salePolicyExecutorLadderVars, String salePolicyExecutorLadderId, String expression) {
    // 对表达式中要求填写的变量和实际填写的变量进行对比
    Map<Integer, String> mustVarNameMapping = AnalyzeExpressionUtils.analyzeLadderExpressionMapping(expression);
    // 如果条件成立，说明这个限量政策不需要输入任何变量，也就不需要做任何验证了
    if(CollectionUtils.isEmpty(mustVarNameMapping)) {
      Validate.isTrue(CollectionUtils.isEmpty(salePolicyExecutorLadderVars) , "创建优惠政策执行信息变量时，发现优惠执行策略无需传递任何变量，请检查!");
      return;
    }
    Set<String> mustVarNames = Sets.newLinkedHashSet(mustVarNameMapping.values());
    Set<String> variableNames = salePolicyExecutorLadderVars.stream()
        .map(SalePolicyExecutorLadderVar::getVariableName)
        .filter(item -> StringUtils.isNotBlank(item))
        .collect(Collectors.toSet());
    Validate.isTrue(mustVarNameMapping.size() == variableNames.size() , "创建优惠政策执行信息变量时，传过来的变量名（%s）和系统表达式的变量名（%s）不匹配", variableNames, mustVarNames);
    Sets.SetView<String> differenceSet = Sets.difference(variableNames, mustVarNames);
    Validate.isTrue(differenceSet.isEmpty() , "创建优惠政策执行信息变量时，发现需要填写的变量信息(%s)，并没有填写，请检查" , StringUtils.join(mustVarNames.toArray(new String[] {}) , ","));
    // 不能有重复的变量信息
    String pattern = "^[A-Za-z]{1}[A-Za-z0-9]*$";
    long distinctCount = salePolicyExecutorLadderVars.stream().filter(item -> StringUtils.isNotBlank(item.getVariableName())).map(SalePolicyExecutorLadderVar::getVariableName).distinct().count();
    Validate.isTrue(distinctCount == salePolicyExecutorLadderVars.size() , "创建优惠政策执行信息时，发现至少一个阶梯中存在重复的变量名，请检查!!");
    // 一条一条验证，变量名的正则表达式(只能是字母或者数字，且首字母必须是字母)
    for (SalePolicyExecutorLadderVar salePolicyExecutorLadderVar : salePolicyExecutorLadderVars) {
      salePolicyExecutorLadderVar.setExecutorLadderId(salePolicyExecutorLadderId);
      salePolicyExecutorLadderVar.setId(null);
      String variableName = salePolicyExecutorLadderVar.getVariableName();
      Validate.notBlank(variableName , "创建优惠政策执行信息时，发现至少一个阶梯中存在错误的变量名信息，请检查!!");
      Validate.matchesPattern(variableName, pattern , "创建优惠政策执行信息时，变量名信息只能是字母或者数字，且首字母必须是字母(%s)，请检查!!" , variableName);
      Integer variableType = salePolicyExecutorLadderVar.getVariableType();
      Validate.inclusiveBetween(1, 4, variableType , "创建优惠政策执行信息时，变量类型只能是1：boolean、2：小数；3：整数；4：字符串，请检查!!");
      String variableValue = salePolicyExecutorLadderVar.getVariableValue();
      Validate.notNull(variableValue , "创建优惠政策执行信息时，变量值必须传入，请检查!!");
    }
  }
}
