package com.biz.crm.dms.business.policy.local.service.internal;

import com.biz.crm.dms.business.policy.local.entity.SalePolicy;
import com.biz.crm.dms.business.policy.local.entity.SalePolicyExecutor;
import com.biz.crm.dms.business.policy.local.entity.SalePolicyExecutorLadder;
import com.biz.crm.dms.business.policy.local.repository.SalePolicyExecutorRepository;
import com.biz.crm.dms.business.policy.local.repository.SalePolicyRepository;
import com.biz.crm.dms.business.policy.local.service.SalePolicyExecutorLadderService;
import com.biz.crm.dms.business.policy.local.service.SalePolicyExecutorService;
import com.biz.crm.dms.business.policy.sdk.register.SalePolicyTempleteRegister;
import com.biz.crm.dms.business.policy.sdk.strategy.SalePolicyCycleRuleStrategy;
import com.biz.crm.dms.business.policy.sdk.strategy.SalePolicyExecuteStrategy;
import com.biz.crm.dms.business.policy.sdk.vo.AbstractSalePolicyExecutorInfo;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.ApplicationContext;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import javax.transaction.Transactional;
import java.util.Collection;
import java.util.List;
import java.util.Set;

/**
 * 标品中默认的优惠政策执行信息部分的服务层实现（默认的）
 * @author yinwenjie
 */
@Service
public class SalePolicyExecutorServiceImpl implements SalePolicyExecutorService {

  @Autowired(required = false)
  private SalePolicyExecutorRepository salePolicyExecutorRepository;
  @Autowired(required = false)
  private SalePolicyExecutorLadderService salePolicyExecutorLadderService;
  @Autowired(required = false)
  private SalePolicyRepository salePolicyRepository;
  /**
   * 目前系统所支持的所有优惠政策情况
   */
  @Autowired(required = false)
  private List<SalePolicyTempleteRegister> salePolicyTempleteRegisters;
  @Autowired(required = false)
  private ApplicationContext applicationContext;
  
  @Override
  @Transactional
  public void create(SalePolicyExecutor salePolicyExecutor) {
    Validate.notNull(salePolicyExecutor , "创建优惠政策执行信息时，执行信息必须传入");
    String salePolicyCode = salePolicyExecutor.getSalePolicyCode();
    Validate.notBlank(salePolicyCode , "创建优惠政策执行信息时，需要指定优惠政策业务编号");
    String tenantCode = TenantUtils.getTenantCode();
    SalePolicy currentSalePolicy = salePolicyRepository.findBySalePolicyCodeAndTenantCode(salePolicyCode, tenantCode);
    Validate.notNull(currentSalePolicy , "创建优惠政策执行信息时，未找到指定的优惠政策基本信息(%s)，请检查!!" , salePolicyCode);
    String type = currentSalePolicy.getType();
    salePolicyExecutor.setId(null);
    
    // 1、===== 对将要添加的信息进行验证
    if(StringUtils.isBlank(salePolicyExecutor.getTenantCode())) {
      salePolicyExecutor.setTenantCode(tenantCode);
    }
    // 验证优惠政策执行策略业务编码
    String executorCode = salePolicyExecutor.getExecutorCode();
    Validate.notBlank(executorCode , "创建优惠政策执行信息时，必须传入执行策略业务编码!!");
    SalePolicyExecutor exsitSalePolicyExecutor = this.salePolicyExecutorRepository.findBySalePolicyCodeAndExecutorCode(salePolicyCode, executorCode, tenantCode);
    Validate.isTrue(exsitSalePolicyExecutor == null , "创建优惠政策执行信息时，发现该指定策略业务编码已被使用，请检查并修改!!");
    // 验证执行器策略是否注册
    String executeStrategyCode = salePolicyExecutor.getExecuteStrategyCode();
    Validate.notBlank(executeStrategyCode , "创建优惠政策执行信息时，必须指定执行策略业务编号，这个编号来自于某个具体的执行策略（SalePolicyExecuteStrategy），请检查!!");
    SalePolicyTempleteRegister salePolicyTempleteRegister = this.findSalePolicyTempleteRegister(type);
    Validate.notNull(salePolicyTempleteRegister , "创建优惠政策执行信息时，没有找到匹配的优惠政策模板注册信息，请检查!!");
    Collection<Class<? extends SalePolicyExecuteStrategy<? extends AbstractSalePolicyExecutorInfo>>> executeStrategyClasses = salePolicyTempleteRegister.getExecuteStrategyClasses();
    Validate.isTrue(!CollectionUtils.isEmpty(executeStrategyClasses) , "当前优惠政策注册器（%s），没有设定任何执行策略（SalePolicyExecuteStrategy），请检查!!" , type);
    boolean matchedExecuteStrategyCode = false;
    String expression = "";
    SalePolicyExecuteStrategy<? extends AbstractSalePolicyExecutorInfo> salePolicyExecuteStrategy = null;
    for (Class<? extends SalePolicyExecuteStrategy<? extends AbstractSalePolicyExecutorInfo>> executeStrategyClass : executeStrategyClasses) {
      try {
        salePolicyExecuteStrategy = applicationContext.getBean(executeStrategyClass);
        String targetExecuteStrategyCode = salePolicyExecuteStrategy.getExecuteStrategyCode();
        if(StringUtils.equals(executeStrategyCode, targetExecuteStrategyCode)) {
          matchedExecuteStrategyCode = true;
          expression = salePolicyExecuteStrategy.getExpression();
          break;
        }
      } catch(RuntimeException e) {
        throw new IllegalArgumentException(e.getMessage());
      }
    }
    Validate.isTrue(matchedExecuteStrategyCode , "创建优惠政策执行信息时，在匹配的注册器中，并没有找到指定的优惠政策执行策略（%s），请检查!!" , executeStrategyCode);
    // 处理执行器中的“阶梯循环规则”信息
    String cycleRuleCode = salePolicyExecutor.getCycleRuleCode();
    Validate.notNull(cycleRuleCode , "创建优惠政策执行信息时，必须传入阶梯规则信息（cycleRuleCode），请检查!!");
    Collection<Class<? extends SalePolicyCycleRuleStrategy>> cycleRuleStrategyClasses = salePolicyExecuteStrategy.getCycleRuleStrategyClasses();
    Validate.isTrue(!CollectionUtils.isEmpty(cycleRuleStrategyClasses) , "发现类型(%s)为的优惠政策模板注册，未设定任何一个“阶梯循环规则”（SalePolicyCycleRuleStrategy），请检查!!" , type);
    boolean matchedSalePolicyCycleRuleStrategy = false;
    for (Class<? extends SalePolicyCycleRuleStrategy> cycleRuleStrategyClass : cycleRuleStrategyClasses) {
      SalePolicyCycleRuleStrategy salePolicyCycleRuleStrategy = this.applicationContext.getBean(cycleRuleStrategyClass);
      if(StringUtils.equals(salePolicyCycleRuleStrategy.getCycleRuleCode(), cycleRuleCode)) {
        matchedSalePolicyCycleRuleStrategy = true;
      }
    }
    Validate.isTrue(matchedSalePolicyCycleRuleStrategy , "未发现指定的“阶梯循环规则”（%s） , 请检查!!" , cycleRuleCode);
    
    // 2、===== 进行执行策略基本信息的添加
    this.salePolicyExecutorRepository.save(salePolicyExecutor);
    
    // 3、===== 对阶梯详情进行验证和保存
    Set<SalePolicyExecutorLadder> salePolicyExecutorLadders = salePolicyExecutor.getSalePolicyExecutorLadders();
    this.salePolicyExecutorLadderService.createBatch(salePolicyExecutorLadders, executeStrategyCode, salePolicyCode, salePolicyExecutor.getId(), expression);
  }
  
  /**
   * 该私有方法，根据type获得正确的SalePolicyRegister营销策略注册器
   * @param type
   * @return
   */
  private SalePolicyTempleteRegister findSalePolicyTempleteRegister(String type) {
    Validate.notBlank("错误的优惠政策类型，请检查传入值");
    Validate.isTrue(!CollectionUtils.isEmpty(salePolicyTempleteRegisters) , "错误的SalePolicyTempleteRegister信息，请检查!!");
    SalePolicyTempleteRegister matchedSalePolicyTempleteRegister = null;
    for (SalePolicyTempleteRegister item : salePolicyTempleteRegisters) {
      if(StringUtils.equals(type, item.getType())) {
        matchedSalePolicyTempleteRegister = item;
        break;
      }
    }
    Validate.notNull(matchedSalePolicyTempleteRegister , "未匹配任何SalePolicyTempleteRegister信息，请检查!!");
    return matchedSalePolicyTempleteRegister;
  }

  @Override
  public List<SalePolicyExecutor> findBySalePolicyCodeAndTenantCode(String salePolicyCode, String tenantCode , String executeStrategyCode) {
    if(StringUtils.isAnyBlank(salePolicyCode , tenantCode , executeStrategyCode)) {
      return null;
    }
    
    return this.salePolicyExecutorRepository.findBySalePolicyCode(salePolicyCode, tenantCode , executeStrategyCode);
  }
}
