package com.biz.crm.dms.business.policy.local.service.internal;

import java.math.BigDecimal;
import java.util.Deque;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;
import java.util.stream.Collectors;

import javax.transaction.Transactional;

import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import com.biz.crm.dms.business.policy.local.context.DefaultPolicyExecuteContext;
import com.biz.crm.dms.business.policy.sdk.context.GiftResultInfo;
import com.biz.crm.dms.business.policy.sdk.context.PolicyStepResult;
import com.biz.crm.dms.business.policy.sdk.context.ProductPolicyStepResult;
import com.biz.crm.dms.business.policy.sdk.context.SalePolicyConProduct;
import com.biz.crm.dms.business.policy.sdk.context.StepType;
import com.biz.crm.dms.business.policy.local.entity.SalePolicyRecord;
import com.biz.crm.dms.business.policy.local.entity.SalePolicyRecordExcludePolicy;
import com.biz.crm.dms.business.policy.local.entity.SalePolicyRecordGift;
import com.biz.crm.dms.business.policy.local.entity.SalePolicyRecordProduct;
import com.biz.crm.dms.business.policy.local.entity.SalePolicyRecordProductSelectedPolicy;
import com.biz.crm.dms.business.policy.local.entity.SalePolicyRecordStepPolicy;
import com.biz.crm.dms.business.policy.local.repository.SalePolicyRecordExcludePolicyRepository;
import com.biz.crm.dms.business.policy.local.repository.SalePolicyRecordGiftRepository;
import com.biz.crm.dms.business.policy.local.repository.SalePolicyRecordProductRepository;
import com.biz.crm.dms.business.policy.local.repository.SalePolicyRecordProductSelectedPolicyRepository;
import com.biz.crm.dms.business.policy.local.repository.SalePolicyRecordRepository;
import com.biz.crm.dms.business.policy.local.repository.SalePolicyRecordStepPolicyRepository;
import com.biz.crm.dms.business.policy.local.service.SalePolicyRecordService;

import com.biz.crm.dms.business.policy.sdk.vo.SalePolicyVo;
import com.google.common.collect.Sets;

@Service
public class SalePolicyRecordServiceImpl implements SalePolicyRecordService {
  @Autowired(required = false)
  private SalePolicyRecordRepository salePolicyRecordRepository;
  @Autowired(required = false)
  private SalePolicyRecordGiftRepository salePolicyRecordGiftRepository;
  @Autowired(required = false)
  private SalePolicyRecordProductRepository salePolicyRecordProductRepository;
  @Autowired(required = false)
  private SalePolicyRecordStepPolicyRepository salePolicyRecordStepPolicyRepository;
  @Autowired(required = false)
  private SalePolicyRecordExcludePolicyRepository salePolicyRecordExcludePolicyRepository;
  @Autowired(required = false)
  private SalePolicyRecordProductSelectedPolicyRepository salePolicyRecordProductSelectedPolicyRepository;

  @Override
  public void create(String billCode , DefaultPolicyExecuteContext executeContext) {
    /*
     * 基于当前正在执行的优惠政策上下文，将优惠政策流水信息保存下来。
     * 发生这样业务场景的情况，一般都是标品中优惠政策的正式执行过程。
     * 
     * 处理过程如下：
     * 1、首先是保存本次优惠执行的基本信息，包括初始化了哪些优惠政策（多个“,”分割即可），初始化了哪些商品信息（多个“,”分割即可），
     * 租户信息如何，客户信息如何，等等
     * 2、特别注意，这里可能存在某个或者某些商品指定了只能执行的优惠政策信息，所以需要进行特别处理
     * 3、然后这个优惠政策执行过程中，那些没有被执行的优惠政策，以及没有被执行的原因，也要记录下来
     * 4、接着是优惠政策的执行过程，也要进行保存
     * 5、记录商品的明细流水，例如商品初始化信息、优惠前的信息、优惠后的信息
     * 包括商品的可能存在的赠品信息（累加）
     * */
    Validate.notBlank(billCode , "创建优惠流水明细时，发现未传入单据信息");
    String tenantCode = executeContext.getTenantCode();
    Validate.notBlank(tenantCode , "创建优惠流水明细时，未发现优惠政策执行的租户信息");
    String customerCode = executeContext.getCustomerCode();
    Validate.notBlank(customerCode , "创建优惠流水明细时，未发现优惠政策执行的客户信息");
    Set<SalePolicyConProduct> initPolicyProducts = executeContext.getInitPolicyProducts();
    Validate.isTrue(!CollectionUtils.isEmpty(initPolicyProducts) , "创建优惠流水明细时，未发现优惠执行上下文的原始商品信息，请检查!!");
    long blankCount = initPolicyProducts.stream().filter(item -> StringUtils.isBlank(item.getProductCode())).count();
    Validate.isTrue(blankCount == 0 , "创建优惠流水明细时，发现有商品编码错误的情况，请检查!!");
    Set<SalePolicyVo> initSalePolicys =  executeContext.getInitSalePolicys();
    Validate.isTrue(!CollectionUtils.isEmpty(initSalePolicys) , "创建优惠流水明细时，未发现优惠执行上下文的原始优惠政策信息，请检查!!");
    blankCount = initSalePolicys.stream().filter(item -> StringUtils.isBlank(item.getSalePolicyCode())).count();
    Validate.isTrue(blankCount == 0 , "创建优惠流水明细时，发现有优惠政策编码错误的情况，请检查!!");
    
    // 1、========
    SalePolicyRecord salePolicyRecord = new SalePolicyRecord();
    salePolicyRecord.setBillCode(billCode);
    salePolicyRecord.setTenantCode(tenantCode);
    salePolicyRecord.setCustomerCode(customerCode);
    // 所有参与执行的商品信息、优惠政策信息，无论它们是否执行
    String productCodes = StringUtils.join(initPolicyProducts.stream().map(SalePolicyConProduct::getProductCode).iterator() , ",");
    salePolicyRecord.setProductCodes(productCodes);
    String salePolicyCodes = StringUtils.join(initSalePolicys.stream().map(SalePolicyVo::getSalePolicyCode).iterator() , ",");
    salePolicyRecord.setSalePolicyCodes(salePolicyCodes);
    // 初始化信息
    BigDecimal initTotalAmount = executeContext.getInitTotalAmount();
    salePolicyRecord.setInitTotalAmount(initTotalAmount);
    Integer initTotalNumber = executeContext.getInitTotalNumber();
    salePolicyRecord.setInitTotalNumber(initTotalNumber);
    boolean kneading = executeContext.isKneading();
    salePolicyRecord.setKneading(kneading);
    // 开始正式保存
    this.salePolicyRecordRepository.save(salePolicyRecord);
    
    // 2、======
    for (SalePolicyConProduct initPolicyProduct : initPolicyProducts) {
      String[] selectedPolicyCodes = initPolicyProduct.getSelectedPolicyCodes();
      // 如果条件成立，说明这个商品指定了其只能参加的优惠政策信息
      if(selectedPolicyCodes != null && selectedPolicyCodes.length != 0) {
        SalePolicyRecordProductSelectedPolicy salePolicyRecordProductSelectedPolicy = new SalePolicyRecordProductSelectedPolicy();
        salePolicyRecordProductSelectedPolicy.setProductCode(initPolicyProduct.getProductCode());
        salePolicyRecordProductSelectedPolicy.setRecordId(salePolicyRecord.getId());
        salePolicyRecordProductSelectedPolicy.setSelectedPolicyCodes(StringUtils.join(selectedPolicyCodes , ","));
        this.salePolicyRecordProductSelectedPolicyRepository.save(salePolicyRecordProductSelectedPolicy);
      }
    }
    
    // 3、=====
    Map<String, String> excludedPolicyMapping = executeContext.getExcludedPolicyMapping();
    if(!CollectionUtils.isEmpty(excludedPolicyMapping)) {
      for (Entry<String,String> excludedPolicyItem : excludedPolicyMapping.entrySet()) {
        String salePolicyCode = excludedPolicyItem.getKey();
        String excludedMsg = excludedPolicyItem.getValue();
        SalePolicyRecordExcludePolicy salePolicyRecordExcludePolicy = new SalePolicyRecordExcludePolicy();
        salePolicyRecordExcludePolicy.setRecordId(salePolicyRecord.getId());
        salePolicyRecordExcludePolicy.setExcludedMsg(excludedMsg);
        salePolicyRecordExcludePolicy.setPolicyCode(salePolicyCode);
        this.salePolicyRecordExcludePolicyRepository.save(salePolicyRecordExcludePolicy);
      }
    }
    
    // 4、======
    // 优惠政策的执行顺序(注意，INIT性质的步进也需要保存，不过其没有salePolicyCode和executorCode)
    Deque<PolicyStepResult> policyStepResults = executeContext.getPolicyStepResults();
    Iterator<PolicyStepResult> policyStepResultIterator = policyStepResults.descendingIterator();
    int sortedIndex = 0;
    while (policyStepResultIterator.hasNext()) {
      PolicyStepResult policyStepResultItem = policyStepResultIterator.next();
      SalePolicyRecordStepPolicy salePolicyRecordStepPolicy = new SalePolicyRecordStepPolicy();
      salePolicyRecordStepPolicy.setRecordId(salePolicyRecord.getId());
      salePolicyRecordStepPolicy.setSalePolicyCode(StringUtils.isBlank(policyStepResultItem.getSalePolicyCode())?"":policyStepResultItem.getSalePolicyCode());
      salePolicyRecordStepPolicy.setExecuteCode(StringUtils.isBlank(policyStepResultItem.getExecutorCode())?"":policyStepResultItem.getExecutorCode());
      salePolicyRecordStepPolicy.setSorted(sortedIndex++);
      salePolicyRecordStepPolicy.setStepType(policyStepResultItem.getStepType().toString());
      this.salePolicyRecordStepPolicyRepository.save(salePolicyRecordStepPolicy);
    }
    
    // 5、======
    Map<String, Deque<ProductPolicyStepResult>> productPolicyStepResults = executeContext.getProductPolicyStepResultMapping();
    Map<String, SalePolicyConProduct> initPolicyProductMappings = initPolicyProducts.stream().collect(Collectors.toMap(SalePolicyConProduct::getProductCode, item -> item));
    // 如果条件成立，说明没有优惠政策的执行明细，逻辑过程到此为止
    if(CollectionUtils.isEmpty(productPolicyStepResults)) {
      return;
    }
    Set<Entry<String, Deque<ProductPolicyStepResult>>> productPolicyStepResultItems = productPolicyStepResults.entrySet();
    for (Entry<String, Deque<ProductPolicyStepResult>> productPolicyStepResultItem : productPolicyStepResultItems) {
      String productCode = productPolicyStepResultItem.getKey();
      SalePolicyConProduct defaultPolicyConProduct = initPolicyProductMappings.get(productCode);
      Deque<ProductPolicyStepResult> productPolicyStepDeque = productPolicyStepResultItem.getValue();
      // 一个一个的流水都要进行添加
      Iterator<ProductPolicyStepResult> descendingIterator = productPolicyStepDeque.descendingIterator();
      for(int sorted = 0 ; sorted < productPolicyStepDeque.size() ; sorted++) {
        ProductPolicyStepResult productPolicyStepResult = descendingIterator.next();
        StepType stepType = productPolicyStepResult.getStepType();
        String salePolicyCode = productPolicyStepResult.getSalePolicyCode();
        SalePolicyRecordProduct salePolicyRecordProduct = new SalePolicyRecordProduct();
        
        // 以下是流水的基本信息
        salePolicyRecordProduct.setRecordId(salePolicyRecord.getId());
        salePolicyRecordProduct.setBillItemCode(defaultPolicyConProduct.getBillItemCode());
        salePolicyRecordProduct.setSalePolicyCode(StringUtils.isBlank(salePolicyCode)?"":salePolicyCode);
        salePolicyRecordProduct.setExecuteCode(StringUtils.isBlank(productPolicyStepResult.getExecuteCode())?"":productPolicyStepResult.getExecuteCode());
        salePolicyRecordProduct.setProductCode(defaultPolicyConProduct.getProductCode());
        salePolicyRecordProduct.setProductName(defaultPolicyConProduct.getProductName());
        salePolicyRecordProduct.setSorted(sorted);
        salePolicyRecordProduct.setStepType(stepType.toString());
        // TODO 没有验证
        // 以下是初始化信息，一边验证一边保存
        BigDecimal initPrices = productPolicyStepResult.getInitPrices();
        Integer initNumbers = productPolicyStepResult.getInitNumbers();
        BigDecimal initSubtotal = productPolicyStepResult.getInitSubtotal();
        salePolicyRecordProduct.setInitPrices(initPrices);
        salePolicyRecordProduct.setInitNumbers(initNumbers);
        salePolicyRecordProduct.setInitSubtotal(initSubtotal);
        // TODO 没有验证
        // 以下是执行后的信息，一边验证一边保存
        BigDecimal lastPrices = productPolicyStepResult.getLastPrices();
        BigDecimal lastSubtotal = productPolicyStepResult.getLastSubtotal();
        BigDecimal lastSurplusTotalAmount = productPolicyStepResult.getLastSurplusTotalAmount();
        Integer lastSurplusTotalNumber =  productPolicyStepResult.getLastSurplusTotalNumber();
        List<GiftResultInfo> lastGiftResultInfos = productPolicyStepResult.getLastGifts();
        salePolicyRecordProduct.setLastPrices(lastPrices);
        salePolicyRecordProduct.setLastSubtotal(lastSubtotal);
        salePolicyRecordProduct.setLastSurplusTotalAmount(lastSurplusTotalAmount);
        salePolicyRecordProduct.setLastSurplusTotalNumber(lastSurplusTotalNumber);
        // 本次商品优惠步进中，相对于上一个步进该本品的小计金额减少小计价格
        BigDecimal preSubtotal = productPolicyStepResult.getPreSubtotal();
        BigDecimal diffLastSubtotal = preSubtotal.subtract(lastSubtotal);
        salePolicyRecordProduct.setDiffLastSubtotal(diffLastSubtotal);
        // 本次商品优惠步进中，相对于上一个步进，该本品有多少数量参与了优惠
        Integer preSurplusTotalNumber = productPolicyStepResult.getPreSurplusTotalNumber();
        salePolicyRecordProduct.setDiffSurplusTotalNumber(preSurplusTotalNumber - lastSurplusTotalNumber);
        // 本次商品优惠步进中，相对于上一个步进，该本品有多少小计金额参与了优惠
        BigDecimal preSurplusTotalAmount = productPolicyStepResult.getPreSurplusTotalAmount();
        salePolicyRecordProduct.setDiffSurplusTotalAmount(preSurplusTotalAmount.subtract(lastSurplusTotalAmount));
        // 本次商品优惠步进中，相对于上一个步进，该本品享受了多少赠品数量的优惠
        Integer preGiftEnjoyedTotalNumber = productPolicyStepResult.getPreGiftEnjoyedTotalNumber() == null?0:productPolicyStepResult.getPreGiftEnjoyedTotalNumber();
        Integer lastGiftEnjoyedTotalNumber = productPolicyStepResult.getLastGiftEnjoyedTotalNumber() == null?0:productPolicyStepResult.getLastGiftEnjoyedTotalNumber();
        Integer diffGiftEnjoyedTotalNumber = lastGiftEnjoyedTotalNumber - preGiftEnjoyedTotalNumber;
        salePolicyRecordProduct.setLastGiftEnjoyedTotalNumber(lastGiftEnjoyedTotalNumber);
        salePolicyRecordProduct.setDiffGiftEnjoyedTotalNumber(diffGiftEnjoyedTotalNumber);
        // 本次商品优惠步进中，相对于上一个步进，该本品享受了多少赠品金额的优惠
        BigDecimal preGiftsEnjoyedTotalAmount = productPolicyStepResult.getPreGiftEnjoyedTotalAmount() == null?BigDecimal.ZERO:productPolicyStepResult.getPreGiftEnjoyedTotalAmount();
        BigDecimal lastGiftsEnjoyedTotalAmount = productPolicyStepResult.getLastGiftEnjoyedTotalAmount() == null?BigDecimal.ZERO:productPolicyStepResult.getLastGiftEnjoyedTotalAmount();
        BigDecimal diffGiftEnjoyedTotalAmount = lastGiftsEnjoyedTotalAmount.subtract(preGiftsEnjoyedTotalAmount);
        salePolicyRecordProduct.setLastGiftEnjoyedTotalAmount(lastGiftsEnjoyedTotalAmount);
        salePolicyRecordProduct.setDiffGiftEnjoyedTotalAmount(diffGiftEnjoyedTotalAmount);
        this.salePolicyRecordProductRepository.save(salePolicyRecordProduct);
        // 可能的赠品信息 
        if(CollectionUtils.isEmpty(lastGiftResultInfos)) {
          continue;
        }
        // 每个赠品的基本信息都要一边验证以便保存
        for (GiftResultInfo giftResultInfo : lastGiftResultInfos) {
          String giftProductCode = giftResultInfo.getProductCode();
          String giftProductName = giftResultInfo.getProductName();
          Integer giftQuantity = giftResultInfo.getQuantity();
          BigDecimal giftSubtotalAmount = giftResultInfo.getSubtotalAmount();
          // TODO 还没有验证
          SalePolicyRecordGift salePolicyRecordGift = new SalePolicyRecordGift();
          salePolicyRecordGift.setRecordProductId(salePolicyRecordProduct.getId());
          salePolicyRecordGift.setGiftCode(giftProductCode);
          salePolicyRecordGift.setGiftName(giftProductName);
          salePolicyRecordGift.setQuantity(giftQuantity);
          salePolicyRecordGift.setSubtotalAmount(giftSubtotalAmount);
          this.salePolicyRecordGiftRepository.save(salePolicyRecordGift);
        }
      }
    }
  }

  @Override
  @Transactional
  public void deleteByBillCode(String billCode, String tenantCode, String customerCode) {
    Validate.notBlank(billCode , "删除/回退优惠政策流水时，发现错误的单据编号!!");
    Validate.notBlank(tenantCode , "删除/回退优惠政策流水时，发现错误的租户信息!!");
    Validate.notBlank(customerCode , "删除/回退优惠政策流水时，发现错误的客户编号信息!!");
    
    /*
     * 删除过程为：
     * 1、首先确定当前流水记录的存在性（如果不存在则不需要报错，因为一个业务单据没有任何的优惠政策历史日志，也是正常现象）
     * 2、接着根据流水记录的id，删除这个流水记录下的商品步进信息salePolicyRecordProductRepository
     * 注意：同时删除可能的赠品流水信息salePolicyRecordGiftRepository
     * 3、删除优惠步进信息salePolicyRecordStepPolicyRepository
     * 4、删除优惠排除信息salePolicyRecordExcludePolicyRepository
     * 以及本品选定的优惠政策信息salePolicyRecordProductSelectedPolicyRepository
     * 5、最后删除优惠政策日志信息本身
     * */
    // 1、======
    SalePolicyRecord salePolicyRecord = this.findByBillCode(billCode, tenantCode, customerCode);
    if(salePolicyRecord == null) {
      return;
    }
    String salePolicyRecordId = salePolicyRecord.getId();
    
    // 2、=====
    List<SalePolicyRecordProduct> salePolicyRecordProducts = this.salePolicyRecordProductRepository.findByRecordId(salePolicyRecordId);
    if(!CollectionUtils.isEmpty(salePolicyRecordProducts)) {
      for (SalePolicyRecordProduct salePolicyRecordProduct : salePolicyRecordProducts) {
        this.salePolicyRecordGiftRepository.deleteByRecordProductId(salePolicyRecordProduct.getId());
      }
      this.salePolicyRecordProductRepository.deleteByRecordId(salePolicyRecordId);
    }
    
    // 3、=====
    this.salePolicyRecordStepPolicyRepository.deleteByRecordId(salePolicyRecordId);
    
    // 4、=====
    this.salePolicyRecordExcludePolicyRepository.deleteByRecordId(salePolicyRecordId);
    this.salePolicyRecordProductSelectedPolicyRepository.deleteByIds(Sets.newHashSet(salePolicyRecordId));
    
    // 5、=====
    this.salePolicyRecordRepository.deleteByIds(Sets.newHashSet(salePolicyRecordId));
  }

  @Override
  public SalePolicyRecord findByBillCode(String billCode, String tenantCode, String customerCode) {
    if(StringUtils.isAnyBlank(billCode , tenantCode, customerCode)) {
      return null;
    }
    return this.salePolicyRecordRepository.findByBillCodeAndTenantCodeAndTenantCode(billCode, customerCode, tenantCode);
  }

  @Override
  public Set<SalePolicyRecord> findByTenantCodeAndSalePolicyCode(String tenantCode, String salePolicyCode) {
    if(StringUtils.isAnyBlank(tenantCode, salePolicyCode)) {
      return null;
    }
    return this.salePolicyRecordRepository.findByTenantCodeAndSalePolicyCode(tenantCode, salePolicyCode);
  }

  @Override
  public Set<SalePolicyRecord> findByTenantCodeAndSalePolicyCodeAndCustomerCode(String tenantCode, String salePolicyCode, String customerCode) {
    if(StringUtils.isAnyBlank(tenantCode, customerCode, salePolicyCode)) {
      return null;
    }
    return this.salePolicyRecordRepository.findByTenantCodeAndSalePolicyCodeAndCustomerCode(tenantCode, salePolicyCode, customerCode);
  }

  @Override
  public Set<SalePolicyRecord> findByTenantCodeAndSalePolicyCodeAndCustomerCodeAndBillCode(String tenantCode, String salePolicyCode, String customerCode, String billCode) {
    if(StringUtils.isAnyBlank(tenantCode, customerCode, billCode, salePolicyCode)) {
      return null;
    }
    return this.salePolicyRecordRepository.findByTenantCodeAndSalePolicyCodeAndCustomerCodeAndBillCode(tenantCode, salePolicyCode, customerCode, billCode);
  }
}
