package com.biz.crm.dms.business.policy.local.service.internal;

import com.biz.crm.dms.business.policy.local.entity.SalePolicy;
import com.biz.crm.dms.business.policy.local.entity.SalePolicyScopeInfo;
import com.biz.crm.dms.business.policy.local.repository.SalePolicyRepository;
import com.biz.crm.dms.business.policy.local.repository.SalePolicyScopeInfoRepository;
import com.biz.crm.dms.business.policy.local.service.SalePolicyScopeInfoService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import javax.transaction.Transactional;
import java.util.List;
import java.util.Set;

/**
 * 在DMS标品模块中，负责进行客户范围选择信息维护的默认服务层实现
 * @author yinwenjie
 */
@Service
public class SalePolicyScopeInfoServiceImpl implements SalePolicyScopeInfoService {
  @Autowired(required = false)
  private SalePolicyRepository salePolicyRepository;
  @Autowired(required = false)
  private SalePolicyScopeInfoRepository salePolicyScopeInfoRepository;
  
  @Override
  @Transactional
  public void createBatch(Set<SalePolicyScopeInfo> salePolicyScopeInfos) {
    Validate.isTrue(!CollectionUtils.isEmpty(salePolicyScopeInfos) , "创建优惠政策适用范围信息时，必须至少传入一条适用范围信息，请检查！！");
    long codeCount = salePolicyScopeInfos.stream().filter(item -> StringUtils.isNotBlank(item.getCode())).map(SalePolicyScopeInfo::getCode).distinct().count();
    Validate.isTrue(codeCount == salePolicyScopeInfos.size() , "创建优惠政策适用范围信息时，至少有两组范围信息的业务编号重复或者没有填写，请检查!!");
    // 加入的范围信息，其中的code不能重复
    for (SalePolicyScopeInfo salePolicyScopeInfo : salePolicyScopeInfos) {
      this.createHandle(salePolicyScopeInfo);
    }
  }

  /**
   * 根据租户编码、优惠政策业务编号、客户范围控制策略的识别类型号查询数据集合
   * @param tenantCode 租户编码
   * @param salePolicyCode 优惠政策业务编号
   * @param customerScopeType 客户范围控制策略的识别类型号
   * @return
   */
  @Override
  public List<SalePolicyScopeInfo> findByTenantCodeAndSalePolicyCodeAndCustomerScopeType(String tenantCode, String salePolicyCode, String customerScopeType) {
    if (StringUtils.isAnyBlank(tenantCode, salePolicyCode, customerScopeType)) {
      return null;
    }
    return this.salePolicyScopeInfoRepository.findByTenantCodeAndSalePolicyCodeAndCustomerScopeType(tenantCode, salePolicyCode, customerScopeType);
  }

  /**
   * 该私有方法负责处理单条客户范围信息
   * @param salePolicyScopeInfo
   */
  private void createHandle(SalePolicyScopeInfo salePolicyScopeInfo) {
    /*
     * 处理过程为：
     * 1、首先验证批量添加时，每一条数据的正确性
     * 2、依次进行每一条信息的添加
     * */
    // 1、=======
    String tenantCode = TenantUtils.getTenantCode();
    salePolicyScopeInfo.setTenantCode(tenantCode);
    salePolicyScopeInfo.setId(null);
    String salePolicyCode = salePolicyScopeInfo.getSalePolicyCode();
    SalePolicy exsitSalePolicy = this.salePolicyRepository.findBySalePolicyCodeAndTenantCode(salePolicyCode, tenantCode);
    Validate.notNull(exsitSalePolicy , "创建优惠政策适用范围信息时，未发现指定的优惠政策信息，请检查!!");
    // 必须填写的信息
    String code = salePolicyScopeInfo.getCode();
    Validate.notBlank(code , "创建优惠政策适用范围信息时，业务编号信息必须填写!!");
    String name = salePolicyScopeInfo.getName();
    Validate.notBlank(name , "创建优惠政策适用范围信息时，业务名称信息必须填写!!");
    String customerScopeType = salePolicyScopeInfo.getCustomerScopeType();
    Validate.notBlank(customerScopeType , "创建优惠政策适用范围信息时，范围类型（customerScopeType）信息必须填写!!");
    String selectionMethod = salePolicyScopeInfo.getSelectionMethod();
    Validate.notBlank(selectionMethod , "创建优惠政策适用范围信息时，选定方式（selectionMethod）信息必须填写!!");
    
    // 2、=====
    this.salePolicyScopeInfoRepository.save(salePolicyScopeInfo);
  }
}
