package com.biz.crm.dms.business.policy.local.service.internal;

import com.biz.crm.dms.business.policy.local.entity.SalePolicy;
import com.biz.crm.dms.business.policy.local.entity.SalePolicyThreshold;
import com.biz.crm.dms.business.policy.local.entity.SalePolicyThresholdDetail;
import com.biz.crm.dms.business.policy.local.entity.SalePolicyThresholdProduct;
import com.biz.crm.dms.business.policy.local.repository.SalePolicyRepository;
import com.biz.crm.dms.business.policy.local.repository.SalePolicyThresholdDetailRepository;
import com.biz.crm.dms.business.policy.local.repository.SalePolicyThresholdProductRepository;
import com.biz.crm.dms.business.policy.local.repository.SalePolicyThresholdRepository;
import com.biz.crm.dms.business.policy.local.service.SalePolicyThresholdService;
import com.biz.crm.dms.business.policy.local.vo.SalePolicyThresholdDetailVo;
import com.biz.crm.dms.business.policy.local.vo.SalePolicyThresholdProductVo;
import com.biz.crm.dms.business.policy.local.vo.SalePolicyThresholdVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;

import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import javax.transaction.Transactional;
import java.util.ArrayList;
import java.util.LinkedHashSet;
import java.util.List;

@Service
public class SalePolicyThresholdServiceImpl implements SalePolicyThresholdService {

  @Autowired(required = false)
  private SalePolicyRepository salePolicyRepository; 
  @Autowired(required = false)
  private SalePolicyThresholdRepository salePolicyThresholdRepository;
  @Autowired(required = false)
  private SalePolicyThresholdDetailRepository salePolicyThresholdDetailRepository;
  @Autowired(required = false)
  private SalePolicyThresholdProductRepository salePolicyThresholdProductRepository;
  @Autowired(required = false)
  private NebulaToolkitService nebulaToolkitService;
  
  @Override
  @Transactional
  public SalePolicyThreshold create(SalePolicyThresholdVo salePolicyThreshold) {
    /*
     * 添加过程为：
     * 1、进行门槛基本信息的验证和保存
     * 2、进行每一条门槛明细信息的验证和保存
     * 3、进行每一条门槛明细中商品明细信息的验证和保存
     * */
    // 1、======
    Validate.notNull(salePolicyThreshold , "创建商品门槛时，门槛基本信息需要传入!!");
    String salePolicyCode = salePolicyThreshold.getSalePolicyCode();
    Validate.notBlank(salePolicyCode , "创建商品门槛时，需要指定优惠政策业务编号");
    String tenantCode = TenantUtils.getTenantCode();
    SalePolicy exsitSalePolicy = salePolicyRepository.findBySalePolicyCodeAndTenantCode(salePolicyCode, tenantCode);
    Validate.notNull(exsitSalePolicy , "创建商品门槛时，未找到指定的优惠政策基本信息(%s)，请检查!!" , salePolicyCode);
    if(StringUtils.isBlank(salePolicyThreshold.getTenantCode())) {
      salePolicyThreshold.setTenantCode(tenantCode);
    }
    SalePolicyThreshold existThreshold = this.salePolicyThresholdRepository.findBySalePolicyCode(salePolicyCode , tenantCode);
    Validate.isTrue(existThreshold == null , "创建优惠政策门槛时，发现指定的的优惠政策已经设定了门槛信息，请检查!!");
    // 总最低限量
    Integer composeUnit = salePolicyThreshold.getComposeUnit();
    Validate.inclusiveBetween(1 , 2 , composeUnit , "创建优惠政策门槛时，总最低限量单位未正确传入（1：总金额；2：总数量）");
    Integer composeNumber = salePolicyThreshold.getComposeNumber();
    Validate.isTrue(composeNumber != null && composeNumber >= 0 , "创建优惠政策门槛时，总最低限量数量未传入（必须大于等于0，如果没有限量，则传入0）");
    SalePolicyThreshold salePolicyThresholdEntity = this.nebulaToolkitService.copyObjectByWhiteList(salePolicyThreshold, SalePolicyThreshold.class, LinkedHashSet.class ,ArrayList.class);
    this.salePolicyThresholdRepository.save(salePolicyThresholdEntity);
    salePolicyThreshold.setId(salePolicyThresholdEntity.getId());
    // 如果是整单优惠政策，就不再执行后续的门槛明细信息保存了
    if(exsitSalePolicy.getWholePolicy()) {
      return salePolicyThresholdEntity;
    }
    
    // 2、===== 
    List<SalePolicyThresholdDetailVo> salePolicyThresholdDetails = salePolicyThreshold.getThresholdDetails();
    // 接着才是一条一条的明细记录依次进行验证
    for (SalePolicyThresholdDetailVo salePolicyThresholdDetail : salePolicyThresholdDetails) {
      salePolicyThresholdDetail.setSalePolicyCode(salePolicyCode);
      salePolicyThresholdDetail.setTenantCode(tenantCode);
      String thresholdDetailCode = salePolicyThresholdDetail.getThresholdDetailCode();
      SalePolicyThresholdDetail exsitSalePolicyThresholdDetail = this.salePolicyThresholdDetailRepository.findBySalePolicyCodeAndThresholdDetailCode(salePolicyCode, tenantCode, thresholdDetailCode);
      Validate.isTrue(exsitSalePolicyThresholdDetail == null , "创建优惠政策门槛明细时，发现指定的优惠政策门槛code（%s）已经被使用，请检查!!" , thresholdDetailCode);
      Validate.notBlank(thresholdDetailCode , "创建优惠政策门槛明细时，发现至少一条优惠政策门槛业务编号没有填写，请检查!!");
      Integer includeType = salePolicyThresholdDetail.getIncludeType();
      Validate.notNull(includeType , "创建优惠政策门槛明细时（%s），全局限量类型（includeType）必须填写，请检查!!" , thresholdDetailCode);
      Validate.inclusiveBetween(1, 2, includeType , "创建优惠政策门槛明细时，发现不支持的门槛类型值（includeType）,请检查!!");
      // 如果条件成立，那么需要验证includeNumber数量必须大于或者等于当前明细门槛下的勾选了certain == true的明细商品门槛数量
      List<SalePolicyThresholdProductVo> thresholdProducts = salePolicyThresholdDetail.getThresholdProducts();
      if(includeType == 2) {
        Integer includeNumber = salePolicyThresholdDetail.getIncludeNumber();
        Validate.isTrue(includeNumber != null && includeNumber >= 0 , "维护优惠政策门槛详情信息时（%s），发现未正确填写的“必含数量”，请检查!" , thresholdDetailCode);
        Validate.isTrue(!CollectionUtils.isEmpty(thresholdProducts) , "维护优惠政策门槛详情信息时（%s），未发现商品门槛详情信息，请检查!" , thresholdDetailCode);
        Validate.isTrue(thresholdProducts.size() >= includeNumber , "维护优惠政策门槛详情信息时（%s），发现必选商品数量大于门槛中选定的商品数量，请检查!!" , thresholdDetailCode);
        int certainCount = (int)thresholdProducts.stream().filter(item -> item.getCertain() != null && item.getCertain()).count();
        Validate.isTrue(thresholdProducts.size() >= certainCount , "维护优惠政策门槛详情信息时（%s），发现设定的商品数量已经小于设定为“必选”的商品数量，请检查!!" , thresholdDetailCode);
        Validate.isTrue(includeNumber >= certainCount , "维护优惠政策门槛详情信息时（%s），发现设定的商品门槛数量已经小于设定为“必选”的商品数量，请检查!!" , thresholdDetailCode);
      }
      SalePolicyThresholdDetail salePolicyThresholdDetailEntity = this.nebulaToolkitService.copyObjectByWhiteList(salePolicyThresholdDetail, SalePolicyThresholdDetail.class, LinkedHashSet.class, ArrayList.class);
      salePolicyThresholdDetailEntity.setSalePolicyThresholdId(salePolicyThreshold.getId());
      this.salePolicyThresholdDetailRepository.save(salePolicyThresholdDetailEntity);
      salePolicyThresholdDetail.setId(salePolicyThresholdDetailEntity.getId());
      
      // 3、===== 对每一条门槛明细的商品明细进行保存
      if(CollectionUtils.isEmpty(thresholdProducts)) {
        continue;
      }
      for (SalePolicyThresholdProductVo thresholdProductItem : thresholdProducts) {
        thresholdProductItem.setTenantCode(tenantCode);
        thresholdProductItem.setSalePolicyCode(salePolicyCode);
        thresholdProductItem.setThresholdDetailCode(thresholdDetailCode);
        String productCode = thresholdProductItem.getProductCode();
        Validate.notBlank(productCode , "创建优惠政策明细门槛商品时，商品限量信息的业务编号没有传入，请检查!!");
        Integer thresholdUnit = thresholdProductItem.getThresholdUnit();
        Validate.inclusiveBetween(1 , 2 , thresholdUnit , "创建优惠政策明细门槛商品时，业务编号为[%s]的商品限量单位未正确传入（1：总金额；2：总数量）" , productCode);
        Integer thresholdNumber = thresholdProductItem.getThresholdNumber();
        Validate.isTrue(thresholdNumber != null && thresholdNumber >= 0 , "创建优惠政策明细门槛商品时，业务编号为[%s]的商品限量数量未传入（必须大于等于0，如果没有限量，则传入0）" , productCode);
        Boolean certain = thresholdProductItem.getCertain();
        Validate.notNull(certain , "创建优惠政策明细门槛商品时，必须指定该商品门槛是否是必须的（certain），请检查!!");
        SalePolicyThresholdProduct salePolicyThresholdProductEntity = this.nebulaToolkitService.copyObjectByBlankList(thresholdProductItem, SalePolicyThresholdProduct.class, LinkedHashSet.class, ArrayList.class);
        salePolicyThresholdProductEntity.setThresholdDetailId(salePolicyThresholdDetailEntity.getId());
        this.salePolicyThresholdProductRepository.save(salePolicyThresholdProductEntity);
        thresholdProductItem.setId(salePolicyThresholdProductEntity.getId());
      }
    }
    
    return salePolicyThresholdEntity;
  }

  @Override
  public SalePolicyThreshold findBySalePolicyCode(String salePolicyCode, String tenantCode) {
    if(StringUtils.isAnyBlank(salePolicyCode , tenantCode)) {
      return null;
    }
    
    return this.salePolicyThresholdRepository.findBySalePolicyCode(salePolicyCode, tenantCode);
  }
}
