package com.biz.crm.dms.business.policy.local.service.task;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.Callable;

import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.security.authentication.UsernamePasswordAuthenticationToken;
import org.springframework.security.core.authority.SimpleGrantedAuthority;
import org.springframework.security.core.context.SecurityContext;
import org.springframework.security.core.context.SecurityContextHolder;
import org.springframework.util.CollectionUtils;

import com.biz.crm.dms.business.policy.sdk.strategy.SalePolicyCustomerScopeMergeStrategy;
import com.biz.crm.dms.business.policy.sdk.strategy.SalePolicyCustomerScopeMergeStrategy.MergeResult;
import com.biz.crm.dms.business.policy.sdk.strategy.SalePolicyCustomerScopeStrategy;
import com.biz.crm.dms.business.policy.sdk.vo.AbstractSalePolicyCustomerScopeInfo;
import com.biz.crm.dms.business.policy.sdk.vo.SalePolicyVo;
import com.bizunited.nebula.security.sdk.config.SimpleSecurityProperties;
import com.bizunited.nebula.security.sdk.vo.DefaultLoginDetails;
import com.bizunited.nebula.security.sdk.vo.LoginDetails;
import com.google.common.collect.Lists;

/**
 * 优惠政策适用范围匹配任务
 * @author yinwenjie
 */
public class SalePolicyCustomerScopeMatchedTask implements Callable<Boolean> {
  /**
   * 目前整个系统中可以使用的用户范围匹配策略
   */
  @Autowired(required = false)
  private Set<SalePolicyCustomerScopeStrategy<? extends AbstractSalePolicyCustomerScopeInfo>> salePolicyCustomerScopeStrategies;
  /**
   * nebula security模块中，关于默认用户身份信息的配置情况
   */
  @Autowired(required = false)
  private SimpleSecurityProperties simpleSecurityProperties;
  @Autowired(required = false)
  private SalePolicyCustomerScopeMergeStrategy salePolicyCustomerScopeMergeStrategy;
  
  private SalePolicyVo salePolicyVo;
  
  private String tenantCode;
  
  private String customerCode;
  
  public SalePolicyCustomerScopeMatchedTask(SalePolicyVo salePolicyVo, String tenantCode , String customerCode) {
    this.salePolicyVo = salePolicyVo;
    this.tenantCode = tenantCode;
    this.customerCode = customerCode;
    Validate.notNull(salePolicyVo , "salePolicyCustomerScopeMatchedTask: salePolicy null !!");
    Validate.notBlank(tenantCode , "salePolicyCustomerScopeMatchedTask: tenantCode blank !!");
    Validate.notBlank(customerCode , "salePolicyCustomerScopeMatchedTask: customerCode blank !!");
  }
  
  @Override
  public Boolean call() throws Exception {
    /*
     * 加载过程为：
     * 1、首先为当前线程初始化相关的身份信息（这个身份信息来源于nebula security模块的基本设定）
     * 2、然后进行优惠政策的匹配
     * */
    
    // 1、======
    SecurityContext securityContext = SecurityContextHolder.getContext();
    List<SimpleGrantedAuthority> authorities = new ArrayList<>();
    String account = this.simpleSecurityProperties.getIndependencyUser();
    String[] independencyRoles = this.simpleSecurityProperties.getIndependencyRoles();
    Integer type = this.simpleSecurityProperties.getDefaultLoginType();
    for (String independencyRole : independencyRoles) {
      SimpleGrantedAuthority authoritie = new SimpleGrantedAuthority(StringUtils.upperCase(independencyRole));
      authorities.add(authoritie);
    }
    // 这里的密码不重要的
    UsernamePasswordAuthenticationToken authentication = new UsernamePasswordAuthenticationToken(account, "123455", authorities);
    LoginDetails loginDetails = new DefaultLoginDetails(type, tenantCode, account, null, null, null);
    authentication.setDetails(loginDetails);
    // 设定成当前线程的鉴权信息，但由于当前线程不再web容器中，所以没有sessionId信息
    securityContext.setAuthentication(authentication);
    
    // 2、======
    try {
      return this.matchedCustomerScopeMerge();
    } finally {
      SecurityContextHolder.clearContext();
    }
  }
  
  
  /**
   * 该私有方法进行具体的优惠政策-用户的匹配确认
   * @return
   */
  @SuppressWarnings({"rawtypes", "unchecked"})
  private boolean matchedCustomerScopeMerge() {
    Map<String, Set<? extends AbstractSalePolicyCustomerScopeInfo>> customerScopeStrategies = this.salePolicyVo.getCustomerScopeMapping();
    if(CollectionUtils.isEmpty(customerScopeStrategies)) {
      return false;
    }
    
    /*
     * 处理过程为：
     * 1、确定当前优惠政策上已经选择的所有用户范围维度（例如按照组织机构关联客户、按照渠道关联客户等等）
     * 2、然后调用每一个维度salePolicyCustomerScopeStrategy的matched方法进行“用户和优惠政策是否匹配”的确认
     * 并结合之前所有的匹配过程，调用salePolicyCustomerScopeMergeStrategy的合并方法merge最终确认用户和优惠政策是否匹配
     * */
    
    // 1、=========
    boolean currentMatched = false;
    List<Boolean> matchedList = Lists.newArrayList();
    int size = customerScopeStrategies.size();
    for (Map.Entry<String,Set<? extends AbstractSalePolicyCustomerScopeInfo>> customerScopeStrategyEntry : customerScopeStrategies.entrySet()) {
       String scopeType = customerScopeStrategyEntry.getKey();
       Set<? extends AbstractSalePolicyCustomerScopeInfo> salePolicyCustomerScopeInfos = customerScopeStrategyEntry.getValue();
       SalePolicyCustomerScopeStrategy<? extends AbstractSalePolicyCustomerScopeInfo> salePolicyCustomerScopeStrategy = this.findCurrentCustomerScopeStrategy(scopeType);
       if(salePolicyCustomerScopeStrategy == null) {
         return false;
       }
       
       // 2、========
       // 记录这些客户范围信息和匹配的客户业务编码信息
       boolean matched = salePolicyCustomerScopeStrategy.matched(this.customerCode, this.tenantCode, this.salePolicyVo.getSalePolicyCode(), (Set)salePolicyCustomerScopeInfos);
       matchedList.add(matched);
       // 开始判定本次匹配结果是否能决定指定客户和指定优惠政策的匹配是否成功
       MergeResult mergeResult = this.salePolicyCustomerScopeMergeStrategy.merge(matchedList, size, this.salePolicyVo, this.customerCode, matched , scopeType);
       if(mergeResult == MergeResult.FAILED) {
         return false;
       } else if(mergeResult == MergeResult.SUCCEEDED) {
         return true;
       }
    }
    return currentMatched;
  }
  
  /**
   * 该私有方法为寻找匹配的人员范围选择策略
   * @return
   */
  private SalePolicyCustomerScopeStrategy<? extends AbstractSalePolicyCustomerScopeInfo> findCurrentCustomerScopeStrategy(String scopeType) {
    if(CollectionUtils.isEmpty(salePolicyCustomerScopeStrategies)) {
      return null;
    }
    
    for (SalePolicyCustomerScopeStrategy<? extends AbstractSalePolicyCustomerScopeInfo> salePolicyCustomerScopeStrategy : salePolicyCustomerScopeStrategies) {
      String currentScopeType = salePolicyCustomerScopeStrategy.getScopeType();
      if(StringUtils.equals(currentScopeType, scopeType)) {
        return salePolicyCustomerScopeStrategy;
      }
    }
    return null;
  }
}
