package com.biz.crm.dms.business.policy.local.service.task;

import java.util.ArrayList;
import java.util.Collection;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.Callable;

import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.ApplicationContext;
import org.springframework.security.authentication.UsernamePasswordAuthenticationToken;
import org.springframework.security.core.authority.SimpleGrantedAuthority;
import org.springframework.security.core.context.SecurityContext;
import org.springframework.security.core.context.SecurityContextHolder;
import org.springframework.util.CollectionUtils;

import com.biz.crm.dms.business.policy.local.entity.SalePolicy;
import com.biz.crm.dms.business.policy.local.entity.SalePolicyTemplete;
import com.biz.crm.dms.business.policy.local.repository.SalePolicyTempleteRepository;
import com.biz.crm.dms.business.policy.sdk.register.SalePolicyTempleteRegister;

import com.biz.crm.dms.business.policy.sdk.strategy.SalePolicyCustomerScopeStrategy;
import com.biz.crm.dms.business.policy.sdk.strategy.SalePolicyExecuteStrategy;
import com.biz.crm.dms.business.policy.sdk.strategy.SalePolicyLimitStrategy;
import com.biz.crm.dms.business.policy.sdk.strategy.SalePolicyStickupListener;
import com.biz.crm.dms.business.policy.sdk.vo.AbstractSalePolicyCustomerScopeInfo;
import com.biz.crm.dms.business.policy.sdk.vo.AbstractSalePolicyExecutorInfo;
import com.biz.crm.dms.business.policy.sdk.vo.AbstractSalePolicyLimitInfo;
import com.biz.crm.dms.business.policy.sdk.vo.AbstractSalePolicyProductInfo;
import com.biz.crm.dms.business.policy.sdk.vo.AbstractSalePolicyThreshold;
import com.biz.crm.dms.business.policy.sdk.vo.SalePolicyVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.security.sdk.config.SimpleSecurityProperties;
import com.bizunited.nebula.security.sdk.vo.DefaultLoginDetails;
import com.bizunited.nebula.security.sdk.vo.LoginDetails;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;

public class SalePolicyLoadingTask implements Callable<SalePolicyVo> {
  @Autowired(required = false)
  private List<SalePolicyTempleteRegister> salePolicyTempleteRegisters;
  @Autowired(required = false)
  private ApplicationContext applicationContext;
  @Autowired(required = false)
  private NebulaToolkitService nebulaToolkitService;
  @Autowired(required = false)
  private SalePolicyTempleteRepository salePolicyTempleteRepository;
  /**
   * nebula security模块中，关于默认用户身份信息的配置情况
   */
  @Autowired(required = false)
  private SimpleSecurityProperties simpleSecurityProperties;
  /**
   * 日志
   */
  private static final Logger LOGGER = LoggerFactory.getLogger(SalePolicyLoadingTask.class);
  /**
   * 注意这里只包括基本信息，不包括任何优惠政策的关联信息
   */
  private SalePolicy salePolicy;
  
  private String tenantCode;
  
  public SalePolicyLoadingTask(SalePolicy salePolicy, String tenantCode) {
    this.salePolicy = salePolicy;
    this.tenantCode = tenantCode;
    Validate.notNull(salePolicy , "PolicyLoadingTask: salePolicy null !!");
    Validate.notBlank(tenantCode , "PolicyLoadingTask: tenantCode blank !!");
  }
  
  @Override
  public SalePolicyVo call() throws Exception {
    /*
     * 加载过程为：
     * 1、首先为当前线程初始化相关的身份信息（这个身份信息来源于nebula security模块的基本设定）
     * 2、然后在根据salePolicy，从数据持久层完善指定二级租户下目前所有执行中的和将要执行的优惠政策的关联信息到缓存中
     * */
    
    // 1、======
    SecurityContext securityContext = SecurityContextHolder.getContext();
    List<SimpleGrantedAuthority> authorities = new ArrayList<>();
    String account = this.simpleSecurityProperties.getIndependencyUser();
    String[] independencyRoles = this.simpleSecurityProperties.getIndependencyRoles();
    Integer type = this.simpleSecurityProperties.getDefaultLoginType();
    for (String independencyRole : independencyRoles) {
      SimpleGrantedAuthority authoritie = new SimpleGrantedAuthority(StringUtils.upperCase(independencyRole));
      authorities.add(authoritie);
    }
    // 这里的密码不重要的
    UsernamePasswordAuthenticationToken authentication = new UsernamePasswordAuthenticationToken(account, "123455", authorities);
    LoginDetails loginDetails = new DefaultLoginDetails(type, tenantCode, account, null, null, null);
    authentication.setDetails(loginDetails);
    // 设定成当前线程的鉴权信息，但由于当前线程不再web容器中，所以没有sessionId信息
    securityContext.setAuthentication(authentication);
    
    // 2、======
    try {
      return this.perfectByTenantCodeAndSalePolicyCodeFromRepository(this.salePolicy, tenantCode);
    } finally {
      SecurityContextHolder.clearContext();
    }
  }
  
  /**
   * 按照指定的二级租户和指定的优惠政策信息（数据层信息），从数据层查询并组装完善优惠政策信息</br>
   * 注意，该私有方法不负责获取操作权
   * @return 注意，如果返回null，则说明组装失败
   */
  private SalePolicyVo perfectByTenantCodeAndSalePolicyCodeFromRepository(SalePolicy salePolicy , String tenantCode) {
    String type = salePolicy.getType();
    String salePolicyCode = salePolicy.getSalePolicyCode();
    SalePolicyTempleteRegister matchedSalePolicyTempleteRegister = this.findSalePolicyTempleteRegister(type);
    if(matchedSalePolicyTempleteRegister == null) {
      LOGGER.warn("在进行优惠政策缓存加载时，未发现特定优惠政策模板的注册器({})，请检查!!" , salePolicyCode);
      return null;
    }
    SalePolicyStickupListener<? extends AbstractSalePolicyThreshold, ? extends AbstractSalePolicyProductInfo> stickupListener = matchedSalePolicyTempleteRegister.getStickupListener();
    SalePolicyVo salePolicyVo = this.nebulaToolkitService.copyObjectByWhiteList(salePolicy, SalePolicyVo.class, LinkedHashSet.class, ArrayList.class);
    // 加载可能的门槛信息(可能有)
    AbstractSalePolicyThreshold salePolicyThreshold = stickupListener.onRequestSalePolicyThreshold(tenantCode , salePolicyCode);
    if(salePolicyThreshold != null) {
      salePolicyVo.setSalePolicyProductThreshold(salePolicyThreshold);
    }
    // 加载执行信息(一定有)
    Collection<Class<? extends SalePolicyExecuteStrategy<? extends AbstractSalePolicyExecutorInfo>>> executeStrategyClasses = matchedSalePolicyTempleteRegister.getExecuteStrategyClasses();
    Set<AbstractSalePolicyExecutorInfo> salePolicyExecutorInfos = Sets.newLinkedHashSet();
    for (Class<? extends SalePolicyExecuteStrategy<? extends AbstractSalePolicyExecutorInfo>> executeStrategyClass : executeStrategyClasses) {
      SalePolicyExecuteStrategy<? extends AbstractSalePolicyExecutorInfo> currentExecuteStrategy = this.applicationContext.getBean(executeStrategyClass);
      List<? extends AbstractSalePolicyExecutorInfo> executorInfos = currentExecuteStrategy.onRequestSalePolicyExecutorInfo(tenantCode, salePolicyCode);
      if(!CollectionUtils.isEmpty(executorInfos)) {
        salePolicyExecutorInfos.addAll(executorInfos);
      }
    }
    if(CollectionUtils.isEmpty(salePolicyExecutorInfos)) {
      LOGGER.warn("在进行优惠政策缓存加载时，未发现特定优惠政策的优惠策略信息({})，可能是因为这个优惠政策信息的业务编号发生了变化（这可能是一种正常现象）!!" , salePolicyCode);
      return null;
    }
    salePolicyVo.setSalePolicyExecutorInfos(salePolicyExecutorInfos);
    // 加载物料信息（可能有）
    boolean supportProduct = matchedSalePolicyTempleteRegister.supportProduct();
    if(supportProduct) {
      List<? extends AbstractSalePolicyProductInfo> salePolicyProductInfos = stickupListener.onRequestSalePolicyProductInfos(tenantCode , salePolicyCode);
      if(CollectionUtils.isEmpty(salePolicyProductInfos)) {
        LOGGER.warn("在进行优惠政策缓存加载时，未发现特定优惠政策的商品本品信息({})，请检查!!" , salePolicyCode);
        return null;
      }
      salePolicyVo.setSalePolicyProductInfos(Sets.newLinkedHashSet(salePolicyProductInfos));
    } else {
      salePolicyVo.setSalePolicyProductInfos(null);
    }
    // 加载可能的预算信息（可能有） TODO 还没有弄
    // 优惠政策涉及的限量控制（可能有）
    Collection<Class<? extends SalePolicyLimitStrategy<? extends AbstractSalePolicyLimitInfo>>> bindableLimitStrategyClasses = matchedSalePolicyTempleteRegister.getBindableLimitStrategyClasses();
    if(!CollectionUtils.isEmpty(bindableLimitStrategyClasses) ) {
      Set<AbstractSalePolicyLimitInfo> salePolicyLimitInfos = Sets.newLinkedHashSet();
      // 寻找正确的策略处理器，并进行保存处理
      for (Class<? extends SalePolicyLimitStrategy<? extends AbstractSalePolicyLimitInfo>> salePolicyLimitStrategyClass : bindableLimitStrategyClasses) {
        SalePolicyLimitStrategy<? extends AbstractSalePolicyLimitInfo> currentSalePolicyLimitStrategy = this.applicationContext.getBean(salePolicyLimitStrategyClass);
        AbstractSalePolicyLimitInfo salePolicyLimitInfo = currentSalePolicyLimitStrategy.onRequestSalePolicyLimitInfos(salePolicyCode);
        if(salePolicyLimitInfo != null) {
          salePolicyLimitInfos.add(salePolicyLimitInfo);
        }
      }
      salePolicyVo.setSalePolicyLimitInfos(salePolicyLimitInfos);
    }
    
    // 加载客户信息（一定有）
    // 2022-06-10 根据项目上的需求，暴露优惠政策匹配客户范围的工作策略
    // 2022-09-24 根据最新的需求，缓存中不再落地存储到具体的客户code，而只是存储特定客户范围渠道的code
    Set<SalePolicyCustomerScopeStrategy<? extends AbstractSalePolicyCustomerScopeInfo>> currentCustomerScopeStrategies = this.findAllBindCustomerScopeStrategy(matchedSalePolicyTempleteRegister);
    Map<String , Set<? extends AbstractSalePolicyCustomerScopeInfo>> salePolicyCustomerInfoObjectMaps = this.customerScopeMerge(salePolicyVo, currentCustomerScopeStrategies);
    salePolicyVo.setCustomerScopeMapping(salePolicyCustomerInfoObjectMaps);
    
    // 最后进行优惠政策模板相关数据的完善工作
    String templeteCode = salePolicyVo.getTempleteCode();
    SalePolicyTemplete currentSalePolicyTemplete = this.salePolicyTempleteRepository.findByTempleteCode(templeteCode, tenantCode);
    if(currentSalePolicyTemplete != null) {
      String templeteType = currentSalePolicyTemplete.getType();
      salePolicyVo.setTempleteName(currentSalePolicyTemplete.getTempleteName());
      SalePolicyTempleteRegister salePolicyTempleteRegister = this.findSalePolicyTempleteRegister(templeteType);
      if(salePolicyTempleteRegister != null) {
        salePolicyVo.setTempleteRegisterType(templeteType);
        salePolicyVo.setTempleteRegisterTypeDesc(salePolicyTempleteRegister.getTypeDesc());
      }
    }
    return salePolicyVo;
  }
  
  /**
   * 该私有方法用于结合用户匹配范围工作策略SalePolicyCustomerScopeMergeStrategy，对指定优惠政策的客户选择范围进行确认
   * @return
   */
  private Map<String , Set<? extends AbstractSalePolicyCustomerScopeInfo>> customerScopeMerge(SalePolicyVo salePolicyVo , Set<SalePolicyCustomerScopeStrategy<? extends AbstractSalePolicyCustomerScopeInfo>> currentCustomerScopeStrategies) {
    Map<String , Set<? extends AbstractSalePolicyCustomerScopeInfo>> salePolicyCustomerInfoObjectMaps = Maps.newLinkedHashMap();
    String salePolicyCode = salePolicyVo.getSalePolicyCode();
    /*
     * SalePolicyLoadingTask任务的正式处理过程为：
     * 循环进行明一个currentCustomerScopeStrategies集合中要素的处理
     * （只包括优惠政策直接关联的用户匹配维度但是不包括具体的用户code对应关系）
     * */
    for (SalePolicyCustomerScopeStrategy<? extends AbstractSalePolicyCustomerScopeInfo> salePolicyCustomerScopeStrategy : currentCustomerScopeStrategies) {
      String scopeType = salePolicyCustomerScopeStrategy.getScopeType();
      Set<? extends AbstractSalePolicyCustomerScopeInfo> salePolicyCustomerInfos = salePolicyCustomerScopeStrategy.requestSalePolicyCustomerInfo(tenantCode, salePolicyCode);
      if(CollectionUtils.isEmpty(salePolicyCustomerInfos)) {
        continue;
      }
      salePolicyCustomerInfoObjectMaps.put(scopeType, salePolicyCustomerInfos); 
    }
    return salePolicyCustomerInfoObjectMaps;
  }
  
  /**
   * 该私有方法，根据type获得正确的SalePolicyRegister营销策略注册器
   * @param cacheResults
   * @return
   */
  private SalePolicyTempleteRegister findSalePolicyTempleteRegister(String type) {
    Validate.notBlank("错误的优惠政策类型，请检查传入值");
    Validate.isTrue(!CollectionUtils.isEmpty(this.salePolicyTempleteRegisters) , "错误的SalePolicyTempleteRegister（集合）信息，请检查!!");
    SalePolicyTempleteRegister matchedSalePolicyTempleteRegister = null;
    for (SalePolicyTempleteRegister item : salePolicyTempleteRegisters) {
      if(StringUtils.equals(type, item.getType())) {
        matchedSalePolicyTempleteRegister = item;
        break;
      }
    }
    Validate.notNull(matchedSalePolicyTempleteRegister , "未匹配任何SalePolicyTempleteRegister信息，请检查!!");
    return matchedSalePolicyTempleteRegister;
  }
  
  /**
   * 该私有方法为寻找匹配的人员范围选择策略
   * @return
   */
  private Set<SalePolicyCustomerScopeStrategy<? extends AbstractSalePolicyCustomerScopeInfo>> findAllBindCustomerScopeStrategy(SalePolicyTempleteRegister salePolicyTempleteRegister) {
    Collection<Class<? extends SalePolicyCustomerScopeStrategy<? extends AbstractSalePolicyCustomerScopeInfo>>> salePolicyCustomerScopeStrategyClasses = salePolicyTempleteRegister.getCustomerScopeStrategyClasses();
    Validate.isTrue(!CollectionUtils.isEmpty(salePolicyCustomerScopeStrategyClasses) , "该优惠政策并没有注册任何客户返回选择方式，请检查!!");
    Set<SalePolicyCustomerScopeStrategy<?>> bindableCustomerScopeStrategies = Sets.newLinkedHashSet();
    for (Class<? extends SalePolicyCustomerScopeStrategy<?>> salePolicyCustomerScopeStrategyClass : salePolicyCustomerScopeStrategyClasses) {
      try {
        SalePolicyCustomerScopeStrategy<?> currentSalePolicyCustomerScopeStrategy = applicationContext.getBean(salePolicyCustomerScopeStrategyClass);
        String currentScopeType = currentSalePolicyCustomerScopeStrategy.getScopeType();
        Validate.notBlank(currentScopeType , "发现至少一个客户范围策略没有指定code信息(%s)，请检查!!" , salePolicyCustomerScopeStrategyClass.getName());
        bindableCustomerScopeStrategies.add(currentSalePolicyCustomerScopeStrategy);
      } catch(RuntimeException e) {
        LOGGER.error(e.getMessage() , e);
        throw new IllegalArgumentException(e.getMessage() , e);
      }
    }
    return bindableCustomerScopeStrategies;
  }
}
