package com.biz.crm.dms.business.policy.local.sharestrategy;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.util.List;
import java.util.Set;

import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;

import com.biz.crm.dms.business.policy.sdk.context.AbstractCycleExecuteContext;
import com.biz.crm.dms.business.policy.local.context.DefaultCycleExecuteContext;
import com.biz.crm.dms.business.policy.local.context.DefaultPolicyExecuteContext;
import com.biz.crm.dms.business.policy.sdk.context.GiftResultInfo;
import com.biz.crm.dms.business.policy.sdk.context.ProductPolicyStepResult;
import com.biz.crm.dms.business.policy.sdk.context.StepType;
import com.biz.crm.dms.business.policy.sdk.strategy.SalePolicyExecuteShareStrategy;
import com.biz.crm.dms.business.policy.sdk.context.AbstractPolicyExecuteContext;
import com.biz.crm.dms.business.policy.sdk.context.CycleStepResult;
import com.biz.crm.dms.business.policy.sdk.vo.SalePolicyVo;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;

/**
 * 将优惠后的金额，基于各个参与优惠的商品明细在优惠前的小计金额比例，分摊到商品小计上（且需要支持揉价和不揉价两种模式）</br>
 * 这种分摊方式适用于按照小计价格进行的打折、满减优惠——无论是商品级别的优惠还是整单级别的优惠。</p>
 * 
 * 该分摊逻辑，从2022-5-10创建，是优惠政策分摊逻辑的第二次需求变更。变更的核心点在于：</p>
 * 
 * 采用这种分摊逻辑进行计算后，各本品未参与优惠执行的剩余数量为0；</br>
 * 各本品未参与优惠执行的剩余金额 = 根据揉价和不揉价两种场景，具体进行计算
 * 
 * @author yinwenjie
 */
@Component
public class SubtotalShareStrategyWithAmountAndNumber implements SalePolicyExecuteShareStrategy {
  
  @Override
  public void share(String customer , SalePolicyVo salePolicy ,  AbstractPolicyExecuteContext abstractPolicyExecuteContext , AbstractCycleExecuteContext abstractCycleExecuteContext , Set<String> productCodes) {
    DefaultPolicyExecuteContext policyExecuteContext = (DefaultPolicyExecuteContext)abstractPolicyExecuteContext;
    DefaultCycleExecuteContext cycleExecuteContext = (DefaultCycleExecuteContext)abstractCycleExecuteContext;
    // 揉价和不揉价的两套逻辑要分开
    if(policyExecuteContext.isKneading()) {
      this.shareForKneading(customer, salePolicy, policyExecuteContext, cycleExecuteContext, productCodes);
    } else {
      this.share(customer, salePolicy, policyExecuteContext, cycleExecuteContext, productCodes);
    }
  }
  
  /**
   * 不使用揉价的方式进行分摊，也就是说本次优惠的金额要按照各本品的小计价格分摊到每个本品上
   */
  private void share(String customer , SalePolicyVo salePolicy ,  DefaultPolicyExecuteContext policyExecuteContext , DefaultCycleExecuteContext cycleExecuteContext , Set<String> productCodes) {
    /* 
     * 处理逻辑为：
     * 1、累加参与本次计算的所有本品（如果需要揉价，则需要考虑赠品小计价值）的总计价值
     * 2、按照各个本品在总价价值中的比例，将总的优惠金额分摊到每个本品上，并记录到policyExecuteContext中
     * 
     * 注意：小计价格为0的本品不进行分摊（经过诸如满减单价这样的优惠政策后，小计价格完全可能为0）
     * */

    // 1、======
    CycleStepResult cycleLastStepResult = cycleExecuteContext.findLastStepResult();
    BigDecimal cycleLastSubtotalAmount = cycleLastStepResult.getLastSubtotalAmount();
    // 这个变量cycleLastSurplusSubtotalAmount代表，经过本次所有阶梯计算后，总计还有多少价格没有享受优惠
    BigDecimal cycleLastSurplusSubtotalAmount = cycleLastStepResult.getLastSurplusSubtotalAmount(); 
    // 变量 sumLastSubtotal 表示没有进行本次优惠前，这些写商品累计的原始价格
    BigDecimal sumLastSubtotal = BigDecimal.ZERO;
    Set<ProductPolicyStepResult> newProductPolicyStepResults = Sets.newLinkedHashSet();
    for (String productCode : productCodes) {
      ProductPolicyStepResult productPolicyStepResult = policyExecuteContext.findLastProductStepResultByProductCode(productCode);
      // 正常情况不会出现这种问题，但如果出现了，则继续处理
      if(productPolicyStepResult == null) {
        continue;
      }
      BigDecimal lastSubtotal = productPolicyStepResult.getLastSubtotal();
      sumLastSubtotal = sumLastSubtotal.add(lastSubtotal);
    }
    // 所以变量 differenceSubtotalAmount 就表示这次优惠一共优惠了多少小计价格(以便进行分摊)
    BigDecimal differenceSubtotalAmount = sumLastSubtotal.subtract(cycleLastSubtotalAmount);
    
    // 2、========
    // deductedDifferenceSubtotalAmount 表示已计扣的本品+赠品优惠金额差值
    BigDecimal deductedDifferenceSubtotalAmount = BigDecimal.ZERO;
    // 计算出每个本品的比例（注意最后一次计算，就是剩余数量，如果再使用除法，可能造成小数精度错误）
    String[] productCodesArray = productCodes.toArray(new String[] {});
    int size = productCodesArray.length;
    for (int index = 0 ; index < size ; index++) {
      String productCode = productCodesArray[index];
      ProductPolicyStepResult productPolicyStepResult = policyExecuteContext.findLastProductStepResultByProductCode(productCode);
      // 正常情况不会出现这种问题，但如果出现了，则继续处理
      if(productPolicyStepResult == null) {
        continue;
      }
      BigDecimal lastSubtotal = productPolicyStepResult.getLastSubtotal();
      if(lastSubtotal.compareTo(BigDecimal.ZERO) != 1) {
        continue;
      }
      // 为当前商品添加一个新的优惠步进
      ProductPolicyStepResult newProductPolicyStepResult = new ProductPolicyStepResult();
      newProductPolicyStepResult.setStepType(salePolicy.getWholePolicy()?StepType.POLICY:StepType.PRODUCT);
      newProductPolicyStepResult.setSalePolicyCode(salePolicy.getSalePolicyCode());
      newProductPolicyStepResult.setExecuteCode(cycleExecuteContext.getExecuteCode());
      newProductPolicyStepResult.setProductCode(productCode);
      newProductPolicyStepResult.setInitNumbers(productPolicyStepResult.getInitNumbers());
      newProductPolicyStepResult.setInitPrices(productPolicyStepResult.getInitPrices());
      newProductPolicyStepResult.setInitSubtotal(productPolicyStepResult.getInitSubtotal());
      newProductPolicyStepResult.setPreGifts(productPolicyStepResult.getLastGifts());
      newProductPolicyStepResult.setPrePrices(productPolicyStepResult.getLastPrices());
      newProductPolicyStepResult.setPreSubtotal(productPolicyStepResult.getLastSubtotal());
      newProductPolicyStepResult.setPreSurplusTotalAmount(productPolicyStepResult.getLastSurplusTotalAmount());
      newProductPolicyStepResult.setPreSurplusTotalNumber(productPolicyStepResult.getLastSurplusTotalNumber());
      newProductPolicyStepResult.setPreGiftEnjoyedTotalNumber(productPolicyStepResult.getLastGiftEnjoyedTotalNumber());
      newProductPolicyStepResult.setPreGiftEnjoyedTotalAmount(productPolicyStepResult.getLastGiftEnjoyedTotalAmount());
      // 因为不是买赠优惠，所以赠品数量、已享受优惠的赠品数量和金额不会改变
      newProductPolicyStepResult.setLastGifts(productPolicyStepResult.getLastGifts());
      newProductPolicyStepResult.setLastGiftEnjoyedTotalNumber(productPolicyStepResult.getLastGiftEnjoyedTotalNumber());
      newProductPolicyStepResult.setLastGiftEnjoyedTotalAmount(productPolicyStepResult.getLastGiftEnjoyedTotalAmount());
      
      /*
       * 本质上要完成以下计算任务：
       * a、计算本次阶梯总体优惠的金额（differenceSubtotalAmount），按照每个本品的小计价格分摊到每个本品后，每个本品分别优惠了多少小计价格：
       * 优惠了多少小计价格 = 本品小计价格 / 总体小计价格 *  differenceSubtotalAmount
       * b、计算这个本品最新的小计价格
       * 最新小计价格 = 商品当前小计价格 - 优惠了多少小计价格
       * c、计算这个本品最新的单价
       * 最新单价 = 最新小计价格 / 商品本品数量
       * d、计算这个本品还有多少小计价格，没有参与任何优惠
       * 还没有参与优惠的本品小计价格 = 商品之前没有参与优惠的小计价格 - (本品小计价格 / 总体小计价格  * 本次有多少总价享受优惠)
       * e、从2022-05-10开始
       * 还没有参与优惠的本品小计数量 = 0
       * */
      // a、===
      BigDecimal lastRealSubtotalAmount = null;
      if(index + 1 < size) {
        lastRealSubtotalAmount = differenceSubtotalAmount.multiply(lastSubtotal.divide(sumLastSubtotal, 20, RoundingMode.HALF_UP)).setScale(4, RoundingMode.HALF_UP);
      }      
      // 注意，最后一次计算不能用除法，因为会造成精度缺失.应该用减法
      else {
        lastRealSubtotalAmount = differenceSubtotalAmount.subtract(deductedDifferenceSubtotalAmount).setScale(4, RoundingMode.DOWN);
      }
      deductedDifferenceSubtotalAmount = deductedDifferenceSubtotalAmount.add(lastRealSubtotalAmount);
      // b、===
      BigDecimal newLastSubtotal = lastSubtotal.subtract(lastRealSubtotalAmount);
      newProductPolicyStepResult.setLastSubtotal(newLastSubtotal);
      // c、===
      BigDecimal newLastPrices = newLastSubtotal.divide(new BigDecimal(productPolicyStepResult.getInitNumbers()) , 20 , RoundingMode.HALF_UP).setScale(4, RoundingMode.HALF_UP);
      newProductPolicyStepResult.setLastPrices(newLastPrices);
      // d、===
      BigDecimal productCycleLastRealSubtotalAmount = cycleLastSurplusSubtotalAmount.multiply(lastSubtotal.divide(sumLastSubtotal, 20, RoundingMode.HALF_UP)).setScale(4, RoundingMode.HALF_UP);
      newProductPolicyStepResult.setLastSurplusTotalAmount(productCycleLastRealSubtotalAmount);
      // e、===（注意，单价必须 > 0 ，否则不进行数量分摊）
      // 按照要求，执行单价满减后，该本品未进行优惠的数量，直接记为 0
      newProductPolicyStepResult.setLastSurplusTotalNumber(0);
      newProductPolicyStepResults.add(newProductPolicyStepResult);
    }
    policyExecuteContext.addPolicyStepResult(newProductPolicyStepResults);
  }
  
  /**
   * 使用揉价的方式进行分摊，也就是说本次优惠的金额要按照各本品、赠品的小计价格分摊到每个本品和赠品上
   */
  private void shareForKneading(String customer , SalePolicyVo salePolicy ,  DefaultPolicyExecuteContext policyExecuteContext , DefaultCycleExecuteContext cycleExecuteContext , Set<String> productCodes) {
    /*
     * 对揉价进行的处理过程：
     * 1、累加参与本次计算的所有本品+赠品的总计价值
     * 注意，要同时得到本次优惠前后的总价价格的差价（实际上由于优惠时只减少了本品的价值，所以差价发生的原因都在本品上）
     * 2、按照各个本品+赠品的总价值比例，将总的优惠金额（价值差价）进行分摊，重新计算各本品和赠品的单价，已形成本品最新的步进信息。
     * （计算公式，参见代码中的详细说明）
     * 
     * 3、注意：由于不清楚最后一个赠品出现在哪一个本品下，所以这里的优惠值精度的修正不能简单采用“最后一次循环商品做减法”的方式进行
     * 而应该采用事后修正最后一个本品优惠值精度的方式进行。
     * 
     * 4、重新汇总，形成最新的优惠步进信息。
     * 
     * 注意：小计价格为0的本品不进行分摊（经过诸如满减单价这样的优惠政策后，小计价格完全可能为0）
     * */

    // 1、======
    CycleStepResult cycleLastStepResult = cycleExecuteContext.findLastStepResult();
    BigDecimal cycleLastSubtotalAmount = cycleLastStepResult.getLastSubtotalAmount();
    // 这个变量cycleLastSurplusSubtotalAmount代表，经过本次所有阶梯计算后，总计还有多少价格没有享受优惠
    BigDecimal cycleLastSurplusSubtotalAmount = cycleLastStepResult.getLastSurplusSubtotalAmount(); 
    // 变量 sumLastSubtotal 表示没有进行本次优惠前，这些商品（本品+赠品）累计的原始价格
    // 变量 sumProductLastSubtotal 表示没有进行本次优惠前，这些本品累计的原始价格
    BigDecimal sumLastSubtotal = BigDecimal.ZERO;
    BigDecimal sumProductLastSubtotal = BigDecimal.ZERO;
    Set<ProductPolicyStepResult> newProductPolicyStepResults = Sets.newLinkedHashSet();
    for (String productCode : productCodes) {
      ProductPolicyStepResult productPolicyStepResult = policyExecuteContext.findLastProductStepResultByProductCode(productCode);
      BigDecimal lastSubtotal = productPolicyStepResult.getLastSubtotal();
      sumLastSubtotal = sumLastSubtotal.add(lastSubtotal);
      sumProductLastSubtotal = sumProductLastSubtotal.add(lastSubtotal);
      // 累加可能存在的赠品价值
      List<GiftResultInfo> lastGifts = productPolicyStepResult.getLastGifts();
      if(!CollectionUtils.isEmpty(lastGifts)) {
        for (GiftResultInfo giftResultInfo : lastGifts) {
          sumLastSubtotal = sumLastSubtotal.add(giftResultInfo.getSubtotalAmount());
        }
      }
    }
    // 所以变量 differenceSubtotalAmount 就表示这次优惠一共优惠了多少小计价格(以便进行分摊)
    BigDecimal differenceSubtotalAmount = sumProductLastSubtotal.subtract(cycleLastSubtotalAmount);
    
    // 2、=======
    // deductedDifferenceSubtotalAmount 表示已计扣的本品+赠品优惠金额差值
    BigDecimal deductedDifferenceSubtotalAmount = BigDecimal.ZERO;
    String[] productCodesArray = productCodes.toArray(new String[] {});
    int size = productCodesArray.length;
    // 该变量lastNewProductPolicyStepResult表示最后一个新增的本品步进信息
    ProductPolicyStepResult lastNewProductPolicyStepResult = null;
    for (int index = 0 ; index < size ; index++) {
      String productCode = productCodesArray[index];
      ProductPolicyStepResult productPolicyStepResult = policyExecuteContext.findLastProductStepResultByProductCode(productCode);
      BigDecimal lastSubtotal = productPolicyStepResult.getLastSubtotal();
      if(lastSubtotal.compareTo(BigDecimal.ZERO) != 1) {
        continue;
      }
      // 为当前商品添加一个新的优惠步进
      ProductPolicyStepResult newProductPolicyStepResult = new ProductPolicyStepResult();
      lastNewProductPolicyStepResult = newProductPolicyStepResult;
      newProductPolicyStepResult.setStepType(salePolicy.getWholePolicy()?StepType.POLICY:StepType.PRODUCT);
      newProductPolicyStepResult.setSalePolicyCode(salePolicy.getSalePolicyCode());
      newProductPolicyStepResult.setExecuteCode(cycleExecuteContext.getExecuteCode());
      newProductPolicyStepResult.setProductCode(productCode);
      newProductPolicyStepResult.setInitNumbers(productPolicyStepResult.getInitNumbers());
      newProductPolicyStepResult.setInitPrices(productPolicyStepResult.getInitPrices());
      newProductPolicyStepResult.setInitSubtotal(productPolicyStepResult.getInitSubtotal());
      List<GiftResultInfo> lastGifts = productPolicyStepResult.getLastGifts();
      newProductPolicyStepResult.setPreGifts(lastGifts);
      newProductPolicyStepResult.setPrePrices(productPolicyStepResult.getLastPrices());
      newProductPolicyStepResult.setPreSubtotal(productPolicyStepResult.getLastSubtotal());
      newProductPolicyStepResult.setPreSurplusTotalAmount(productPolicyStepResult.getLastSurplusTotalAmount());
      newProductPolicyStepResult.setPreSurplusTotalNumber(productPolicyStepResult.getLastSurplusTotalNumber());
      newProductPolicyStepResult.setPreGiftEnjoyedTotalNumber(productPolicyStepResult.getLastGiftEnjoyedTotalNumber());
      newProductPolicyStepResult.setPreGiftEnjoyedTotalAmount(productPolicyStepResult.getLastGiftEnjoyedTotalAmount());
      // 因为不是按照数量进行的优惠，所以“还有多少本品数量还未进行优惠”的值，也不会改变
      newProductPolicyStepResult.setLastGiftEnjoyedTotalNumber(productPolicyStepResult.getLastGiftEnjoyedTotalNumber());
      newProductPolicyStepResult.setLastGiftEnjoyedTotalAmount(productPolicyStepResult.getLastGiftEnjoyedTotalAmount());
      
      /*
       * 本质上要完成以下计算任务：
       * a、计算本次阶梯总体优惠的金额（differenceSubtotalAmount），按照每个本品的小计价格分摊到每个本品后，每个本品分别优惠了多少小计价格：
       * 优惠了多少小计价格 = 本品小计价格 / 总体小计价格 *  differenceSubtotalAmount
       * b、计算这个本品最新的小计价格
       * 最新小计价格 = 商品当前小计价格 - 优惠了多少小计价格
       * d、计算这个本品还有多少小计价格，没有参与任何优惠
       * 还没有参与优惠的本品小计价格 = 商品之前没有参与优惠的小计价格 - (本品小计价格 / 总体小计价格  * 本次有多少总价享受优惠)
       * e、如果这个本品存在赠品，还需要按照a、b、c的公式，计算赠品的小计价格和单价变化
       * */
      // a、===
      BigDecimal lastRealDifferenceSubtotalAmount = differenceSubtotalAmount.multiply(lastSubtotal.divide(sumLastSubtotal, 20, RoundingMode.HALF_UP)).setScale(4, RoundingMode.HALF_UP);
      deductedDifferenceSubtotalAmount = deductedDifferenceSubtotalAmount.add(lastRealDifferenceSubtotalAmount);
      // b、===
      BigDecimal newLastSubtotal = lastSubtotal.subtract(lastRealDifferenceSubtotalAmount);
      newProductPolicyStepResult.setLastSubtotal(newLastSubtotal);
      // c、===
      BigDecimal newLastPrices = newLastSubtotal.divide(new BigDecimal(productPolicyStepResult.getInitNumbers()) , 20 , RoundingMode.HALF_UP).setScale(4, RoundingMode.HALF_UP);
      newProductPolicyStepResult.setLastPrices(newLastPrices);
      // d、===
      // 无论是否分摊：“还未进行优惠的金额”都按照本品小计金额进行计算
      // 无论是否分摊，该本品未进行优惠的数量，直接记为 0
      BigDecimal productCycleLastRealSubtotalAmount = cycleLastSurplusSubtotalAmount.multiply(lastSubtotal.divide(sumProductLastSubtotal, 20, RoundingMode.HALF_UP)).setScale(4, RoundingMode.HALF_UP);
      newProductPolicyStepResult.setLastSurplusTotalAmount(productCycleLastRealSubtotalAmount);
      newProductPolicyStepResult.setLastSurplusTotalNumber(0);
      // e、===
      if(!CollectionUtils.isEmpty(lastGifts)) {
        List<GiftResultInfo> newGiftResultInfos = Lists.newArrayList();
        for (int giftIndex = 0 ; giftIndex < lastGifts.size() ; giftIndex++) {
          GiftResultInfo lastGift = lastGifts.get(giftIndex);
          GiftResultInfo newGiftResultInfo = new GiftResultInfo();
          newGiftResultInfo.setProductCode(lastGift.getProductCode());
          newGiftResultInfo.setProductName(lastGift.getProductName());
          newGiftResultInfo.setQuantity(lastGift.getQuantity());
          BigDecimal lastSubtotalAmount = lastGift.getSubtotalAmount();
          // a、===
          BigDecimal gift_lastRealSubtotalAmount = differenceSubtotalAmount.multiply(lastSubtotalAmount.divide(sumLastSubtotal, 20, RoundingMode.HALF_UP)).setScale(4, RoundingMode.HALF_UP);
          deductedDifferenceSubtotalAmount = deductedDifferenceSubtotalAmount.add(gift_lastRealSubtotalAmount);
          // b、===
          BigDecimal gift_newLastSubtotal = lastSubtotalAmount.subtract(gift_lastRealSubtotalAmount);
          newGiftResultInfo.setSubtotalAmount(gift_newLastSubtotal);
          newGiftResultInfos.add(newGiftResultInfo);
        }
        newProductPolicyStepResult.setLastGifts(newGiftResultInfos);
      } else {
        newProductPolicyStepResult.setLastGifts(productPolicyStepResult.getLastGifts());
      }
      newProductPolicyStepResults.add(newProductPolicyStepResult);
    }
    
    this.correctAccuracy(lastNewProductPolicyStepResult, differenceSubtotalAmount, deductedDifferenceSubtotalAmount);
    policyExecuteContext.addPolicyStepResult(newProductPolicyStepResults);
  }
  
  /**
   * 步骤3：开始进行精度调整，精度缺失或者增益的0.0001，将在最后一个本品处进行调整
   */
  private void correctAccuracy(ProductPolicyStepResult lastNewProductPolicyStepResult , BigDecimal differenceSubtotalAmount , BigDecimal deductedDifferenceSubtotalAmount) {
    // 如果条件成立，说明精度增益了0.0001
    if(lastNewProductPolicyStepResult != null && differenceSubtotalAmount.compareTo(deductedDifferenceSubtotalAmount) == -1) {
      BigDecimal lastSurplusTotalAmount = lastNewProductPolicyStepResult.getLastSurplusTotalAmount();
      BigDecimal lastSubtotal = lastNewProductPolicyStepResult.getLastSubtotal();
      lastNewProductPolicyStepResult.setLastSurplusTotalAmount(lastSurplusTotalAmount.subtract(new BigDecimal(0.0001)).setScale(4, RoundingMode.DOWN));
      lastNewProductPolicyStepResult.setLastSubtotal(lastSubtotal.subtract(new BigDecimal(0.0001)).setScale(4, RoundingMode.DOWN));
    } 
    // 如果条件成立，说明精度缺失了0.0001
    else if(lastNewProductPolicyStepResult != null && differenceSubtotalAmount.compareTo(deductedDifferenceSubtotalAmount) == 1) {
      BigDecimal lastSurplusTotalAmount = lastNewProductPolicyStepResult.getLastSurplusTotalAmount();
      BigDecimal lastSubtotal = lastNewProductPolicyStepResult.getLastSubtotal();
      lastNewProductPolicyStepResult.setLastSurplusTotalAmount(lastSurplusTotalAmount.add(new BigDecimal(0.0001)).setScale(4, RoundingMode.DOWN));
      lastNewProductPolicyStepResult.setLastSubtotal(lastSubtotal.add(new BigDecimal(0.0001)).setScale(4, RoundingMode.DOWN));
    }
  }
}
