package com.biz.crm.dms.business.policy.local.config;

import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;

import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Scope;

import com.biz.crm.dms.business.policy.local.entity.SalePolicy;
import com.biz.crm.dms.business.policy.local.scopestrategy.UnionSalePolicyCustomerScopeMergeStrategy;
import com.biz.crm.dms.business.policy.local.service.internal.SalePolicyGiftInquiryStrategyImpl;
import com.biz.crm.dms.business.policy.local.service.internal.SalePolicyMatchedStrategyImpl;
import com.biz.crm.dms.business.policy.local.service.task.SalePolicyCustomerScopeMatchedTask;
import com.biz.crm.dms.business.policy.local.service.task.SalePolicyLoadingTask;
import com.biz.crm.dms.business.policy.sdk.strategy.SalePolicyCustomerScopeMergeStrategy;
import com.biz.crm.dms.business.policy.sdk.strategy.SalePolicyGiftInquiryStrategy;
import com.biz.crm.dms.business.policy.sdk.strategy.SalePolicyMatchedStrategy;
import com.biz.crm.dms.business.policy.sdk.vo.SalePolicyVo;

/**
 * 标品中默认的优惠政策底座功能的配置设定</br>
 * 主要是针对缓存加载的专用线程池进行的配置
 * @author yinwenjie
 */
@Configuration
public class DefaultPolicyLocalConfig {
  private final static Integer PROCESSINGCOUNT = Runtime.getRuntime().availableProcessors();
  
  /**
   * 该线程池，专门用于优惠政策的数据从持久层加载到进程内存中的工作过程。用于提高加载效率
   * @return
   */
  @Bean("policyLoadingExecutor")
  @ConditionalOnMissingBean
  public ThreadPoolExecutor getPolicyLoadingExecutor() { 
    return new ThreadPoolExecutor(PROCESSINGCOUNT, PROCESSINGCOUNT, 1, TimeUnit.HOURS, new LinkedBlockingQueue<>() , new PolicyLoadingExecutorThreadFactory());
  } 
  
  /**
   * 具体的优惠政策加载任务（多例）
   * @return
   */
  @Bean
  @Scope("prototype")
  public SalePolicyLoadingTask getPolicyLoadingTask(SalePolicy salePolicy , String tenantCode) {
    return new SalePolicyLoadingTask(salePolicy , tenantCode);
  }
  
  /**
   * 具体人员和具体优惠政策的匹配任务
   * @return
   */
  @Bean
  @Scope("prototype")
  public SalePolicyCustomerScopeMatchedTask getSalePolicyCustomerScopeMatchedTask(SalePolicyVo salePolicyVo , String tenantCode , String customerCode) {
    return new SalePolicyCustomerScopeMatchedTask(salePolicyVo, tenantCode, customerCode);
  }
  
  /**
   * 默认的赠品/商品询价逻辑的实现，适用于大部分项目情况；如果项目有需要，也可自行实现
   * @return
   */
  @Bean
  @ConditionalOnMissingBean
  public SalePolicyGiftInquiryStrategy getSalePolicyGiftInquiryStrategy() {
    return new SalePolicyGiftInquiryStrategyImpl();
  }
  
  /**
   * 标品中的默认匹配策略，只匹配其中客户和商品本身的限制
   * @return
   */
  @Bean
  @ConditionalOnMissingBean
  public SalePolicyMatchedStrategy getSalePolicyMatchedStrategy() {
    return new SalePolicyMatchedStrategyImpl();
  }
  
  /**
   * 标品中默认提供的优惠政策客户有效范围的确认策略（取并集）
   * @return
   */
  @Bean
  @ConditionalOnMissingBean
  public SalePolicyCustomerScopeMergeStrategy getSalePolicyCustomerScopeMergeStrategy() {
    return new UnionSalePolicyCustomerScopeMergeStrategy();
  }
  
  private class PolicyLoadingExecutorThreadFactory implements ThreadFactory {
    private AtomicInteger count = new AtomicInteger(1);
    @Override
    public Thread newThread(Runnable r) {
      return new Thread(r, "policy-loading-task-" + count.getAndIncrement());
    }
  }
}
