package com.biz.crm.dms.business.policy.local.controller;

import java.util.Map;
import java.util.Set;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.util.CollectionUtils;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import com.biz.crm.business.common.sdk.model.Result;
import com.biz.crm.dms.business.policy.local.context.DefaultPolicyExecuteContext;
import com.biz.crm.dms.business.policy.sdk.context.SalePolicyConProduct;
import com.biz.crm.dms.business.policy.sdk.service.SalePolicyExecuteService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.google.common.collect.Maps;

import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;

/**
 * 该http层功能提供的接口，都与优惠政策计算/预计算/预判定有关
 * @author yinwenjie
 */
@Api(tags = "该http层功能提供的接口，都与优惠政策计算/预计算/预判定有关")
@RestController
@RequestMapping("/v1/salepolicies/execute")
public class SalePolicyExecuteController {
  /**
   * 日志
   */
  private static final Logger LOGGER = LoggerFactory.getLogger(SalePolicyExecuteController.class);
  /**
   * 现在系统中支持的所有优惠政策注册信息
   */
  @Autowired(required = false)
  private SalePolicyExecuteService<DefaultPolicyExecuteContext> salePolicyExecuteService;
  
  /**
   * 预判定执行。预判定场景无需知晓商品数量，不考虑优惠政策的门槛、不考虑订单总价，不考虑政策叠加、政策阶梯</br>
   * 预判定场景适用于商品列表展示功能、购物车列表展示功能
   * */
  @ApiOperation(value = "预判定执行。预判定场景无需知晓商品数量，不考虑优惠政策的门槛、不考虑订单总价，不考虑政策叠加、政策阶梯" ,  notes = "预判定场景适用于商品列表展示功能、购物车列表展示功能")
  @PostMapping(value="/preJudgment")
  public Result<DefaultPolicyExecuteContext> preJudgment(@RequestBody Set<SalePolicyConProduct> initPolicyProducts , @RequestParam("customerCode") String customerCode) {
    try {
      String tenantCode = TenantUtils.getTenantCode();
      DefaultPolicyExecuteContext policyExecuteContext = salePolicyExecuteService.preJudgment(initPolicyProducts, tenantCode, customerCode);
      return Result.ok(policyExecuteContext);
    } catch(RuntimeException e) {
      LOGGER.error(e.getMessage() , e);
      return Result.error(e.getMessage());
    }
  }
  
  /**
   * 让一批商品规格，基于经销商为指定客户提供的正在执行中的活动策略进行预计算，最终得到这批物资在各个维度层面上的优惠信息
   * */
  @ApiOperation(value = "让一批商品规格，基于经销商为指定客户提供的正在执行中的活动策略进行预计算，最终得到这批物资在各个维度层面上的优惠信息")
  @PostMapping(value="/preexecute")
  public Result<DefaultPolicyExecuteContext> preExecute(@RequestBody Set<SalePolicyConProduct> initPolicyProducts , @RequestParam("customerCode") String customerCode , @RequestParam(value = "kneading" , required = false , defaultValue = "false") boolean kneading) {
    try {
      String tenantCode = TenantUtils.getTenantCode();
      // 根据initPolicyProducts中，是否指定了某些本品只能参与的优惠政策的情况，决定调用哪个接口
      Map<String , String[]> mustCorrectPolicyMapping = Maps.newLinkedHashMap();
      for (SalePolicyConProduct defaultPolicyConProduct : initPolicyProducts) {
        String[] selectedPolicyCodes = defaultPolicyConProduct.getSelectedPolicyCodes();
        if(selectedPolicyCodes != null) {
          mustCorrectPolicyMapping.put(defaultPolicyConProduct.getProductCode(), selectedPolicyCodes);
        }
      }
      
      DefaultPolicyExecuteContext policyExecuteContext = null;
      if(CollectionUtils.isEmpty(mustCorrectPolicyMapping)) {
        policyExecuteContext = salePolicyExecuteService.preExecute(initPolicyProducts, tenantCode, customerCode , kneading);
      } else {
        policyExecuteContext = salePolicyExecuteService.preExecute(initPolicyProducts, mustCorrectPolicyMapping, tenantCode, customerCode , kneading);
      }
      return Result.ok(policyExecuteContext);
    } catch(RuntimeException e) {
      LOGGER.error(e.getMessage() , e);
      return Result.error(e.getMessage());
    }
  }
  
  /**
   * 让一批商品规格，基于经销商为指定客户提供的正在执行中的活动策略进行正式计算（该接口专注于过程测试，后续将被删除），最终得到这批物资在各个维度层面上的优惠信息
   * */
  @ApiOperation(value = "让一批商品规格，基于经销商为指定客户提供的正在执行中的活动策略进行正式计算（该接口专注于过程测试，后续将被删除），最终得到这批物资在各个维度层面上的优惠信息")
  @PostMapping(value="/execute")
  public Result<DefaultPolicyExecuteContext> execute(@RequestBody Set<SalePolicyConProduct> initPolicyProducts ,
                                                     @RequestParam("relevanceCode") String relevanceCode ,
                                                     @RequestParam("customerCode") String customerCode ,
                                                     @RequestParam(value = "kneading" , required = false , defaultValue = "false") boolean kneading) {
    try {
      String tenantCode = TenantUtils.getTenantCode();
      // 根据initPolicyProducts中，是否指定了某些本品只能参与的优惠政策的情况，决定调用哪个接口
      Map<String , String[]> mustCorrectPolicyMapping = Maps.newLinkedHashMap();
      for (SalePolicyConProduct defaultPolicyConProduct : initPolicyProducts) {
        String[] selectedPolicyCodes = defaultPolicyConProduct.getSelectedPolicyCodes();
        if(selectedPolicyCodes != null) {
          mustCorrectPolicyMapping.put(defaultPolicyConProduct.getProductCode(), selectedPolicyCodes);
        }
      }
      DefaultPolicyExecuteContext policyExecuteContext = this.salePolicyExecuteService.execute(relevanceCode, initPolicyProducts, mustCorrectPolicyMapping, tenantCode, customerCode, kneading);
      return Result.ok(policyExecuteContext);
    } catch(RuntimeException e) {
      LOGGER.error(e.getMessage() , e);
      return Result.error(e.getMessage());
    }
  }
  
  /**
   * 该方法按照指定的单据信息、租户信息和客户信息，查找历史上已完成优惠政策执行的结果（以上下文的情况返回）
   * */
  @ApiOperation(value = "该方法按照指定的单据信息、租户信息和客户信息，查找历史上已完成优惠政策执行的结果（以上下文的情况返回）")
  @GetMapping(value="/findByRelevanceCode")
  public Result<DefaultPolicyExecuteContext> findByRelevanceCode(@RequestParam("relevanceCode") String relevanceCode ,
                                                     @RequestParam("customerCode") String customerCode) {
    try {
      String tenantCode = TenantUtils.getTenantCode();
      DefaultPolicyExecuteContext policyExecuteContext = this.salePolicyExecuteService.findByRelevanceCode(relevanceCode, tenantCode, customerCode);
      return Result.ok(policyExecuteContext);
    } catch(RuntimeException e) {
      LOGGER.error(e.getMessage() , e);
      return Result.error(e.getMessage());
    }
  }
}
