package com.biz.crm.dms.business.policy.local.limitstrategy;


import com.alibaba.fastjson.JSON;
import com.biz.crm.dms.business.policy.local.context.DefaultPolicyExecuteContext;
import com.biz.crm.dms.business.policy.local.entity.SalePolicyLimit;
import com.biz.crm.dms.business.policy.local.entity.SalePolicyLimitVar;
import com.biz.crm.dms.business.policy.local.entity.SalePolicyRecord;
import com.biz.crm.dms.business.policy.local.entity.SalePolicyRecordProduct;
import com.biz.crm.dms.business.policy.local.service.SalePolicyLimitService;
import com.biz.crm.dms.business.policy.local.service.SalePolicyLimitVarService;
import com.biz.crm.dms.business.policy.local.utils.AnalyzeExpressionUtils;
import com.biz.crm.dms.business.policy.local.vo.SalePolicyLimitVarVo;
import com.biz.crm.dms.business.policy.local.vo.SalePolicyLimitVo;
import com.biz.crm.dms.business.policy.sdk.context.AbstractCycleExecuteContext;
import com.biz.crm.dms.business.policy.sdk.context.AbstractPolicyExecuteContext;
import com.biz.crm.dms.business.policy.sdk.strategy.SalePolicyLimitStrategy;
import com.biz.crm.dms.business.policy.sdk.vo.AbstractSalePolicyLimitInfo;
import com.biz.crm.dms.business.policy.sdk.vo.SalePolicyVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.google.common.collect.Sets;

import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;

import javax.transaction.Transactional;
import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.Map;
import java.util.concurrent.atomic.AtomicReference;

import java.util.LinkedHashSet;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * 这个抽象父类，帮助DMS标品中的优惠政策限量控制功能管理冗余代码，并没有任何业务意义
 *
 * 当前包其他抽象接口类定义思路如下，结合蓝湖原型设计“DMS后台_V3.020211020->促销政策”菜单中的五种类型：
 *
 * 第一步：分析共性
 *
 * 一、单价特价
 * 本政策每客每单（单价特价）商品数量上限
 * 本政策每客（单价特价）商品数量上限
 * 本政策（单价特价）商品数量总上限
 * （公共）
 *
 * 二、打折
 * 本政策每客每单（打折）商品数量上限
 * 本政策每客（打折）商品数量上限
 * 本政策（打折）商品数量总上限
 * （公共）
 *
 * 三、整单满减
 * （公共）
 *
 * 四、单价满减
 * 本政策每客每单（满减）商品数量上限
 * 本政策每客（满减）商品数量上限
 * 本政策（满减）商品数量总上限
 * （公共）
 *
 * 五、买赠
 * 本政策每客每单（赠品）数量上限
 * 本政策每客（赠品）数量上限
 * 本政策（赠品）数量总上限
 * （公共）
 *
 * 公共
 * 本政策每客每单优惠金额上限
 * 本政策每客优惠金额上限
 * 本政策优惠金额总上限
 *
 * 第二步：抽离共性
 *
 * 一、本政策每客每单优惠金额上限 AbstractCustomerBillAmountLimitStrategy
 * 二、本政策每客优惠金额上限 AbstractCustomerAmountLimitStrategy
 * 三、本政策优惠金额总上限 AbstractTotalAmountLimitStrategy
 * 四、本政策每客每单（类型）商品数量上限 AbstractCustomerBillQuantityLimitStrategy
 * 五、本政策每客（类型）商品数量上限 AbstractCustomerQuantityLimitStrategy
 * 六、本政策（类型）数量总上限 AbstractTotalQuantityLimitStrategy
 *
 * @author pengxi
 * @date 2022/01/13
 */
@Slf4j
public abstract class AbstractBaseSalePolicyLimitStrategy implements SalePolicyLimitStrategy<SalePolicyLimitVo> {
  @Autowired(required = false)
  private SalePolicyLimitService salePolicyLimitService;
  @Autowired(required = false)
  private SalePolicyLimitVarService salePolicyLimitVarService;
  @Autowired(required = false)
  @Qualifier("nebulaToolkitService")
  private NebulaToolkitService nebulaToolkitService;
  @Override
  public Class<SalePolicyLimitVo> getSalePolicyLimitInfoClass() {
    return SalePolicyLimitVo.class;
  }

  @Override
  public SalePolicyLimitVo onRequestSalePolicyLimitInfos(String salePolicyCode) {
    Validate.notBlank(salePolicyCode, "促销政策编号不能为空！");
    Validate.notBlank(this.getLimitStrategyCode(), "限量政策业务编号不能为空！");
    SalePolicyLimit salePolicyLimit = salePolicyLimitService.findBySalePolicyCodeAndTenantCodeAndLimitStrategyCode(salePolicyCode, TenantUtils.getTenantCode(), this.getLimitStrategyCode());
    if(salePolicyLimit == null) {
      return null;
    }
    return this.nebulaToolkitService.copyObjectByWhiteList(salePolicyLimit, SalePolicyLimitVo.class, LinkedHashSet.class, ArrayList.class, "limitVars");
  }

  @Override
  @Transactional
  public void onSaveSalePolicyLimitInfos(boolean update , SalePolicyVo currentSalePolicy , SalePolicyVo oldSalePolicy) {
    /*
     * 维护过程的步骤如下：
     * 1、重新进行每一个门槛信息的验证（包括传值的验证）
     * 2、进行每一个门槛信息的添加
     */
    Validate.notBlank(this.getLimitStrategyCode(), "限量政策业务编号不能为空！");
    Validate.notBlank(this.getExpression() , "维护优惠政策限量信息时，发现表达式没有值(%s)，请检查！" );
    // 1、=======
    if(update) {
      // 查询老的优惠政策中，是否涉及本门槛的信息，如果存在则需要删除
      Set<AbstractSalePolicyLimitInfo> salePolicyLimitInfos = oldSalePolicy.getSalePolicyLimitInfos();
      if(!CollectionUtils.isEmpty(salePolicyLimitInfos)) {
        AbstractSalePolicyLimitInfo oldPolicyLimitInfo = salePolicyLimitInfos.stream().filter(item -> StringUtils.equals(item.getLimitStrategyCode(), this.getLimitStrategyCode())).findFirst().orElse(null);
        // 如果条件成立，则说明要进行删除了
        if(oldPolicyLimitInfo != null) {
          Validate.notBlank(oldPolicyLimitInfo.getId() , "修改优惠政策限量信息时，没有传出原来优惠政策限量信息的ID，请检查！");
          this.salePolicyLimitVarService.deleteBySalePolicyLimitId(oldPolicyLimitInfo.getId());
          this.salePolicyLimitService.deleteByIds(Sets.newHashSet(oldPolicyLimitInfo.getId()));
        }
      }
    }

    // 2、=====
    String salePolicyCode = currentSalePolicy.getSalePolicyCode();
    String tenantCode = currentSalePolicy.getTenantCode();
    Set<AbstractSalePolicyLimitInfo> salePolicyLimitInfos = currentSalePolicy.getSalePolicyLimitInfos();
    if(CollectionUtils.isEmpty(salePolicyLimitInfos)) {
      return;
    }
    AbstractSalePolicyLimitInfo currentPolicyLimitInfo = salePolicyLimitInfos.stream().filter(item -> StringUtils.equals(item.getLimitStrategyCode(), this.getLimitStrategyCode())).findFirst().orElse(null);
    // 注意，这里的添加，只对符合本身处理要求的限量政策起作用
    if(currentPolicyLimitInfo == null) {
      return;
    }
    SalePolicyLimitVo salePolicyLimitInfoVo = (SalePolicyLimitVo)currentPolicyLimitInfo;
    // 同一个优惠政策下，不能有重复的优惠限量政策信息
    String limitStrategyCode = salePolicyLimitInfoVo.getLimitStrategyCode();
    Validate.notBlank(limitStrategyCode , "维护优惠政策限量信息时，没有传出优惠政策限量信息的业务编码，请检查！");
    if(!update) {
      SalePolicyLimit exsitSalePolicyLimit = this.salePolicyLimitService.findBySalePolicyCodeAndTenantCodeAndLimitStrategyCode(salePolicyCode, tenantCode, limitStrategyCode);
      Validate.isTrue(exsitSalePolicyLimit == null , "维护优惠政策限量信息时，发现特定的优惠政策下存在重复的限量政策信息（%s），请检查！" , limitStrategyCode);
    }
    // 其它基本信息的验证
    salePolicyLimitInfoVo.setSalePolicyCode(salePolicyCode);
    salePolicyLimitInfoVo.setTenantCode(tenantCode);
    salePolicyLimitInfoVo.setId(null);
    // 根据表达式，验证传入的变量信息
    Set<SalePolicyLimitVarVo> salePolicyLimitVarVos = salePolicyLimitInfoVo.getLimitVars();
    this.validateVars(limitStrategyCode, salePolicyLimitVarVos);

    // 2、再每一个进行验证和添加
    SalePolicyLimit salePolicyLimitInfo = this.nebulaToolkitService.copyObjectByWhiteList(salePolicyLimitInfoVo, SalePolicyLimit.class, LinkedHashSet.class, ArrayList.class);
    this.salePolicyLimitService.create(salePolicyLimitInfo);
    salePolicyLimitInfoVo.setId(salePolicyLimitInfo.getId());
    if(!CollectionUtils.isEmpty(salePolicyLimitVarVos)) {
      for (SalePolicyLimitVarVo salePolicyLimitVarVo : salePolicyLimitVarVos) {
        SalePolicyLimitVar salePolicyLimitVar = this.nebulaToolkitService.copyObjectByWhiteList(salePolicyLimitVarVo, SalePolicyLimitVar.class, LinkedHashSet.class, ArrayList.class);
        salePolicyLimitVar.setSalePolicyLimit(salePolicyLimitInfo);
        salePolicyLimitVar.setSalePolicyLimitId(salePolicyLimitInfo.getId());
        this.salePolicyLimitVarService.create(salePolicyLimitVar);
        salePolicyLimitVarVo.setId(salePolicyLimitVar.getId());
      }
    }
  }

  @Override
  public void preValidate(AbstractPolicyExecuteContext policyExecuteContext, AbstractCycleExecuteContext cycleExecuteContext, SalePolicyVo currentSalePolicy, AbstractSalePolicyLimitInfo salePolicyLimitInfo) {
    /*
     * 执行逻辑：
     * 1、取得上下文中本次使用量
     * 2、从限量政策变量中取得设定的上限值，并检查本次使用量是否超出
     */
   }

  @Override
  @Transactional
  public void validate(AbstractPolicyExecuteContext policyExecuteContext, AbstractCycleExecuteContext cycleExecuteContext, SalePolicyVo currentSalePolicy, AbstractSalePolicyLimitInfo salePolicyLimitInfo) {
    /*
     * 执行逻辑：
     * 1、首先重新执行一次预置校验
     * 2、预置校验通过，记录本次促销政策的限量政策使用流水
     */
    this.preValidate(policyExecuteContext, cycleExecuteContext, currentSalePolicy, salePolicyLimitInfo);
  }

  /**
   * 金额统一校验方法
   *
   * 执行逻辑：
   * 1、获取设定的限量政策上限值，若没设置表示不需要校验
   * 2、获取流水记录中优惠金额
   * 3、当前上下文中的优惠金额
   * 4、检查本次使用量是否超出
   */
  protected void amountValidate(AbstractPolicyExecuteContext policyExecuteContext, AbstractCycleExecuteContext cycleExecuteContext, SalePolicyVo currentSalePolicy, String varValue, Set<SalePolicyRecord> salePolicyRecords) {
    AtomicReference<BigDecimal> recordAmount = new AtomicReference<>(BigDecimal.ZERO);
    if (CollectionUtils.isNotEmpty(salePolicyRecords)) {
      salePolicyRecords.forEach(r->{
        // 例如：销售金额为1000，优惠了400，最终付款600。这里就取的其中400
        BigDecimal item = Optional.ofNullable(r.getSalePolicyRecordProducts()).orElse(Sets.newHashSet()).stream()
            .filter(s-> StringUtils.equals(currentSalePolicy.getSalePolicyCode(), s.getSalePolicyCode()))
            .map(SalePolicyRecordProduct::getDiffLastSubtotal).reduce(BigDecimal.ZERO,BigDecimal::add);
        recordAmount.set(recordAmount.get().add(item));
      });
    }
    // 3、
    DefaultPolicyExecuteContext executeContext = (DefaultPolicyExecuteContext) policyExecuteContext;
    BigDecimal currentAmount = executeContext.findEnjoyedTotalAmount(currentSalePolicy.getSalePolicyCode());
    BigDecimal totalAmount = currentAmount.add(recordAmount.get());
    log.info("【本品】上下文总金额：{}，历史总金额：{}，最终汇总金额：{}，设定金额：{}", currentAmount, recordAmount.get(), totalAmount, varValue);
    // 4、
    BigDecimal settingAmount = new BigDecimal(varValue);
    Validate.isTrue(settingAmount.compareTo(totalAmount) >= 0, String.format("%s超出设定限量额：%s，当前剩余：%s",this.getLimitStrategyDesc(),varValue, settingAmount.subtract(recordAmount.get())));
  }

  /**
   * 数量统一校验方法
   *
   * 执行逻辑：
   * 1、获取设定的限量政策上限值，若没设置表示不需要校验
   * 2、获取流水记录中优惠商品数量
   * 3、当前上下文中的优惠商品数量
   * 4、检查本次使用量是否超出
   */
  public void quantityValidate(AbstractPolicyExecuteContext policyExecuteContext, AbstractCycleExecuteContext cycleExecuteContext, SalePolicyVo currentSalePolicy, String varValue, Set<SalePolicyRecord> salePolicyRecords) {
    AtomicReference<Integer> recordQuantity = new AtomicReference<>(0);
    if (CollectionUtils.isNotEmpty(salePolicyRecords)) {
      salePolicyRecords.forEach(r->{
        Integer item = Optional.ofNullable(r.getSalePolicyRecordProducts()).orElse(Sets.newHashSet()).stream()
            .filter(s-> StringUtils.equals(currentSalePolicy.getSalePolicyCode(), s.getSalePolicyCode()))
            .mapToInt(SalePolicyRecordProduct::getDiffSurplusTotalNumber).sum();
        recordQuantity.set(recordQuantity.get()+item);
      });
    }
    // 3、
    DefaultPolicyExecuteContext executeContext = (DefaultPolicyExecuteContext)policyExecuteContext;
    Integer currentQuantity = executeContext.findEnjoyedTotalNumber(currentSalePolicy.getSalePolicyCode());
    int totalQuantity = currentQuantity + recordQuantity.get();
    log.info("【本品】上下文总数量：{}，历史总数量：{}，最终汇总数量：{}，设定数量：{}", currentQuantity, recordQuantity.get(), totalQuantity, varValue);
    // 4、
    int settingQuantity = Integer.parseInt(varValue);
    Validate.isTrue(settingQuantity>=totalQuantity, String.format("%s超出设定限量额：%s，当前剩余：%s",this.getLimitStrategyDesc(),varValue, settingQuantity-recordQuantity.get()));
  }


  /**
   * 金额统一校验方法
   *
   * 执行逻辑：
   * 1、获取设定的限量政策上限值，若没设置表示不需要校验
   * 2、获取流水记录中优惠金额
   * 3、当前上下文中的优惠金额
   * 4、检查本次使用量是否超出
   */
  protected void amountValidateGift(AbstractPolicyExecuteContext policyExecuteContext, AbstractCycleExecuteContext cycleExecuteContext, SalePolicyVo currentSalePolicy, String varValue, Set<SalePolicyRecord> salePolicyRecords) {
    AtomicReference<BigDecimal> recordAmount = new AtomicReference<>(BigDecimal.ZERO);
    if (CollectionUtils.isNotEmpty(salePolicyRecords)) {
      salePolicyRecords.forEach(r->{
        // 例如：销售金额为1000，优惠了400，最终付款600。这里就取的其中400
        BigDecimal item = Optional.ofNullable(r.getSalePolicyRecordProducts()).orElse(Sets.newHashSet()).stream()
            .filter(s-> StringUtils.equals(currentSalePolicy.getSalePolicyCode(), s.getSalePolicyCode()))
            //.map(SalePolicyRecordProduct::getDiffLastSubtotal).reduce(BigDecimal.ZERO,BigDecimal::add);
            .map(SalePolicyRecordProduct::getDiffGiftEnjoyedTotalAmount).reduce(BigDecimal.ZERO,BigDecimal::add);
        recordAmount.set(recordAmount.get().add(item));
      });
    }
    // 3、
    DefaultPolicyExecuteContext executeContext = (DefaultPolicyExecuteContext) policyExecuteContext;
    BigDecimal currentAmount = executeContext.findGiftEnjoyedTotalAmount(currentSalePolicy.getSalePolicyCode());
    BigDecimal totalAmount = currentAmount.add(recordAmount.get());
    log.info("【赠品】上下文总金额：{}，历史总金额：{}，最终汇总金额：{}，设定金额：{}", currentAmount, recordAmount.get(), totalAmount, varValue);
    // 4、
    BigDecimal settingAmount = new BigDecimal(varValue);
    Validate.isTrue(settingAmount.compareTo(totalAmount) >= 0, String.format("%s超出设定限量额：%s，当前剩余：%s",this.getLimitStrategyDesc(),varValue, settingAmount.subtract(recordAmount.get())));
  }

  /**
   * 数量统一校验方法
   *
   * 执行逻辑：
   * 1、获取设定的限量政策上限值，若没设置表示不需要校验
   * 2、获取流水记录中优惠商品数量
   * 3、当前上下文中的优惠商品数量
   * 4、检查本次使用量是否超出
   */
  public void quantityValidateGift(AbstractPolicyExecuteContext policyExecuteContext, AbstractCycleExecuteContext cycleExecuteContext, SalePolicyVo currentSalePolicy, String varValue, Set<SalePolicyRecord> salePolicyRecords) {
    AtomicReference<Integer> recordQuantity = new AtomicReference<>(0);
    if (CollectionUtils.isNotEmpty(salePolicyRecords)) {
      salePolicyRecords.forEach(r->{
        Integer item = Optional.ofNullable(r.getSalePolicyRecordProducts()).orElse(Sets.newHashSet()).stream()
            .filter(s-> StringUtils.equals(currentSalePolicy.getSalePolicyCode(), s.getSalePolicyCode()))
            //.mapToInt(SalePolicyRecordProduct::getDiffSurplusTotalNumber).sum();
            .mapToInt(SalePolicyRecordProduct::getDiffGiftEnjoyedTotalNumber).sum();
        recordQuantity.set(recordQuantity.get()+item);
      });
    }
    // 3、
    DefaultPolicyExecuteContext executeContext = (DefaultPolicyExecuteContext)policyExecuteContext;
    Integer currentQuantity = executeContext.findGiftEnjoyedTotalNumber(currentSalePolicy.getSalePolicyCode());
    int totalQuantity = currentQuantity + recordQuantity.get();
    log.info("【赠品】上下文总数量：{}，历史总数量：{}，最终汇总数量：{}，设定数量：{}", currentQuantity, recordQuantity.get(), totalQuantity, varValue);
    // 4、
    int settingQuantity = Integer.parseInt(varValue);
    Validate.isTrue(settingQuantity>=totalQuantity, String.format("%s超出设定限量额：%s，当前剩余：%s",this.getLimitStrategyDesc(),varValue, settingQuantity-recordQuantity.get()));
  }

/////////////////////////////////////////////// 工具方法 /////////////////////////////////////////////

  /**
   * 金额上限变量key
   */
  protected final static String AMOUNT_LIMIT_VAR_KEY = "amountLimitValue";

  /**
   * 数量上限变量key
   */
  protected final static String QUANTITY_LIMIT_VAR_KEY = "quantityLimitValue";

  /**
   * 根据表达式，验证传入的变量信息
   */
  protected void validateVars(String limitStrategyCode, Set<SalePolicyLimitVarVo> salePolicyLimitVarVos) {
    String expression = this.getExpression();
    Map<Integer, String> mustVarNames = AnalyzeExpressionUtils.analyzeLadderExpressionMapping(expression);
    // 如果条件成立，说明这个限量政策不需要输入任何变量，也就不需要做任何验证了
    if(mustVarNames.isEmpty()) {
      Validate.isTrue(CollectionUtils.isEmpty(salePolicyLimitVarVos) , "维护优惠政策限量信息时，发现当前限量政策（%s）无需传递任何变量，请检查！" , limitStrategyCode);
      return;
    }
    Set<String> mustVarNameSet = new HashSet<>();
    mustVarNames.forEach((k,v)-> mustVarNameSet.add(v));
    Set<String> inputVarNames = salePolicyLimitVarVos.stream().map(SalePolicyLimitVarVo::getVariableName).filter(StringUtils::isNotBlank).collect(Collectors.toSet());
    Validate.isTrue(mustVarNames.size() == inputVarNames.size() , "维护优惠政策限量信息时，现在当前限量政策（%s）传递了无效的变量信息（或为空或重复），请检查！" , limitStrategyCode);
    Sets.SetView<String> differenceSet = Sets.difference(inputVarNames, mustVarNameSet);
    Validate.isTrue(differenceSet.isEmpty() , "维护优惠政策限量信息时，发现需要填写的变量信息(%s)，并没有填写，请检查" , StringUtils.join(differenceSet.toArray(new String[] {}) , ","));

    // 一个一个的变量，依次检查
    String pattern = "^[A-Za-z]{1}[A-Za-z0-9]*$";
    for (SalePolicyLimitVarVo salePolicyLimitVarVo : salePolicyLimitVarVos) {
      Integer variableType = salePolicyLimitVarVo.getVariableType();
      Validate.inclusiveBetween(1, 4 , variableType , "维护优惠政策限量信息时，发现变量（%s）不符合变量类型要求（1：boolean、2：小数；3：整数；4：字符串），请检查！");
      String variableName = salePolicyLimitVarVo.getVariableName();
      Validate.matchesPattern(variableName, pattern , "维护优惠政策限量信息时，发现变量（%s）不符合变量命名要求，请检查！" , variableName);
      String variableValue = salePolicyLimitVarVo.getVariableValue();
      Validate.notBlank(variableValue , "维护优惠政策限量信息时，发现变量（%s）没有传入值，请检查！" , variableName);
      salePolicyLimitVarVo.setId(null);
    }
  }

  /**
   * 从限量政策变量中取得设定的上限值（可能是金额或者数量）
   */
  protected String getVarValue(AbstractSalePolicyLimitInfo salePolicyLimitInfo) {
    log.info("salePolicyLimitInfo:{}", JSON.toJSONString(salePolicyLimitInfo));
    if (salePolicyLimitInfo == null) {
      return null;
    }
    SalePolicyLimitVo salePolicyLimitInfoVo = (SalePolicyLimitVo) salePolicyLimitInfo;
    Set<SalePolicyLimitVarVo> varVos = salePolicyLimitInfoVo.getLimitVars();
    if (CollectionUtils.isEmpty(varVos) || !varVos.iterator().hasNext()) {
      return null;
    }
    // 促销政策限量控制的变量只有一个表达式，所以从集合取第一个即可
    SalePolicyLimitVarVo salePolicyLimitVarVo = varVos.iterator().next();
    if (StringUtils.isBlank(salePolicyLimitVarVo.getVariableValue())) {
      return null;
    }
    return salePolicyLimitVarVo.getVariableValue();
  }

}
