package com.biz.crm.dms.business.policy.local.notifier;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.apache.commons.lang3.tuple.Pair;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.util.CollectionUtils;

import com.biz.crm.dms.business.policy.local.context.DefaultPolicyExecuteContext;
import com.biz.crm.dms.business.policy.local.entity.SalePolicyThresholdDetail;
import com.biz.crm.dms.business.policy.local.repository.SalePolicyThresholdDetailRepository;
import com.biz.crm.dms.business.policy.local.repository.SalePolicyThresholdProductRepository;
import com.biz.crm.dms.business.policy.local.repository.SalePolicyThresholdRepository;

import com.biz.crm.dms.business.policy.local.service.SalePolicyThresholdService;
import com.biz.crm.dms.business.policy.local.vo.SalePolicyProductVo;
import com.biz.crm.dms.business.policy.local.vo.SalePolicyThresholdDetailVo;
import com.biz.crm.dms.business.policy.local.vo.SalePolicyThresholdProductVo;
import com.biz.crm.dms.business.policy.local.vo.SalePolicyThresholdVo;
import com.biz.crm.dms.business.policy.sdk.context.AbstractPolicyExecuteContext;
import com.biz.crm.dms.business.policy.sdk.context.PolicyStepResult;
import com.biz.crm.dms.business.policy.sdk.context.ProductPolicyStepResult;
import com.biz.crm.dms.business.policy.sdk.context.SalePolicyConProduct;
import com.biz.crm.dms.business.policy.sdk.service.SalePolicyVoService;
import com.biz.crm.dms.business.policy.sdk.vo.AbstractSalePolicyProductInfo;
import com.biz.crm.dms.business.policy.sdk.vo.AbstractSalePolicyThreshold;
import com.biz.crm.dms.business.policy.sdk.vo.SalePolicyVo;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;

/**
 * 这个抽象类帮助标品中默认的折扣、特价、买赠、满减等优惠政策完成统一的基本数据运维工作，以达到减少代码重复度的要求
 * @author yinwenjie
 */
public abstract class AbstractSalePolicyStickupListener {
  @Autowired(required = false)
  private SalePolicyThresholdRepository salePolicyThresholdRepository;
  @Autowired(required = false)
  private SalePolicyThresholdDetailRepository salePolicyThresholdDetailRepository;
  @Autowired(required = false)
  private SalePolicyThresholdProductRepository salePolicyThresholdProductRepository;
  @Autowired(required = false)
  private SalePolicyThresholdService salePolicyThresholdService;
  @Autowired(required = false)
  private SalePolicyVoService salePolicyVoService;
  
  /**
   * 该方法可以帮助标品中默认的折扣、特价、满减 完成统一方式的优惠政策门槛数据的维护（注意非标品的优惠政策和买赠类型的优惠政策不适用）
   * @param update
   * @param currentSalePolicy
   * @param oldSalePolicy
   * @param salePolicyThreshold
   */
  protected void onSaveSalePolicyProductThreshold(boolean update , SalePolicyVo currentSalePolicy , SalePolicyVo oldSalePolicy, SalePolicyThresholdVo salePolicyThreshold) {
    /*
     * 这里存在两种逻辑：
     * 1、如果当前维护场景属于优惠信息修改，在首先根据oldSalePolicy中的优惠政策业务编号，进行历史门槛信息的删除
     * 2、如果当前维护场景属于优惠信息创建，则进行门槛添加即可
     * */
    
    // 1、=====
    if(update) {
      AbstractSalePolicyThreshold exsitSalePolicyThreshold = oldSalePolicy.getSalePolicyProductThreshold();
      if(exsitSalePolicyThreshold != null) {
        Set<SalePolicyThresholdDetail> salePolicyThresholdDetails = this.salePolicyThresholdDetailRepository.findBySalePolicyThresholdId(exsitSalePolicyThreshold.getId());
        // 一定有(明细和商品明细)，没有就提示警告信息
        if(!CollectionUtils.isEmpty(salePolicyThresholdDetails)) {
          for (SalePolicyThresholdDetail salePolicyThresholdDetail : salePolicyThresholdDetails) {
            this.salePolicyThresholdProductRepository.deleteByThresholdDetailId(salePolicyThresholdDetail.getId());
          }
          this.salePolicyThresholdDetailRepository.deleteBySalePolicyThresholdId(exsitSalePolicyThreshold.getId());
        }
        this.salePolicyThresholdRepository.deleteById(exsitSalePolicyThreshold.getId());
      }
    }
    // 如果已经当前currentSalePolicy优惠政策，已经存在门槛信息，则不再进行保存
    String tenantCode = TenantUtils.getTenantCode();
    if(this.salePolicyThresholdService.findBySalePolicyCode(currentSalePolicy.getSalePolicyCode(), tenantCode) != null) {
      return;
    }
    
    // 2、===== 进行优惠政策门槛基本信息的验证和保存
    // （实际上大部分的验证和添加操作都是在salePolicyThresholdService.create(salePolicyThreshold)）完成的
    // 这里之所以要验证，主要是因为涉及优惠政策中的商品本品信息，所以不得不在这里先做
    List<SalePolicyThresholdDetailVo> salePolicyThresholdDetails = salePolicyThreshold.getThresholdDetails();
    if(!currentSalePolicy.getWholePolicy()) {
      Validate.isTrue(!CollectionUtils.isEmpty(salePolicyThresholdDetails) , "维护优惠政策门槛时，至少有一条优惠政策门槛详情信息，请检查!!");
      // 如果条件成立说明当前只有一条门槛明细
      if(salePolicyThresholdDetails.size() == 1) {
        // 以下确认商品本品的业务编码情况
        Set<AbstractSalePolicyProductInfo> salePolicyProducts = currentSalePolicy.getSalePolicyProductInfos();
        Validate.isTrue(!CollectionUtils.isEmpty(salePolicyProducts) , "维护商品门槛信息时，发现优惠政策中的商品本品信息没有填写，请检查!!");
        long productNullCount = salePolicyProducts.stream().filter(item -> StringUtils.isBlank(((SalePolicyProductVo)item).getProductCode())).count();
        Validate.isTrue(productNullCount == 0 , "维护商品门槛信息时，发现优惠政策中至少一条商品本品信息的业务编码没有填写，请检查!!");
        Set<String> salePolicyProductCodes = salePolicyProducts.stream().map(item -> ((SalePolicyProductVo)item).getProductCode()).distinct().collect(Collectors.toSet());
        Validate.isTrue(salePolicyProductCodes.size() == salePolicyProducts.size() ,  "维护商品门槛信息时，发现优惠政策中至少两条商品本品的业务编码信息重复，请检查!!");
        // 以下确认门槛中的商品信息
        List<SalePolicyThresholdProductVo> thresholdProducts = salePolicyThresholdDetails.iterator().next().getThresholdProducts();
        Validate.isTrue(!CollectionUtils.isEmpty(thresholdProducts) , "维护商品门槛信息时，没有发现任何门槛商品信息，请检查!!");
        productNullCount = thresholdProducts.stream().filter(item -> StringUtils.isBlank(item.getProductCode())).count();
        Validate.isTrue(productNullCount == 0 , "维护商品门槛信息时，发现优惠政策中至少一条商品门槛信息的业务编码没有填写，请检查!!");
        Set<String> thresholdProductCodes = thresholdProducts.stream().map(item -> item.getProductCode()).distinct().collect(Collectors.toSet());
        Validate.isTrue(thresholdProductCodes.size() == thresholdProducts.size() ,  "维护商品门槛信息时，发现优惠政策中至少两条门槛商品的业务编码信息重复，请检查!!");
        // 以下比较差异（不能有差异）
        Validate.isTrue(Sets.difference(salePolicyProductCodes, thresholdProductCodes).isEmpty()
          && Sets.difference(thresholdProductCodes , salePolicyProductCodes).isEmpty() , "维护商品门槛信息时，发现优惠政策的商品信息和商品门槛信息不一致，请检查!!");
        // 如果条件成立，那么说明条门槛明细设定了需要包括优惠政策中所有的商品信息
        if(salePolicyThresholdDetails.iterator().next().getIncludeType() == 1) {
          thresholdProducts.stream().forEach(item -> item.setCertain(true));
        }
      }
    }
    salePolicyThreshold.setSalePolicyCode(currentSalePolicy.getSalePolicyCode());
    this.salePolicyThresholdService.create(salePolicyThreshold);
  }
  
  /**
   * TODO 完善测试，再完善注释
   * @param policyConProducts
   * @param abstractPolicyExecuteContext
   * @param abstractSalePolicyThreshold
   * @return
   */
  protected Pair<Map<String, Set<String>>, String> executeThreshold(Set<SalePolicyConProduct> policyConProducts , AbstractPolicyExecuteContext abstractPolicyExecuteContext , AbstractSalePolicyThreshold abstractSalePolicyThreshold) {
    DefaultPolicyExecuteContext executeContext = (DefaultPolicyExecuteContext)abstractPolicyExecuteContext;
    /*
     * 进行门槛计算，这个门槛计算将会返回一个映射信息，告诉调用者：
     * 哪些商品（编号），因为满足哪一个门槛信息，所以可以允许开始进行优惠计算。这些商品，一定存在于policyConProducts集合中
     * 
     * 而那些没有满足任何门槛商品信息将被调用者记为“不匹配任何门槛”的商品，被排除在后续的优惠政策执行之外。
     * 
     * 如果当前计算返回的情况是，没有任何商品匹配任何的门槛，那么整个优惠政策将被记为“不匹配的优惠”，排除在后续的优惠政策执行之外
     * 
     * 具体计算过程如下：
     * 1、先确定当前门槛涉及的单据中，匹配门槛的商品的最新状态，是否满足优惠政策总门槛的要求（本品累加的总数量、累加总金额的角度出发进行判定）
     * 
     * 2、每一个优惠政策门槛明细，依次进行匹配操作
     *   2.1、当前还未匹配任何门槛的商品，至少和当前门槛明细中的商品信息存在交集
     *   注意，如果该政策要求进行全局优惠（这主要看includeType是等于1还是等于2），则需要保证门槛明细中的商品规格是未匹配任何门槛的商品的子级
     *   2.2、存在交集（子级）的商品规格在基本单位的情况下，必须满足政策中设定参照基本单位的起订数量要求
     *   2.3、如果是全局优惠的方式，还需要判定商品组合限额
     *   2.4、将匹配成功的门槛详情记录下来，并进行下一个门槛详情的匹配，直到所有门槛明细都匹配成功
     * */
    
    // 1、========
    SalePolicyThresholdVo salePolicyThreshold = (SalePolicyThresholdVo) abstractSalePolicyThreshold;
    Integer composeNumber = salePolicyThreshold.getComposeNumber();
    Integer composeUnit = salePolicyThreshold.getComposeUnit();
    List<SalePolicyThresholdDetailVo> salePolicyThresholdDetailVos = salePolicyThreshold.getThresholdDetails();
    
    // 开始统计门槛中，各门槛明细下的商品信息，以便根据这些商品信息寻找单据中参与确认总最低限量的商品范围
    // 如果没有具体的商品信息，那么说明是整单优惠，那么从上下文中取出所有本品信息
    Set<String> thresholdProductCodes = policyConProducts.stream().map(SalePolicyConProduct::getProductCode).collect(Collectors.toSet());
    if(!CollectionUtils.isEmpty(salePolicyThresholdDetailVos)) {
      for (SalePolicyThresholdDetailVo salePolicyThresholdDetail : salePolicyThresholdDetailVos) {
        List<SalePolicyThresholdProductVo> thresholdProducts = salePolicyThresholdDetail.getThresholdProducts();
        thresholdProductCodes.addAll(thresholdProducts.stream().map(SalePolicyThresholdProductVo::getProductCode)
          .filter(currentProductCode -> thresholdProductCodes.contains(currentProductCode))
          .collect(Collectors.toSet()));
      }
    } else {
      salePolicyThresholdDetailVos = Lists.newArrayList();
    }
    
    // 开始进行总门槛的判定
    BigDecimal productsTotalAmount = BigDecimal.ZERO;
    BigDecimal productsTotalNumber = BigDecimal.ZERO;
    for (String productCode : thresholdProductCodes) {
      ProductPolicyStepResult productPolicyStepResult = executeContext.findLastProductStepResultByProductCode(productCode);
      // 如果条件成立，说明单据中并没有选择这样的商品
      if(productPolicyStepResult == null) {
        continue;
      }
      BigDecimal lastSurplusSubtotal = productPolicyStepResult.getLastSurplusTotalAmount();
      Integer lastSurplusTotalNumber = productPolicyStepResult.getLastSurplusTotalNumber();
      productsTotalAmount = productsTotalAmount.add(lastSurplusSubtotal);
      productsTotalNumber = productsTotalNumber.add(new BigDecimal(lastSurplusTotalNumber));
    }
    boolean matched = false;
    // 按照总金额进行判定
    if(composeUnit == 1) {
      matched = productsTotalAmount.intValue() >= composeNumber.intValue();
    } 
    // 按照总数量进行判定
    else {
      matched = productsTotalNumber.intValue() >= composeNumber.intValue();
    } 
    // 如果条件成立，说明不满足总门槛要求，则不再做后续处理
    if(!matched) {
      PolicyStepResult lastPolicyStepResult = abstractPolicyExecuteContext.findLastStepResult();
      if(lastPolicyStepResult == null || StringUtils.isBlank(lastPolicyStepResult.getSalePolicyCode())) {
        return Pair.of(Maps.newHashMap(), "初始量未达到该优惠政策的总体门槛");
      } else {
        String lastPolicyCode = lastPolicyStepResult.getSalePolicyCode();
        SalePolicyVo currentSalePolicyVo = this.salePolicyVoService.findDetailsByCode(lastPolicyCode);
        if(currentSalePolicyVo == null) {
          return Pair.of(Maps.newHashMap(), "(上一优惠政策执行后的剩余量) 未达到该优惠政策的总体门槛");
        } else {
          return Pair.of(Maps.newHashMap(), "(" + currentSalePolicyVo.getSalePolicyName() + " 执行后的剩余量) 未达到该优惠政策的总体门槛");
        }
      }
    }
    
    // 2、======== 
    String salePolicyCode = salePolicyThreshold.getSalePolicyCode();
    // 该优惠政策中，各个门槛明细失败的原因记录在这里，K：门槛明细编号，V：失效原因
    Map<String , String> failedThresholdDetailInfos = Maps.newHashMap();
    // 该优惠政策最终匹配的门槛明细和每个门槛明细对应的商品业务编号，K：门槛明细编号，V：匹配的商品集合
    Map<String , Set<String>> matchedThresholdDetailMappings = Maps.newHashMap();
    // 这个变量存储没有匹配任何门槛的商品信息
    Map<String , ProductPolicyStepResult> remainSourceProductMapping = Maps.newLinkedHashMap();
    for (SalePolicyConProduct defaultPolicyConProduct : policyConProducts) {
      ProductPolicyStepResult productPolicyStepResult = executeContext.findLastProductStepResultByProductCode(defaultPolicyConProduct.getProductCode());
      remainSourceProductMapping.put(defaultPolicyConProduct.getProductCode(), productPolicyStepResult);
    }
    Set<String> remainSourceProductCodes = policyConProducts.stream().map(item -> ((SalePolicyConProduct)item).getProductCode()).collect(Collectors.toSet());
    // 开始进行每一个门槛明细的匹配判定
    // 注意：如果是整单优惠，就认为所有本品都匹配
    if(!CollectionUtils.isEmpty(salePolicyThresholdDetailVos)) {
      TD: for (SalePolicyThresholdDetailVo salePolicyThresholdDetailVo : salePolicyThresholdDetailVos) {        
        Integer includeType = salePolicyThresholdDetailVo.getIncludeType();
        String thresholdDetailCode = salePolicyThresholdDetailVo.getThresholdDetailCode();
        if(includeType == null || (includeType != 1 && includeType != 2)) {
          failedThresholdDetailInfos.put(thresholdDetailCode, String.format("在进行优惠政策计算/预计算时，优惠政策%s:%s发现错误的“门槛所包含商品的方式”", salePolicyCode , salePolicyThresholdDetailVo.getThresholdDetailCode()));
          continue;
        }
        List<SalePolicyThresholdProductVo> targetSalePolicyThresholdProductVos = salePolicyThresholdDetailVo.getThresholdProducts();
        if(CollectionUtils.isEmpty(targetSalePolicyThresholdProductVos)) {
          failedThresholdDetailInfos.put(thresholdDetailCode, String.format("在进行优惠政策计算/预计算时，优惠政策{}:{}发现错误的“门槛明细商品”" , salePolicyCode , salePolicyThresholdDetailVo.getThresholdDetailCode()));
          continue;
        }
        // 根据2022-05-24的讨论结果，如果参与优惠的商品没有任何一个商品属于当前正在处理的优惠政策门槛明细
        // 那么说明这个优惠政策门槛无需进行验证
        Set<String> targetThresholdProductCodes = targetSalePolicyThresholdProductVos.stream().map(SalePolicyThresholdProductVo::getProductCode).collect(Collectors.toSet());
        if(Sets.intersection(thresholdProductCodes, targetThresholdProductCodes).isEmpty()) {
          continue;
        }
        // 2.1、========
        Map<String , SalePolicyThresholdProductVo> targetThresholdProductCodeMapping = targetSalePolicyThresholdProductVos.stream().collect(Collectors.toMap(SalePolicyThresholdProductVo::getProductCode, item -> item));
        // 门槛明细中的商品业务编号是否和remainSourceProductCodes中的商品规格存在子级关系
        boolean isSubset = false;
        // 门槛明细中的商品业务编号是否和remainSourceProductCodes中的商品规格存在交集关系
        boolean isIntersectionSet = false;
        Set<String> intersectionProductCodes = Sets.intersection(targetThresholdProductCodes , remainSourceProductCodes);
        // 如果条件成立，说明是当前还未匹配优惠门槛的商品和门槛明细中的商品是子级关系
        if (!CollectionUtils.isEmpty(intersectionProductCodes) && intersectionProductCodes.size() == targetThresholdProductCodes.size()) {
          isSubset = true;
          isIntersectionSet = true;
        }
        // 如果条件成立，说明是交集关系
        else if (!CollectionUtils.isEmpty(intersectionProductCodes)) {
          isIntersectionSet = true;
        }
        // 除此以外，都说明不匹配门槛明细
        else {
          failedThresholdDetailInfos.put(thresholdDetailCode, "不包括或缺少必须的商品信息");
          continue;
        }
        
        // 2.2、判是判定是否满足门槛，分为includeType是等于1还是等于2的两种情况
        // 如果条件成立，说明这个门槛明细下的所有商品信息，必须是未匹配任何门槛的商品信息的子级
        if(includeType == 1 && isSubset) {
          // 对子级中每一个商品的最低限量进行判定
          for (String intersectionProductCode : intersectionProductCodes) {
            String failedMsg = this.validate(remainSourceProductMapping, targetThresholdProductCodeMapping, intersectionProductCode);
            if(StringUtils.isNotBlank(failedMsg)) {
              failedThresholdDetailInfos.put(thresholdDetailCode, failedMsg);
              continue TD;
            }
          }
        } 
        // 2.3、======
        else if (includeType == 2 && isIntersectionSet) {
          // 首先确定交集中那些设定为“必含”的商品数量，是否达到了要求
          Set<String> targetCertainProductCodes = targetSalePolicyThresholdProductVos.stream().filter(item -> item.getCertain()).map(SalePolicyThresholdProductVo::getProductCode).collect(Collectors.toSet());
          int concurrentCertainCount = Sets.intersection(remainSourceProductCodes, targetCertainProductCodes).size();
          if(concurrentCertainCount < targetCertainProductCodes.size()) {
            failedThresholdDetailInfos.put(thresholdDetailCode, String.format("优惠门槛要求必选数量为%d,但实际必选数量为%d", concurrentCertainCount , targetCertainProductCodes.size()));
            continue TD;
          }
          // 然后确定所有参与门槛判定的商品数量，是否达到了“至少包括X（includeNumber）”属性的要求
          Integer includeNumber = salePolicyThresholdDetailVo.getIncludeNumber();
          if(intersectionProductCodes.size() < includeNumber) {
            failedThresholdDetailInfos.put(thresholdDetailCode, String.format("优惠门槛至少要求商品数量%d,但实际能参与优惠的商品数量为%d", includeNumber , intersectionProductCodes.size()));
            continue TD;
          }        
          // 然后再对交集中每一个商品的最低限量进行判定
          for (String intersectionProductCode : intersectionProductCodes) {
            String failedMsg = this.validate(remainSourceProductMapping, targetThresholdProductCodeMapping, intersectionProductCode);
            if(StringUtils.isNotBlank(failedMsg)) {
              failedThresholdDetailInfos.put(thresholdDetailCode, failedMsg);
              continue TD;
            }
          }
        } else {
          failedThresholdDetailInfos.put(thresholdDetailCode, "不包括或缺少必须的商品信息");
          continue;
        }
        // 2.4、========
        // 如果条件成立，说明当前门槛明细根据其中每一个商品信息进行匹配，符合继续做优惠政策执行的前提
        matchedThresholdDetailMappings.put(thresholdDetailCode, targetThresholdProductCodes);
        remainSourceProductCodes.removeAll(Sets.newHashSet(targetThresholdProductCodes));
      }
    } else {
      matchedThresholdDetailMappings.put(salePolicyThreshold.getId(), thresholdProductCodes);
    }
    
    // 如果条件成立，则说明至少有一个商品不满足单品门槛
    Pair<Map<String, Set<String>>, String> resultPair = null;
    if(!CollectionUtils.isEmpty(failedThresholdDetailInfos)) {
      resultPair = Pair.of(Maps.newHashMap(), CollectionUtils.isEmpty(failedThresholdDetailInfos.values())?"不包括或缺少必须的商品信息":failedThresholdDetailInfos.values().iterator().next());
    } else {
      resultPair = Pair.of(matchedThresholdDetailMappings, "");
    }
    return resultPair;
  }
  
  /**
   * TODO 完善注释
   * @param remainSourceProductMapping
   * @param targetThresholdProductCodeMapping
   * @param targetProductCode
   * @return
   */
  private String validate(Map<String , ProductPolicyStepResult> remainSourceProductMapping , Map<String , SalePolicyThresholdProductVo> targetThresholdProductCodeMapping , String targetProductCode) {
    SalePolicyThresholdProductVo targetThresholdProduct = targetThresholdProductCodeMapping.get(targetProductCode);
    ProductPolicyStepResult sourcePolicyConProduct = remainSourceProductMapping.get(targetProductCode);
    Integer targetThresholdUnit = targetThresholdProduct.getThresholdUnit();
    if(targetThresholdUnit == null || (targetThresholdUnit != 1 && targetThresholdUnit != 2)) {
      return "优惠政策中存在错误的计数方式";
    }
    // 如果条件成立，说明需要进行小计金额性质的比较
    if(targetThresholdProduct.getThresholdUnit() == 1) {
      BigDecimal lastSurplusSubtotal = sourcePolicyConProduct.getLastSurplusTotalAmount();
      if(targetThresholdProduct.getThresholdNumber() > lastSurplusSubtotal.setScale(4, RoundingMode.HALF_UP).floatValue()) {
        return String.format("商品（%s）中剩余未参与优惠的小计金额，没有达到优惠政策的使用门槛", targetProductCode);
      }
    } 
    // 否则说明，需要进行小计数量性质的比较
    else {
      Integer lastSurplusTotalNumber = sourcePolicyConProduct.getLastSurplusTotalNumber();
      if(targetThresholdProduct.getThresholdNumber() > lastSurplusTotalNumber.intValue()) {
        return String.format(String.format("商品（%s）中剩余未参与优惠的数量，没有达到优惠政策的使用门槛", targetProductCode));
      }
    }
    return null;
  }
}
